// idep_ldep.c
#include "idep_linkdep.h"
#include "idep_namearray.h"
#include "idep_nameindexmap.h"
#include "idep_binrel.h"
#include "idep_string.h"
#include "idep_tokeniter.h"
#include "idep_aliastable.h"
#include "idep_aliasutil.h"

#include <math.h>       // log()
#include <ctype.h>      // isascii() isspace()
#include <string.h>     // strchr() strrchr() strlen() strcmp()
#include <memory.h>     // memcpy() memset()
#include <fstream.h>    // ifstream
#include <strstream.h>  // ostrstream
#include <iostream.h>
#include <assert.h>

                // -*-*-*- funkcje statyczne -*-*-*-

static ostream& warn(ostream& ing, int index)
{
    return ing << "Ostrzeenie<" << index << ">: "; }

static ostream& err(ostream& or)
{
    return or << "Bd: ";
}

static const char *stripDotSlash(const char *originalPath)
{
    if (originalPath) {
        while ('.' == originalPath[0] && '/' == originalPath[1]) {
            originalPath += 2;
        }
    }
    return originalPath;
}

static int isLocal(const char *dirFile)
{
    return !strchr(dirFile, '/');
}

static char *removeFileName(char *dirPath)
{
    char *slash = strrchr(dirPath, '/');
    if (slash) {
        slash[1] = '\0';
    }
    else {
        dirPath[0] = '\0';
    }
    return dirPath;
}

static char *removeSuffix(char *dirPath)
{
    char *dot = strrchr(dirPath, '.');
    if (dot && !strchr(dot, '/')) {
        dot[0] = '\0';
    }
    return dirPath;
}

static int digits(int n)
{
    int fieldWidth = 10;
    assert (sizeof (long int) >= 4);
    long int x = 1000 * 1000 * 1000;    // wymaga 4-bajtowej liczby cakowitej.
    while (fieldWidth > 1 && 0 == n / x) {
        --fieldWidth;
        x /= 10;
    }
    return fieldWidth;
}

static double logBase2(double x)
{
    // log (x) = ln(x)/ln(a)
    //    a

    assert(x > 0.0);
    return log(x)/log(2);
}

static double ccdBalencedBinaryTree(int n)
{
    // CCD(n)         = (n + 1) * (log (n + 1) - 1) + 1
    //       zrwnowaone             2
    //       drzewo
    //       binarne

    assert(n >= 0);
    return (n + 1) * (logBase2(n + 1) - 1) + 1;
}

                // -*-*-*- idep_LinkDep_i -*-*-*-

struct idep_LinkDep_i {
    idep_NameIndexMap d_unaliases;          // np., ".", "/usr/include"
    idep_AliasTable d_aliases;              // np., fooa -> fooarray
    idep_NameArray d_dependencyFiles;       // tablica przechowujca zalenoci fazy kompilacji
    idep_NameIndexMap *d_componentNames_p;  // klucze relacji
    idep_BinRel *d_dependencies_p;          // zalenoci fazy kompilacji
    int *d_map_p;                           // odwzorowanie wg. numerw poziomw
    int *d_levels_p;                        // liczba komponentw na poziom
    int *d_levelNumbers_p;                  // numer poziomu dla kadego komponentu
    int *d_cycles_p;                        // komponenty - etykiety dla kadego cyklu
    int *d_weights_p;                       // liczba komponentw w cyklu
    int *d_cycleIndices_p;                  // kolejny indeks cyklu
    int d_numComponents;                    // liczba komponentw w systemie
    int d_numLevels;                        // liczba poziomw w systemie
    int d_numCycles;                        // liczba cykli w systemie     
    int d_numMembers;                       // liczba komponentw wszystkich cykli
    int d_ccd;                              // skumulowane zaenoci komponentu
    idep_LinkDep_i();
    ~idep_LinkDep_i();

    int entry(const char *name, int suffixFlag);
    void loadDependencies(istream& in, int suffixFlag);
    void createCycleArray();
    int calculate(ostream& or, int canonicalFlag, int suffixFlag);
};

idep_LinkDep_i::idep_LinkDep_i()
: d_componentNames_p(0)
, d_dependencies_p(0)
, d_map_p(0)
, d_levels_p(0)
, d_levelNumbers_p(0)
, d_cycles_p(0)
, d_weights_p(0)
, d_cycleIndices_p(0)
, d_numComponents(-1)
, d_numLevels(-1)
, d_numCycles(-1)
, d_numMembers(-1)
, d_ccd(-1)
{
}

idep_LinkDep_i::~idep_LinkDep_i()
{
    delete d_componentNames_p;
    delete d_dependencies_p;
    delete d_map_p;
    delete d_levels_p;
    delete d_levelNumbers_p;
    delete d_cycles_p;
    delete d_weights_p;
    delete d_cycleIndices_p;
}

int idep_LinkDep_i::entry(const char *name, int suffixFlag)
{
    int size = strlen(name) + 1;
    char *buf = new char[size];
    memcpy(buf, name, size);

    if (!isLocal(buf)) {
        removeFileName(buf);
        if (d_unaliases.lookup(buf) >= 0) {             // znaleziono odwoanie aliasu            memcpy(buf, name, size);                    // odtworzenie bufora
        }
    }

    if (suffixFlag) {
        removeSuffix(buf);
    }

    const char *componentName = d_aliases.lookup(buf);
    if (!componentName) {
        const char SLASH_CHAR = '/';
        const char NULL_CHAR = '\0';
        int len = strlen(buf);
        int last = len - 1;
        if (SLASH_CHAR == buf[last]) {             // Jeeli argument wejciowy koczy si na '/'
            buf[last] = NULL_CHAR;                 // prba usunicia znaku '/'             
            componentName = d_aliases.lookup(buf); // i sprawdzenie takiego aliasu.
            if (!componentName) {
                buf[last] = SLASH_CHAR; // odtworzenie
            }
        }
    }

    if (!componentName) {
        componentName = buf;
    }

    int length = d_componentNames_p->length();
    int index = d_componentNames_p->entry(componentName);
    if (d_componentNames_p->length() > length) {
        d_dependencies_p->appendEntry();
    }

    delete [] buf;

    return index;
}

void idep_LinkDep_i::loadDependencies(istream& in, int suffixFlag)
{
    enum { INVALID_INDEX = -1 };
    const char NEWLINE_CHAR = '\n';
    int lineno = 1;
    int fromIndex = INVALID_INDEX;
    int lastTokenWasNewline = 1;

    for (idep_TokenIter it(in); it; ++it) {
        if ('#' == *it()) {                          // obcicie komentarza
            while (it && '\n' != *it()) {
                ++it;
            }
            if (!it) {                               // !it lub '\n'
                continue;
            }
        }

        if (NEWLINE_CHAR == *it()) {
            ++lineno;                                // koniec wiersza
            if (lastTokenWasNewline) {
                fromIndex = INVALID_INDEX;           // koniec biecej sekwencji            
            }
            lastTokenWasNewline = 1;                 // zapisanie statusu nowego wiersza
        }
        else {
            if (INVALID_INDEX == fromIndex) {
                fromIndex = entry(it(), suffixFlag); // pocztek nowej sekwencji        
            }
            else {                                   // znaleziono zaleno
                int toIndex = entry(it(), suffixFlag);
                d_dependencies_p->set(fromIndex, toIndex);
            }
            lastTokenWasNewline = 0;                 // zarejestrowanie stanu nowego wiersza
        }
    }
}

void idep_LinkDep_i::createCycleArray()
{
    assert (!d_cycles_p);               // nie powinno ju istnie
    assert (!d_weights_p);              // nie powinno ju istnie
    assert (!d_cycleIndices_p);         // nie powinno ju istnie
    assert (d_numComponents >= 0);      // powinno by poprawne

    // Utworzenie tablicy cykli w celu wykrywania i identyfikowania potencjalnych cykli.

    d_cycles_p = new int[d_numComponents];  // identyfikuje elementy cyklu
    d_weights_p = new int[d_numComponents]; // identyfikuje wagi cyklu
    d_cycleIndices_p = new int[d_numComponents];  // kolejne indeksy

    // Elementy kadego cyklu s identyfikowane przez indeks ich pierwszego elementu.

    enum { NO_CYCLE = -1 };
    int i;
    for (i = 0; i < d_numComponents; ++i) {
        d_cycles_p[i] = NO_CYCLE;       // Pocztkowo kada pozycja jest niepoprawna. 
        d_weights_p[i] = 0;             // Pocztkowo waga kadego cyklu wynosi 0.
        d_cycleIndices_p[i] = NO_CYCLE; // Pocztkowo kada pozycja jest niepoprawna. 
                                        // Indeksy bd okrelone po podziale komponentw na                                         
                                        // poziomy.
    }

    d_numCycles = 0;                    // liczba niepowtarzalnych cykli projektowych    
    d_numMembers = 0;                   // liczba komponentw zalenych cyklicznie
    // Zaadowanie tablicy cykli. Dla kadego wykrytego cyklu, nieujemny indeks     
    // najmniejszego komponentu uywany jest jako znacznik identyfikujacy ten cykl.
    // Dla kadego komponentu bdziemy skanowa tablic w przd, aby dowiedzie si od     
    // jakich innych komponentw ten komponent jest wzajemnie zaleny. 
    // W przypadku znalezienia jednego lub wikszej liczby takich komponentw,
    // kada z takich lokalizacji bdzie ustawiona na warto indeksu pierwszego
    // (podstawowego) komponentu w cyklu. 
    // W idelanej sytuacji nie bdzie cykli - wwczas kazda pozycja tablicy pozostanie     
    // ustawiona na warto -1. Tablica cykli bdzie wykorzystana na pocztku, w celu zgoszenia     
    // wszystkich zalenoci cyklicznych oraz pniej w celu usprawnienia wykonywania algorytmu     
    // podziau na poziomy w sytuacji, kiedy wystpuj cykle.

    for (i = 0; i < d_numComponents; ++i) {
        if (d_cycles_p[i] >= 0) {
            continue;               // cz poprzednio zgoszonego cyklu
        }
        int cycleFound = 0;
        d_cycles_p[i] = i;          // pierwszy komponent w ramach potencjalnego cyklu
        int j;
	for (j = i + 1; j < d_numComponents; ++j) {
            if (d_cycles_p[j] >= 0) {
                continue;           // cz poprzednio zgoszonego cyklu
            }
            if (d_dependencies_p->get(i, j) && d_dependencies_p->get(j, i)) {
                cycleFound = 1;     // zarejestrowanie faktu wystpienia cyklu
                d_cycles_p[j] = i;  // zarejestrowanie indeksu pierwszego komponentu
            }
        }
        if (cycleFound) {           // jeeli znaleziono cykl obejmujcy komponent `i'
            int weight = 0;
            for (j = i; j < d_numComponents; ++j) { // znajd kady element cyklu
                if (d_cycles_p[j] == i) {
                    ++weight;       // liczba elementw w cyklu
                }
            }
            for (j = i; j < d_numComponents; ++j) { // znajd kady element cyklu
                if (d_cycles_p[j] == i) {
                    d_weights_p[j] = weight;  // zapisz wag kadego elementu
                }
            }
            d_numMembers += weight;   // cakowita liczba elementw w cyklach
            ++d_numCycles;
        }
        else {
            d_cycles_p[i] = NO_CYCLE; // komponent `i' nie jest czci cyklu
        }
    }
}

int idep_LinkDep_i::calculate(ostream& or, int canonicalFlag, int suffixFlag)
{
    enum { IOERRR = -1 };

    // zniszczenie pozostaoci po poprzednich obliczeniach
    delete d_componentNames_p;
    delete d_dependencies_p;
    delete d_map_p;
    delete d_levels_p;
    delete d_levelNumbers_p;
    delete d_cycles_p;
    delete d_weights_p;
    delete d_cycleIndices_p;

    // zaalokowanie nowych struktur danych do oblicze
    d_componentNames_p = new  idep_NameIndexMap;
    d_dependencies_p = new idep_BinRel;
    d_map_p = 0;                // zaalokujemy pniej, kiedy bdzie znany rozmiar                d_levels_p = 0;             // zaalokujemy pniej, kiedy bdzie znany rozmiar             d_levelNumbers_p = 0;       // zaalokujemy pniej, kiedy bdzie znany rozmiar
    d_cycles_p =  0;            // zaalokujemy pniej, kiedy bdzie znany rozmiar     
    d_numLevels = -1;           // na razie uniewaniamy
    d_numComponents = -1;       // na razie uniewaniamy (warto -1 ma znaczenie)
    d_numCycles = -1;           // na razie uniewaniamy
    d_numMembers = -1;          // na razie uniewaniamy
    d_ccd = -1;                 // na razie uniewaniamy

    // Teraz sprbujemy odczyta zalenoci z wymienionego zestawu plikw.
    // W przypadku wystapienia bdu We/Wy - wyjcie z procedury. W innym przypadku     
    // kontynuujemy przetwarzanie.

    int i;
    for (i = 0; i < d_dependencyFiles.length(); ++i) {
        const int INSANITY = 1000;
        if (d_dependencies_p->length() > INSANITY) {
            or << "SPRAWDZENIE POPRAWNOCI: Liczba komponentw obecnie wynosi "
               << d_dependencies_p->length() << " !!!!" << endl;

        }
        enum { IOERROR = -1 };
        const char *file = d_dependencyFiles[i];

        if ('\0' == *file) {
            loadDependencies(cin, suffixFlag);
            cin.clear(0);         // zerujcy znak koca pliku dla standardowego wejcia
        }
        else {
            ifstream in(file);
            if (!in) {
                err(or) << "nie znaleziono pliku zalenoci \"" << file
                        << "\"." << endl;
                return IOERROR;
            }
            loadDependencies(in, suffixFlag);
        }
    }

    // Teraz mona zaalokowa tablice o staym rozmiarze:

    d_numComponents = d_dependencies_p->length();
    assert (d_componentNames_p->length() == d_numComponents);

    // Utworzenie tablicy poziomw dla indeksw nazw komponentw. Tablic poziomw wypenimy     
    // liczb komponentw dla kadego poziomu. 

    d_levels_p = new int[d_numComponents]; // zmienna przechowujca liczb komponentw na poziom
    d_numLevels = 0;                       // pocztkowo nie ma poziomw

    // Utworzenie tablicy numerw poziomw suc do przechowywania numeru poziomu     
    // dla kadego komponentu. Tablic t wypenimy danymi na kocu dziaania tej funkcji.

    d_levelNumbers_p = new int[d_numComponents]; // zmienna przechowujca numery poziomw
    // Utworzenie tablicy odwzorowa dla indeksw nazw komponentw. Tablica ta bdzie     
    // zawiera posortowane indeksy oryginalnych nazw komponentw.

    d_map_p = new int[d_numComponents]; // tablica indeksw komponentw z podziaem na poziomy
    int pIndex = 0;                     // biecy rozmiar odwzorowania
    // Utworzenie tablicy wektorw bajtw.  Kady wektor bajtw identyfikuje komponenty     
    // znajdujce si poniej wskazanego indeksu poziomu. Wiersze s alokowane w miar     
    // definiowania poziomw w systemie, poczwszy do poziomu odpowiadajcego sytuacji,     
    // kiedy d_numLevels = 0.  Po zakoczeniu tych oblicze, naley jawnie usun pierwszych     
    // d_numLevels + 1 wektorw bajtowych.
    //
    // indeks
    // poziomu
    //       +-------+
    //   N   |   ?   |
    //       +-------+
    //  N-1  |   ?   |         +-----------------------------------+
    //       +-------+         | `i' zdefiniowano jako d_numLevels |
    //       :       :         +-----------------------------------+--+
    //       :       :         | `N' zdefiniowano jako d_numComponents|
    //       +-------+         +--------------------------------------+
    //  i+1  |   ?   |
    //       +-------+
    //   i   |   o---|--->[ 1 ][ 1 ][ 0 ][ 1 ][ 1 ][ 1 ][ 0 ] ... [ 0 ]
    //       +-------+
    //       :       :
    //       :       :
    //       +-------+
    //   1   |   o---|--->[ 0 ][ 1 ][ 0 ][ 0 ][ 1 ][ 0 ][ 0 ] ... [ 0 ]
    //       +-------+
    //   0   |   o---|--->[ 0 ][ 0 ][ 0 ][ 0 ][ 0 ][ 0 ][ 0 ] ... [ 0 ]
    //       +-------+
    // indeks komponentu:   0    1    2    3    4    5    6        N-1

    char **lowerThan = new char *[d_numComponents + 1]; // tablica zbiorw bajtw
    lowerThan[0] = new char[d_numComponents];  // zbir komponentw niskiego poziomu        
    memset(lowerThan[0], 0, d_numComponents);  // pocztkowo ten zbir jest pusty
    char *current = new char[d_numComponents]; // komponenty na biecym poziomie
    d_dependencies_p->makeTransitive(); // wykonania algorytmu tranzytywnego zakaczania
    createCycleArray();            // okrelenie i oznaczenie elementw wszystkich cykli
    // Teraz mona uy tranzytywnej relacji zalenoci w celu posortowania komponentw w tym      
    // systemie wedug porzdku poziomw (pod warunkiem, e graf jest acykliczny). W celu     
    // usprawnienia dziaania algorytmu podziau na poziomy, dla podstawowych elementw kadego     
    // cyklu tymczasowo usuwane s zalenoci od innych elementw tego cyklu.
    // Wszystkie pozostae elementy bd pominite przez algorytm podziau na poziomy
    // (zostan oznaczone jako ju nalece do niszego poziomu). 

    for (i = 0; i < d_numComponents; ++i) {
        if (d_cycles_p[i] == i) {
            for (int j = i + 1; j < d_numComponents; ++j) {
                if (d_cycles_p[j] == i) {
                    assert(d_dependencies_p->get(i,j));
                    d_dependencies_p->clr(i,j); // obcicie cyklicznych zalenoci
                    lowerThan[0][j] = 1; // zignorowanie pozostaych elementw w algorytmie
                }
            }
        }
    }

    // Teraz skonstruujemy podzielon na poziomy tablic indeksw komponentw.      
    // Dla kadego poziomu w systemie przeanalizujemy wszystkie komponenty, aby sprawdzi,
    // czy zale one od komponentw wyszego poziomu. Zwrmy uwag, e zgodnie z opisem 
    // znajdujcym si powyej ju odrzucilimy zalenoci cykliczne. Jeeli komponent okae 
    // si czci niszego poziomu - bdzie pominity. W innym przypadku sprawdzimy, czy 
    // komponent zaley od dowolnego innego komponentu, znajdujcego si na innym poziomie 
    // ni niszy.  Jeeli tak jest, komponent ten bdzie pominity. W innym przypadku 
    // indeks komponentu zostanie dodany do biecego poziomu i jednoczenie dodany na 
    // koniec listy uporzdkowanej wedug poziomw. Jeeli indeks ten odpowiada zasadniczemu 
    // elementowi cyklu, uyjemy wagi cyklu, aby okreli poziom poniej poziomu biecego 
    // sucy do sprawdzania zalenoci.  Jeeli zasadniczy element cyklu zaley tylko 
    // od komponentw, ktrych waga poziomu jest nisza od poziomu biecego,  
    // wszystkie pozostae elementy zostan natychmiast dodane do listy w 
    // oryginalnym porzdku. W tym momencie odtworzymy cykliczne zalenoci tego 
    // zasadniczego elementu od innych elementw cyklu. Po wykonaniu iteracji dla wszystkich 
    // komponentw, liczba komponentw na biecym poziomie zostanie dodana do tablicy 
    // poziomw, a biecy zbir poziomw zostanie p oaczony ze zbiorem niszych poziomw.  
    // Proces ten bdzie powtarzany do czasu, kiedy poziomy zostan przypisane do wszystkich 
    // komponentw.

    while (pIndex < d_numComponents) {  // nie wszystkim komponentom przypisano poziom
        int componentCount = 0;         // liczba komponentw na tym poziomie
        memset(current, 0, d_numComponents); // pocztkowo biecy poziom jest pusty
        for (i = 0; i < d_numComponents; ++i) {
            if (d_cycles_p[i] >= 0 && d_cycles_p[i] != i) {
                continue;   // komponent nie jest podstawowym elementem cyklu            
            }

            if (lowerThan[d_numLevels][i]) {
                continue;   // ujty na niszym poziomie;
            }

            int weight = 1; // przypisanie wagi dla komponentw, ktre nie s zalene cyklicznie
            if (d_cycles_p[i] == i) {   // jeeli to jest zasadniczy element cyklu                
                weight = d_weights_p[i];
                if (d_weights_p[i] > d_numLevels + 1) {
                    continue; // waga cyklu o dwa wysza od biecego poziomu
                }
            }

            int searchLevel = d_numLevels + 1 - weight; // dozwolone zalenoci
	    int j;
	    for (j = 0; j < d_numComponents; ++j) {
                if (i == j) {
                    continue;
                }
                if (d_dependencies_p->get(i,j) && !lowerThan[searchLevel][j]) {
                    break; // zaley od komponentu na innym lub na niszym poziomie od                            
                           // poziomu przeszukiwania
                }
            }
            if (j < d_numComponents) {
                continue;   // przeanalizujemy nastpnego kandydata i
            }
            current[i] = 1; // rejestrujemy fakt, e komponent znajduje si na biecym poziomie    
            d_map_p[pIndex++] = i; // dodanie komponentu w porzdku poziomw
            ++componentCount;
            if (d_cycles_p[i] == i) {   // jezeli jest to zasadniczy element cyklu
                for (j = i + 1; j < d_numComponents; ++j) {
                    if (d_cycles_p[j] == i) {  // jeeli inne elementy tego cyklu
                        d_map_p[pIndex++] = j; // doday ten komponent do biecego poziomu
                        ++componentCount;
                        d_dependencies_p->set(i,j); // odtworzenie zalenoci
                    }
                }
            }
        }

        // przejcie do nastpnego poziomu
        for (i = 0; i < d_numComponents; ++i) {
            current[i] |= lowerThan[d_numLevels][i];
        }
        d_levels_p[d_numLevels++] = componentCount; // liczba komponentw na poziom        
        lowerThan[d_numLevels] = current;    // doczenie nowego niszego poziomu        
        current = new char[d_numComponents]; // ponowne zaalokowanie biecej tablicy
    }

    assert (pIndex == d_numComponents);

    // w tym momencie mona zaadowa tablic numerw poziomw
    int start = 0;

    for (i = 0; i < d_numLevels; ++i) {
        int top = start + d_levels_p[i];
        for (int j = start; j < top; ++j) {
            d_levelNumbers_p[d_map_p[j]] = i;
        }
        start = top;
    }

    // Leksykograficzne posortowanie komponentw w ramach kadego poziomu w celu 
    // uatwienia wyszukiwania nazw oraz zastosowanie porzdku kanonicznego w celu 
    // uatwienia odnajdywania rnic w przypadku modyfikacji programw (np. 
    // za pomoc polecenia Unix - diff).

    start = 0;
    int k;
    for (k = 0; k < d_numLevels; ++k) {
        int top = start + d_levels_p[k];
        for (int i = start + 1; i < top; ++i) {
            for (int j = start; j < i; ++j) {
               if (strcmp((*d_componentNames_p)[d_map_p[i]],
                          (*d_componentNames_p)[d_map_p[j]]) < 0) {
                    int tmp = d_map_p[i];
                    d_map_p[i] = d_map_p[j];
                    d_map_p[j] = tmp;
                }
            }
        }
        start = top;
    }

    // Moemy teraz skorzysta z tablicy cykli oraz mapy poziomw w celu utworzenia 
    // tablicy cycleIndex array.  W tej tablicy przypisano wszystkim komponentom 
    // kadego cyklu niepowtarzalny indeks cyklu (w rosncym porzdku poziomu w.r.t.).

    int cycleCount = 0;
    for (i = 0; i < d_numComponents; ++i) {
        const int label = d_cycles_p[d_map_p[i]];
        if (label < 0) {
            continue; // komponent nie jest czci cyklu
        }

        const int index = d_cycleIndices_p[d_map_p[i]];
        if (index >= 0 && index < cycleCount) {
            continue; // komponent jest czci cyklu niszego poziomu
        }

        // Odszukano kolejny cykl - przeanalizowanie pozostaej czci sekwencji i         
        // oznaczenie kadego komponentu za pomoc biecego indeksu cyklu 
        // w tablicy cycleIndices.

        for (int j = i; j < d_numComponents; ++j) {
            if (label == d_cycles_p[d_map_p[j]]) {
                d_cycleIndices_p[d_map_p[j]] = cycleCount;
            }
        }

        ++cycleCount;
    }

    assert(cycleCount == d_numCycles);

    // Teraz ponownie posortujemy komponenty w ramach kadego cyklu, ale tym razem pogrupujemy      
    // komponenty zalene cyklicznie razem tak, aby indeks cyklu uatwia zrozumienie kompozycji     
    // cyklu. Poprzedni porzdek zapewnia wystpowanie indeksw cyklu w porzdku zgodnym z 
    // leksykograficznie najmniejszym elementem cyklu danego poziomu.

    start = 0;
    for (k = 0; k < d_numLevels; ++k) {
        int top = start + d_levels_p[k];
        for (int i = start + 1; i < top; ++i) {
            for (int j = start; j < i; ++j) {
                int ci = d_cycleIndices_p[d_map_p[i]];
                int cj = d_cycleIndices_p[d_map_p[j]];
                if (ci < cj || ci == cj &&
                               strcmp((*d_componentNames_p)[d_map_p[i]],
                                      (*d_componentNames_p)[d_map_p[j]]) < 0) {
                    int tmp = d_map_p[i];
                    d_map_p[i] = d_map_p[j];
                    d_map_p[j] = tmp;
                }
            }
        }
        start = top;
    }

    // Zniszczenie lokalnych struktur danych.
    for (i = 0; i <= d_numLevels; ++i) {
        delete [] lowerThan[i];
    }
    delete [] lowerThan;
    delete [] current;

    // Obliczenie wspczynnika CCD i przechowanie wartoci w polu obiektu.
    idep_BinRel tmp = *d_dependencies_p; // tymczasowa zmienna do oblicze wspczynnika CCD
   for (i = 0; i < d_numComponents; ++i) {
        if (0 == d_levelNumbers_p[i]) {
            for (int j = 0; j < d_numComponents; ++j) {
                tmp.clr(j, i); // zignorowanie zalenoci dla wszystkich komponentw poziomu 0
   }
        }
        else {
            tmp.set(i, i);     // kady komponent ma wpyw na wag jednostki translacji
        }
    }

    int sum = 0;

    for (i = 0; i < d_numComponents; ++i) {
        for (int j = 0; j < d_numComponents; ++j) {
            if (tmp.get(i,j)) {
                ++sum;
            }
        }
    }

    d_ccd = sum;        // Przechowamy t warto -- pniejsze jej obliczenie jest zbyt trudne.

    if (canonicalFlag) {
        // W celu zapewnienia usunicia kanonicznych reprezentacji z redundantnymi zalenociami
        // konieczne jest utworzenie kanonicznej, posortowanej binarnie relacji, na podstawie 
        // tablicy d_map_p. Po usuniciu z tej relacji elementw tranzytywnych, wyniki s
        // z poworotem odwzorowywane na relacj d_dependencies_p.

        idep_BinRel tmp(d_numComponents);

        for (i = 0; i < d_numComponents; ++i) {
            int u = d_map_p[i];
            for (int j = 0; j < d_numComponents; ++j) {
                int v = d_map_p[j];
                int bit = d_dependencies_p->get(u, v);
                tmp.set(i, j, bit);
            }
        }

        tmp.makeNonTransitive();

        for (i = 0; i < d_numComponents; ++i) {
            int u = d_map_p[i];
            for (int j = 0; j < d_numComponents; ++j) {
                int v = d_map_p[j];
                int bit = tmp.get(i, j);
                d_dependencies_p->set(u, v, bit);
            }
        }
    }

    return d_numMembers;
}

                // -*-*-*- idep_LinkDep -*-*-*-

idep_LinkDep::idep_LinkDep() 
: d_this(new idep_LinkDep_i)
{
}

idep_LinkDep::~idep_LinkDep()
{
    delete d_this;
}

void idep_LinkDep::addDependencyFile(const char *fileName)
{
    d_this->d_dependencyFiles.append(fileName);
}

const char *idep_LinkDep::addAlias(const char *alias, const char *component)
{
    return d_this->d_aliases.add(alias, component) < 0 ?
                                        d_this->d_aliases.lookup(alias) : 0;
}

int idep_LinkDep::readAliases(ostream& or, const char *file)
{
    return idep_AliasUtil::readAliases(&d_this->d_aliases, or, file);
}

void idep_LinkDep::addUnaliasDirectory(const char *dirName)
{
    if (*dirName) {
        int len = strlen(dirName);
        if ('/' == dirName[len-1]) {            // ju si koczy na '/'
            const char *n = stripDotSlash(dirName);
            if (*n) {                           // unikamy dodawania pustych katalogw
               d_this->d_unaliases.add(n);
            }
        }
        else {                                  // dodajemy koczcy znak '/'
            char *buf = new char[len+2];
            memcpy (buf, dirName, len);
            buf[len] = '/';
            buf[len+1] = '\0';
            const char *n = stripDotSlash(buf);
            if (*n) {                           // unikamy dodawania pustego katalogu
               d_this->d_unaliases.add(n);
            }
            delete [] buf;
        }
    }
}

int idep_LinkDep::readUnaliasDirectories(const char *file)
{
    enum { BAD = -1, GOOD = 0 };

    ifstream in(file);
    if (!in) {
        return BAD;
    }

    for (idep_TokenIter it(in); it; ++it) {
        if ('\n' != *it()) {
            d_this->d_unaliases.add(it());
        }
    }

    return GOOD;
}

int idep_LinkDep::calculate(ostream& or, int canonicalFlag, int suffixFlag)
{
    return d_this->calculate(or, canonicalFlag, suffixFlag);
}

int idep_LinkDep::numComponents() const
{
    return d_this->d_numComponents;
}

int idep_LinkDep::numPackages() const
{
    return numComponents() > 0 ? d_this->d_levels_p[0] : 0;
}

int idep_LinkDep::numLocalComponents() const
{
    return numComponents() - numPackages();
}

int idep_LinkDep::numLevels() const
{
    return d_this->d_numLevels - 1; // gboko grafu zalenoci komponentu
}

int idep_LinkDep::numCycles() const
{
    return d_this->d_numCycles;
}

int idep_LinkDep::numMembers() const
{
    return d_this->d_numMembers;
}

int idep_LinkDep::ccd() const
{
    return d_this->d_ccd;
}

double idep_LinkDep::acd() const
{
    return numLocalComponents() > 0 ? ccd()/(double)numLocalComponents() : 0.0;
}

double idep_LinkDep::nccd() const
{
    return numLocalComponents() > 0 ? 
        ccd()/ccdBalencedBinaryTree(numLocalComponents()) : 0.0;
}

void idep_LinkDep::printAliases(ostream& o) const
{
    idep_AliasIter it(*this);
    if (it) {
        int fieldWidth = 0;
        {
            for (idep_AliasIter it(*this); it; ++it) {
                int len = strlen(it.fromName());
                if (fieldWidth < len) {
                     fieldWidth = len;
                }
            }
        }
        o << "ALIASY:" << endl;
        for (; it; ++it) {
            o.width(fieldWidth);
            o << it.fromName() << " -> " << it.toName() << endl;
        }
        o << endl;
    }
}

void idep_LinkDep::printUnaliases(ostream& o) const
{
    idep_UnaliasIter it(*this);
    if (it) {
        o << "ODWOANIA ALIASW:" << endl;
        for (; it; ++it) {
             o << it() << endl;
        }
        o << endl;
    }
}

void idep_LinkDep::printCycles(ostream& ing) const
{
    const char *const SPACE = "    ";
    for (idep_CycleIter cit(*this); cit; ++cit) {
        warn(ing, cit.cycle()) << "Nastpujce " << cit.weight()
                  << " komponenty s zalene cyklicznie:" << endl;

        for (idep_MemberIter mit(cit); mit; ++mit) {
            ing << SPACE << mit() << endl;
        }
        ing << endl;
    }
}

void idep_LinkDep::printLevels(ostream& o, int longFlag, int supressFlag) const
{
    if (!supressFlag) {
        o << "POZIOMY:" << endl;
    }

    const char CY_LT = '<';     // definicja znakw otaczajcych indeks cyklu
    const char CY_RT = '>';     // definicja znakw otaczajcych indeks cyklu
    int numLevelDigits = digits(numLevels() - 1);
    int componentNameFieldWidth = 0;

    int numCycleDigits = digits(numCycles() - 1);
    int cycleFieldWidth = numCycleDigits + 2;

    if (longFlag) {
        for (idep_LevelIter lit(*this); lit; ++lit) {
            for (idep_ComponentIter cit(lit); cit; ++cit) {
                int len = strlen(cit());
                if (componentNameFieldWidth < len) {
                     componentNameFieldWidth = len;
                }
            }
        }
    }

    for (idep_LevelIter lit(*this); lit; ++lit) {
        if (!supressFlag) {
            o.width(numLevelDigits);
            o << lit() << ". ";
        }
        int firstFlag = 1;
        for (idep_ComponentIter cit(lit); cit; ++cit) {
            if (firstFlag) {
                firstFlag = 0;
            }
            else {
                if (!supressFlag) {
                    o.width(numLevelDigits + 2);
                    o << "";
                }
            }

            if (longFlag) {
                o.width(componentNameFieldWidth);
            }
            o << cit();

            if (numCycles() > 0 && !supressFlag) {
                char field[100]; // taka warto na pewno wystarczy
                ostrstream f(field, sizeof field);
                f << CY_LT << cit.cycle() << CY_RT << ends;
                o.width(cycleFieldWidth);
                if (cit.cycle()) {
                    long int oldState = o.flags();
                    o.setf(ios::left, ios::adjustfield);
                    o << field;
                    o.flags(oldState);
                }
                else {
                    o << ""; 
                }
            }

            if (longFlag) {
                int firstFlag = 1;
                for (idep_DependencyIter dit(cit); dit; ++dit) {
                    if (firstFlag) {
                        firstFlag = 0;
                    }
                    else {
                        if (!supressFlag) {
                            o.width(numLevelDigits + 2);
                            o << "";
                            if (numCycles() > 0) {
                                o.width(numCycleDigits + 2);
                                o << "";
                            }
                        }
                        o.width(componentNameFieldWidth);
                        o << "";
                    }
                    o << ' ';
                    if (!supressFlag) {
                        o.width(numLevelDigits);
                        o << dit.level() << ". ";
                    }
                    o << dit();
                    if (dit.cycle() && !supressFlag) {
                        o << CY_LT << dit.cycle() << CY_RT;
                    }
                    o << endl;
                }
            }
            o << endl;
        }
        o << endl;
    }
}

inline const char *s(int n) { return (n == 1) ? "" : ""; }
   // oryginalnie: inline const char *s(int n) { return (n == 1) ? "" : "s";
   // Uatwia podanie formy liczby pojedynczej i mnogiej (dotyczy jzyka angielskiego).
void idep_LinkDep::printSummary(ostream& o) const
{
    long iostate = o.setf(ios::left, ios::adjustfield);
    const int FIELD_BUFFER_SIZE = 100;   // Ta warto nie jest cakowicie dowolna
    char field[FIELD_BUFFER_SIZE];       // Taka warto z pewnoci wystarczy!
    o << "PODSUMOWANIE:" << endl;

    const int N = 12;           // szeroko pola numerycznego
    const int G = 1;            // szeroko przerwy
    const int W = 23;           // szeroko caej kolumny

    if (numCycles() > 0) {
        {
            ostrstream f(field, sizeof field);
            f.width(N);
            f << numCycles() << " Cykl(i)" << s(numCycles()) << ends;
            o.width(W);
            o << field;
        }
        o.width(G);
        o << "";
        {
            ostrstream f(field, sizeof field);
            f.width(N);
            f << numMembers() << " Elementw" << ends;
            o.width(W);
            o << field;
        }
        o << endl;
    }
    {
        ostrstream f(field, sizeof field);
        f.width(N);
        f << numLocalComponents() << " Komponent(w)"
                                  << s(numLocalComponents()) << ends;
        o.width(W);
        o << field;
    }
    o.width(G);
    o << "";
    {
        ostrstream f(field, sizeof field);
        f.width(N);
        f << numLevels() << " Poziom(w)" << s(numLevels()) << ends;
        o.width(W);
        o << field;
    }
    o.width(G);
    o << "";
    {
        ostrstream f(field, sizeof field);
        f.width(N);
        f << numPackages() << " Pakiet(w)" << s(numPackages()) << ends;
        o.width(W);
        o << field;
    }
    o << endl;
    {
        ostrstream f(field, sizeof field);
        f.width(N);
        f << ccd() << " CCD" << ends;
        o.width(W);
        o << field;
    }
    o.width(G);
    o << "";
    {
        ostrstream f(field, sizeof field);
        f.width(N);
        f << acd() << " ACD" << ends;
        o.width(W);
        o << field;
    }
    o.width(G);
    o << "";
    {
        ostrstream f(field, sizeof field);
        f.width(N);
        f << nccd() << " NCCD" << ends;
        o.width(W);
        o << field;
    }
    o << endl;

    o.setf(iostate);

    o << endl;
}

                // -*-*-*- wolne operatory -*-*-*-
 
ostream &operator<<(ostream& o, const idep_LinkDep& dep) 
{
    dep.printAliases(o);
    dep.printUnaliases(o);
    dep.printCycles(o);
    dep.printLevels(o, 1);
    dep.printSummary(o);
    return o;
}

                // -*-*-*- idep_AliasIter_i -*-*-*-

struct idep_AliasIter_i {
    idep_AliasTableIter d_iter;

    idep_AliasIter_i(const idep_AliasTable& table);
};

idep_AliasIter_i::idep_AliasIter_i(const idep_AliasTable& table)
: d_iter(table)
{
}

                // -*-*-*- idep_AliasIter -*-*-*-

idep_AliasIter::idep_AliasIter(const idep_LinkDep& dep) 
: d_this(new idep_AliasIter_i(dep.d_this->d_aliases))
{
}

idep_AliasIter::~idep_AliasIter()
{
    delete d_this;
}

void idep_AliasIter::operator++() 
{
    ++d_this->d_iter;
}
 
idep_AliasIter::operator const void *() const
{
    return d_this->d_iter;
}
 
const char *idep_AliasIter::fromName() const
{
    return d_this->d_iter.alias();
}

const char *idep_AliasIter::toName() const
{
    return d_this->d_iter.originalName();
}

                // -*-*-*- idep_UnaliasIter_i -*-*-*-

struct idep_UnaliasIter_i {
    const idep_NameIndexMap& d_array;
    int d_index;

    idep_UnaliasIter_i(const idep_NameIndexMap& array);
};

idep_UnaliasIter_i::idep_UnaliasIter_i(const idep_NameIndexMap& array) 
: d_array(array)
, d_index(0)
{
}

                // -*-*-*- idep_UnaliasIter -*-*-*-

idep_UnaliasIter::idep_UnaliasIter(const idep_LinkDep& dep) 
: d_this(new idep_UnaliasIter_i(dep.d_this->d_unaliases))
{
}

idep_UnaliasIter::~idep_UnaliasIter()
{
    delete d_this;
}


void idep_UnaliasIter::operator++() 
{
    assert (*this);
    ++d_this->d_index;
}
 
idep_UnaliasIter::operator const void *() const
{
    return d_this->d_index < d_this->d_array.length() ? this : 0;
}

const char *idep_UnaliasIter::operator()() const
{
    return d_this->d_array[d_this->d_index];
}

                // -*-*-*- idep_CycleIter_i -*-*-*-

struct idep_CycleIter_i {
    const idep_LinkDep_i& d_dep;
    int d_cycleIndex;
    int d_componentIndex;

    idep_CycleIter_i(const idep_LinkDep_i& dep);
};

idep_CycleIter_i::idep_CycleIter_i(const idep_LinkDep_i& dep)
: d_dep(dep)
, d_componentIndex(-1)
, d_cycleIndex(-1)
{
}

                // -*-*-*- idep_CycleIter -*-*-*-

idep_CycleIter::idep_CycleIter(const idep_LinkDep& dep)
: d_this(new idep_CycleIter_i(*dep.d_this))
{
    ++*this;    // ustawienie na pierwszy cykl
}

idep_CycleIter::~idep_CycleIter()
{
    delete d_this;
}

void idep_CycleIter::operator++() 
{
    assert(*this);
    ++d_this->d_cycleIndex;
    do {
        ++d_this->d_componentIndex;
    } 
    while (*this && d_this->d_dep.d_cycleIndices_p[d_this->d_dep.d_map_p[
                        d_this->d_componentIndex]] != d_this->d_cycleIndex);
}
 
idep_CycleIter::operator const void *() const
{
    return d_this->d_componentIndex < d_this->d_dep.d_numComponents ? this : 0;
}

int idep_CycleIter::weight() const
{
    return d_this->d_dep.d_weights_p[d_this->d_dep.d_map_p[
                                d_this->d_componentIndex]];
}

int idep_CycleIter::cycle() const
{
    return d_this->d_cycleIndex + 1; // cykl 0 zarezerwowany dla komponentw acyklicznych
}

                // -*-*-*- idep_MemberIter_i -*-*-*-

struct idep_MemberIter_i {
    const idep_LinkDep_i& d_dep;
    const int d_cycleIndex;
    int d_index;

    idep_MemberIter_i(const idep_CycleIter_i& iter);
};

idep_MemberIter_i::idep_MemberIter_i(const idep_CycleIter_i& iter)
: d_dep(iter.d_dep)
, d_cycleIndex(iter.d_cycleIndex)
, d_index(iter.d_componentIndex)
{
}

                // -*-*-*- idep_MemberIter -*-*-*-

idep_MemberIter::idep_MemberIter(const idep_CycleIter& iter) 
: d_this(new idep_MemberIter_i(*iter.d_this))
{
}

idep_MemberIter::~idep_MemberIter()
{
    delete d_this;
}

void idep_MemberIter::operator++() 
{
    assert(*this);
    do {
        ++d_this->d_index;
    } 
    while (*this && d_this->d_dep.d_cycleIndices_p[d_this->d_dep.d_map_p[
                                d_this->d_index]] != d_this->d_cycleIndex);
}
 
idep_MemberIter::operator const void *() const
{
    return d_this->d_index < d_this->d_dep.d_numComponents ? this : 0;
}
 
const char *idep_MemberIter::operator()() const
{
    return (*d_this->d_dep.d_componentNames_p)[d_this->d_dep.d_map_p[
                                                        d_this->d_index]];
}

                // -*-*-*- idep_LevelIter_i -*-*-*-

struct idep_LevelIter_i {
    const idep_LinkDep_i& d_dep;
    int d_level;
    int d_start;

    idep_LevelIter_i(const idep_LinkDep_i& dep);
};

idep_LevelIter_i::idep_LevelIter_i(const idep_LinkDep_i& dep)
: d_dep(dep)
, d_level(0)
, d_start(0)
{
}

                // -*-*-*- idep_LevelIter -*-*-*-

idep_LevelIter::idep_LevelIter(const idep_LinkDep& dep) 
: d_this(new idep_LevelIter_i(*dep.d_this))
{
}

idep_LevelIter::~idep_LevelIter()
{
    delete d_this;
}

void idep_LevelIter::operator++() 
{
    assert(*this);
    d_this->d_start += d_this->d_dep.d_levels_p[d_this->d_level++];
}
 
idep_LevelIter::operator const void *() const
{
    return d_this->d_level < d_this->d_dep.d_numLevels ? this : 0;
}
 
int idep_LevelIter::operator()() const
{
    return d_this->d_level;
}


                // -*-*-*- idep_ComponentIter_i -*-*-*-

struct idep_ComponentIter_i {
    const idep_LinkDep_i& d_dep;
    int d_index;
    int d_top;

    idep_ComponentIter_i(const idep_LevelIter_i& iter);
};

idep_ComponentIter_i::idep_ComponentIter_i(const idep_LevelIter_i& iter)
: d_dep(iter.d_dep)
, d_index(iter.d_start)
, d_top(iter.d_start + iter.d_dep.d_levels_p[iter.d_level])
{
}

                // -*-*-*- idep_ComponentIter -*-*-*-

idep_ComponentIter::idep_ComponentIter(const idep_LevelIter& iter) 
: d_this(new idep_ComponentIter_i(*iter.d_this))
{
}

idep_ComponentIter::~idep_ComponentIter()
{
    delete d_this;
}

void idep_ComponentIter::operator++() 
{
    assert(*this);
    ++d_this->d_index;
}
 
idep_ComponentIter::operator const void *() const
{
    return d_this->d_index < d_this->d_top ? this : 0;
}

const char *idep_ComponentIter::operator()() const
{
    return (*d_this->d_dep.d_componentNames_p)[
                                d_this->d_dep.d_map_p[d_this->d_index]];
}

int idep_ComponentIter::cycle() const
{
    return d_this->d_dep.d_cycleIndices_p[
                                d_this->d_dep.d_map_p[d_this->d_index]] + 1;
}

                // -*-*-*- idep_DependencyIter_i -*-*-*-

struct idep_DependencyIter_i {
    const idep_LinkDep_i& d_dep;
    int d_row;
    int d_col;

    idep_DependencyIter_i(const idep_ComponentIter_i& iter);
};

idep_DependencyIter_i::idep_DependencyIter_i(const idep_ComponentIter_i& iter)
: d_dep(iter.d_dep)
, d_row(iter.d_dep.d_map_p[iter.d_index])
, d_col(-1)
{
}

                // -*-*-*- idep_DependencyIter -*-*-*-

idep_DependencyIter::idep_DependencyIter(const idep_ComponentIter& iter)
: d_this(new idep_DependencyIter_i(*iter.d_this))
{
    ++*this;
}

idep_DependencyIter::~idep_DependencyIter()
{
    delete d_this;
}


void idep_DependencyIter::operator++()
{
    assert(*this);
    do {
        ++d_this->d_col;
    }
    while (*this &&                             // przegldamy w kolejnoci poziomw
       !d_this->d_dep.d_dependencies_p->get(d_this->d_row,
       d_this->d_dep.d_map_p[d_this->d_col]));
}
 
idep_DependencyIter::operator const void *() const
{
    return d_this->d_col < d_this->d_dep.d_numComponents ? this : 0;
}

const char *idep_DependencyIter::operator()() const
{
    return (*d_this->d_dep.d_componentNames_p)[ // porzdek wg poziomw
                                d_this->d_dep.d_map_p[d_this->d_col]];
}

int idep_DependencyIter::level() const
{
    return d_this->d_dep.d_levelNumbers_p[      // porzdek wg poziomw
                                d_this->d_dep.d_map_p[d_this->d_col]];
}

int idep_DependencyIter::cycle() const
{
    return d_this->d_dep.d_cycleIndices_p[      // porzdek wg poziomw
                                d_this->d_dep.d_map_p[d_this->d_col]] + 1;
}



