// idep_cdep.c
#include "idep_compiledep.h"
#include "idep_namearray.h"
#include "idep_nameindexmap.h"
#include "idep_binrel.h"
#include "idep_string.h"
#include "idep_filedepiter.h"
#include "idep_tokeniter.h"

#include <ctype.h>      // isascii() isspace()
#include <string.h>     // strlen()
#include <fstream.h>    // ifstream
#include <iostream.h>
#include <assert.h>

                // -*-*-*- funkcje statyczne -*-*-*-

static ostream& err(ostream& or)
{
    return or << "Bd: ";
}

static const char *stripDotSlash(const char *originalPath)
{
    if (originalPath) {
        while ('.' == originalPath[0] && '/' == originalPath[1]) {
            originalPath += 2;
        }
    }
    return originalPath;
}

static int isAbsolutePath(const char *originalPath)
{
    return '/' == *originalPath;
}

static int isAsciiFile(const char *fileName)
{
    enum { NO = 0, YES = 1 };
    ifstream in(fileName);
    if (!in) {
        return NO;
    }

    // sprawdzenie, czy w cigu znajduj si znaki spoza podstawowego kodu ASCII
    char c;
    while (in && !in.get(c).eof()) {
       if (!isascii(c)) {
          return NO;
       }
    }
    return YES;
}

typedef void (idep_CompileDep::*Func)(const char *);
static void loadFromStream(istream& in, idep_CompileDep *dep, Func add)
{
    assert(in);
    for (idep_TokenIter it(in); it; ++it) {
        if ('#' == *it()) {                     // obcicie komentarzy
             while (it && '\n' != *it()) {
                ++it;
            }
            continue;                           // !it lub '\n'
        }
        if ('\n' != *it()) {
            (dep->*add)(it());
        }
    }
}

static int loadFromFile(const char *file, idep_CompileDep *dep, Func add)
{
    enum { BAD = -1, GOOD = 0 };
    if (!isAsciiFile(file)) {
        return BAD;
    }
    ifstream in(file);
    assert(in);
    loadFromStream(in, dep, add);
    return GOOD;
}

static const char *search(idep_String *s, const char *includeDir,
                                                            const char *file)
{
    assert(!isAbsolutePath(file));
    (*s = includeDir) += file;
    const char *dirFile = stripDotSlash(*s);
    return ifstream(dirFile) ? dirFile : 0;
}

static const char *search(idep_String *s, const idep_NameArray& a, 
                                                            const char *file)
{
    if (isAbsolutePath(file)) {
        *s = file;
        const char *absFile = *s;
        return ifstream(absFile) ? absFile : 0;
    }

    const char *dirFile = 0;
    for (int i = 0; i < a.length(); ++i) {
        dirFile = search(s, a[i], file);
        if (dirFile) {
            break;
        }
    }

    return dirFile;
}

                // -*-*-*- statyczne funkcje rekurencyjne -*-*-*-

// Ponisze tymczasowe zmienne wskanikowe zasigu pliku
// s uywane tylko podczas rekurencyjnych wywoa pojedynczej, statycznej funkcji
// getDep(). Ma to na celu uniknicie niepotrzebnych kosztw pamiciowych i czasowych 
// zwizanych z przekazywaniem kilku niezmiennych argumentw  na stos programu.
static idep_BinRel  *s_dependencies_p;  // naley ustawi bezporednio przed pierwszym
                                        // wywoaniem funkcji getDep
static idep_NameIndexMap *s_files_p;    // naley ustawi bezporednio przed pierwszym
                                        // wywoaniem funkcji getDep
static idep_NameArray *s_includes_p;    // naley ustawi bezporednio przed pierwszym
                                        // wywoaniem funkcji getDep
static int s_recurse;                   // naley ustawi bezporednio przed pierwszym
                                        // wywoaniem funkcji getDep
static ostream  *s_err_p;               // naley ustawi bezporednio przed pierwszym
                                        // wywoaniem funkcji getDep
static int getDep (int index)
{
    enum { BAD = -1, GOOD = 0 } status = GOOD;

    idep_String buffer; // bufor cigu znakw, nie naley uywa bezporednio 
    idep_FileDepIter it((*s_files_p)[index]);
    for (; it; ++it) {
        const char *dirFile = search(&buffer, *s_includes_p, it());
        if (!dirFile) {
            err(*s_err_p) << "nie okrelono katalogu include dla pliku \""
                 << it() << "\"." << endl;
            status = BAD;
            continue;
        }

        int length = s_files_p->length();
        int otherIndex = s_files_p->entry(dirFile);

        if (s_files_p->length() > length) {
            // first time looking at this file
            s_dependencies_p->appendEntry();

            if (s_recurse && getDep(otherIndex)) {
                status = BAD;
            }
        }

        s_dependencies_p->set(index, otherIndex, 1);
    }

    if (!it.isValidFile()) {
       err(*s_err_p) << "nie mona otworzy pliku \""
         << (*s_files_p)[index] << "\" do odczytu." << endl;
        status = BAD;
    }

    return status;
}

                // -*-*-*- idep_CompileDep_i -*-*-*-

struct idep_CompileDep_i {
    idep_NameArray d_includeDirectories;      // np., ".", "/usr/include"
    idep_NameArray d_rootFiles;               // pliki do analizy

    idep_NameIndexMap *d_fileNames_p;         // klucze relacji
    idep_BinRel *d_dependencies_p;            // zalenoci fazy kompiolacji
    int d_numRootFiles;                       // liczba korzeni w relacji    
    idep_CompileDep_i();
    ~idep_CompileDep_i();
};

idep_CompileDep_i::idep_CompileDep_i()
: d_fileNames_p(0)
, d_dependencies_p(0)
, d_numRootFiles(-1)
{
}

idep_CompileDep_i::~idep_CompileDep_i()
{
    delete d_fileNames_p;
    delete d_dependencies_p;
}

                // -*-*-*- idep_CompileDep -*-*-*-

idep_CompileDep::idep_CompileDep()
: d_this(new idep_CompileDep_i)
{
}

idep_CompileDep::~idep_CompileDep()
{
    delete d_this;
}

void idep_CompileDep::addIncludeDirectory(const char *dirName)
{
    if (*dirName) {
        int len = strlen(dirName);
        if ('/' == dirName[len-1]) {            // ju si koczy na '/'
            d_this->d_includeDirectories.append(dirName);
        }
        else {                                  // dodanie koczcego '/'
            char *buf = new char[len+2];
            memcpy(buf, dirName, len);
            buf[len] = '/';
            buf[len+1] = '\0';
            d_this->d_includeDirectories.append(buf);
            delete [] buf;
        }
    }
}

int idep_CompileDep::readIncludeDirectories(const char *file)
{
    return loadFromFile(file, this, &idep_CompileDep::addIncludeDirectory);
}

void idep_CompileDep::addRootFile(const char *fileName)
{
    d_this->d_rootFiles.append(fileName);      
}

int idep_CompileDep::readRootFiles(const char *file)
{
    return loadFromFile(file, this, &idep_CompileDep::addRootFile);
}

void idep_CompileDep::inputRootFiles()
{
    if (cin) {
        loadFromStream(cin, this, &idep_CompileDep::addRootFile);
        cin.clear(0);             // wyzerowanie znaku koca pliku dla standardowego                                   // urzdzenie wejciowego
}
}

int idep_CompileDep::calculate(ostream& or, int recursionFlag)
{
    enum { BAD = -1, GOOD = 0 } status = GOOD;

    // zniszczenie wszystkich zmiennych roboczych zwizanych z poprzednimi obliczeniami
    delete d_this->d_fileNames_p;
    delete d_this->d_dependencies_p;

    // przydzielenie nowych struktur danych dla biecych oblicze
    d_this->d_fileNames_p = new idep_NameIndexMap;
    d_this->d_dependencies_p = new idep_BinRel;
    d_this->d_numRootFiles = 0;


    // umieszczenie wszystkich plikw - korzeni na pocztku relacji
    int i;
    for (i = 0; i < d_this->d_rootFiles.length(); ++i) {
        idep_String s;
        const char *file = d_this->d_rootFiles[i];
        const char *dirFile = search(&s, d_this->d_includeDirectories, file);

        if (!dirFile) {
            err(or) << "nie znaleziono pliku \"" << file
                    << "\"." << endl;
            status = BAD;
        }
        else if (d_this->d_fileNames_p->add(dirFile) < 0) {
            err(or) << "plik \"" << file
                    << "\" wymieniony wicej ni raz." << endl;
            status = BAD;
        }
        else {
            ++d_this->d_numRootFiles;
            d_this->d_dependencies_p->appendEntry();
        }
    }

    // Teraz trzeba rekurencyjnie przeanalizowa zalenoci fazy kompilacji
    // dla kadej jednostki translacji. Najpierw skonfigurujemy kilka     
    // wskanikw zasigu pliku. Pozwoli to na zmniejszenie iloci oblicze     
    // zwizanych z zastosowaniem rekurencji.

    s_dependencies_p = d_this->d_dependencies_p;
    s_files_p = d_this->d_fileNames_p;
    s_includes_p = &d_this->d_includeDirectories;
    s_recurse = recursionFlag;
    s_err_p = &or;

    // Kada jednostka translacji tworzy korze drzewa zalenoci.
    // Kady wze odwiedzimy tylko raz, przy okazji zapisujc wyniki.
    // Pocztkowo w relacji istniej tylko jednostki translacji.

    for (i = 0; i < d_this->d_numRootFiles; ++i) {
        const char *name = (*d_this->d_fileNames_p)[i];
        if (getDep(i)) {
            err(or) << "nie mona okreli wszystkich zalenoci dla \""
                    << name << "\"." << endl;
            status = BAD;
        }
    }

    if (recursionFlag) {
        d_this->d_dependencies_p->makeTransitive();
    }

    return status;
}

                // -*-*-*- wolne operatory -*-*-*-

ostream &operator<<(ostream& o, const idep_CompileDep&(dep))
{
    const char *INDENT = "    ";
    for (idep_RootFileIter rit(dep); rit; ++rit) {
        idep_NameArray a;
        o << rit() << endl;
        for (idep_HeaderFileIter hit(rit); hit; ++hit) {
            if (isAbsolutePath(hit())) {
                a.append(hit());
            }
            else {
                o << INDENT << hit() << endl;
            }
        }
        for (int i = 0; i < a.length(); ++i) {
           o << INDENT << a[i] << endl;
        }
        o << endl;
    }
    return o;
}

                // -*-*-*- idep_RootFileIter_i -*-*-*-

struct idep_RootFileIter_i {
    const idep_CompileDep_i& d_dep;
    int d_index;

    idep_RootFileIter_i(const idep_CompileDep_i& dep);
};

idep_RootFileIter_i::idep_RootFileIter_i(const idep_CompileDep_i& dep)
: d_dep(dep)
, d_index(0)
{
}

                // -*-*-*- idep_RootFileIter -*-*-*-

idep_RootFileIter::idep_RootFileIter(const idep_CompileDep& dep) 
: d_this(new idep_RootFileIter_i(*dep.d_this))
{
}

idep_RootFileIter::~idep_RootFileIter()
{
    delete d_this;
}

void idep_RootFileIter::operator++() 
{
    assert(*this);
    ++d_this->d_index;
}
 
idep_RootFileIter::operator const void *() const
{
    return d_this->d_index < d_this->d_dep.d_numRootFiles ? this : 0;
}
 
const char *idep_RootFileIter::operator()() const
{
    return (*d_this->d_dep.d_fileNames_p)[d_this->d_index];
}

                // -*-*-*- idep_HeaderFileIter_i -*-*-*-

struct idep_HeaderFileIter_i {
    const idep_RootFileIter_i& d_iter;
    int d_index;

    idep_HeaderFileIter_i(const idep_RootFileIter_i& iter);
};

idep_HeaderFileIter_i::idep_HeaderFileIter_i(const idep_RootFileIter_i& iter) 
: d_iter(iter)
, d_index(-1)
{
}

                // -*-*-*- idep_HeaderFileIter -*-*-*-

idep_HeaderFileIter::idep_HeaderFileIter(const idep_RootFileIter& iter) 
: d_this(new idep_HeaderFileIter_i(*iter.d_this))
{
    ++*this;
}

idep_HeaderFileIter::~idep_HeaderFileIter()
{
    delete d_this;
}


void idep_HeaderFileIter::operator++() 
{
    assert(*this);
    idep_BinRel *rel = d_this->d_iter.d_dep.d_dependencies_p;
    
    do {
        ++d_this->d_index;
    }
    while (   d_this->d_index < rel->length() 
           && !rel->get(d_this->d_iter.d_index, d_this->d_index)
    );
}
 
idep_HeaderFileIter::operator const void *() const
{
    idep_BinRel *rel = d_this->d_iter.d_dep.d_dependencies_p;
    return d_this->d_index < rel->length() ? this : 0;
}
 
const char *idep_HeaderFileIter::operator()() const
{
    return (*d_this->d_iter.d_dep.d_fileNames_p)[d_this->d_index];
}

