// idep_alutil.c
#include "idep_aliasutil.h"
#include "idep_aliastable.h"
#include "idep_tokeniter.h"
#include "idep_string.h"

#include <fstream.h>   // ifstream
#include <iostream.h>
#include <assert.h>

                // -*-*-*- funkcje statyczne -*-*-*-

static ostream& warning(ostream& or, const char *file, int lineno)
{
    return or << "Ostrzeenie w " << file << '(' << lineno << "): ";
}

static ostream& err(ostream& or, const char *file, int lineno)
{
    return or << "Bd w " << file << '(' << lineno << "): ";
}

static int tryToAlias(idep_AliasTable *table, ostream& or,
                      const char *inputName, int lineno,
                      const char *componentName, const char *alias)
{
    if (table->add(alias, componentName) < 0) {
        const char *previousName = table->lookup(alias);
        err(or, inputName, lineno) << "dwie nazwy dla aliasu \""
            << alias << "\":" << endl << "    \"" << previousName
            << "\" i \"" << componentName << "\"" << endl;
        return 1;
    }
    return 0;
}

                // -*-*-*- idep_AliasUtil -*-*-*-

int idep_AliasUtil::readAliases(idep_AliasTable *table, ostream& or,
						istream& in, const char *inputName)
{

    // Poniej znajduje si opis jzyka aliasw
language:
    enum State {
        START,          // pocztek nowego aliasu
        CONTINUE_0,     // kontynuacja   - brak poprzednich identyfikatorw
        IDENT_0,        // identyfikator - brak poprzednich identyfikatorw        
        CONTINUE_1,     // kontynuacja   - jeden poprzedni identyfikator
        CONTINUE_NL,    // kontynuacja   - zakoczone znakiem nowego wiersza
        IDENT_NL,       // identyfikator - zakoczono znakiem nowego wiersza
        NEWLINE_BL,     // nowy wiersz   - zakoczono pustym wierszem
        CONTINUE_BL,    // kontynuacja   - zakoczono pustym wierszem
        IDENT_BL,       // identyfikator - zakoczono pustym wierszem
        NUM_STATES      // musi wystpowa jako ostatni zapis
    } state = START;

    enum Input {
        CONTINUE,       // znak kontynuacji
        NEWLINE,        // koniec wiersza
        IDENT,          // indentyfikator (nie poprzednie dwa)
        NUM_INPUTS      // musi wystpowa jako ostatni zapis
    } input;

    enum Action {
        NOP,            // nic nie rb
        BEG_CUR,        // ustaw nazw komponentu na biecy cig         
        BEG_PRE,        // ustaw nazw komponentu na poprzedni cig         
        BEG_PRE_CUR,    // ustaw nazw komponentu na poprzedni i wyprbuj biecy cig         
        TRY_CUR,        // sprbuj przypisa biecy cig jako alias nazwy komponentu        
        TRY_PRE,        // sprbuj przypisa poprzedni cig jako alias nazwy komponentu
        TRY_PRE_CUR,    // sprbuj przypisa oba cigi jako aliasy nazwy komponentu        
        END,            // nie ustawiona nazwa komponentu
        NUM_ACTIONS     // musi wystpowa jako ostatni zapis
    };

    static State nextStateTable[NUM_STATES][NUM_INPUTS] = {
        //CONTINUE      NEWLINE         IDENT
        { CONTINUE_0,   START,          IDENT_0         }, // START
        { CONTINUE_1,   START,          IDENT_NL        }, // CONTINUE_0
        { CONTINUE_1,   NEWLINE_BL,     IDENT_NL        }, // IDENT_0
        { CONTINUE_NL,  IDENT_0,        IDENT_NL        }, // CONTINUE_1
        { CONTINUE_NL,  IDENT_NL,       IDENT_NL        }, // CONTINUE_NL
        { CONTINUE_NL,  START,          IDENT_NL        }, // IDENT_NL
        { CONTINUE_BL,  START,          IDENT_BL        }, // NEWLINE_BL
        { CONTINUE_BL,  IDENT_BL,       IDENT_BL        }, // CONTINUE_BL
        { CONTINUE_BL,  NEWLINE_BL,     IDENT_BL        }  // IDENT_BL
    };

    static Action actionTable[NUM_STATES][NUM_INPUTS] = {
        //CONTINUE      NEWLINE         IDENT
        { NOP,          NOP,            BEG_CUR         }, // START
        { BEG_PRE,      NOP,            BEG_PRE_CUR     }, // CONTINUE_0
        { NOP,          NOP,            TRY_CUR         }, // IDENT_0
        { TRY_PRE,      NOP,            TRY_PRE_CUR     }, // CONTINUE_1
        { TRY_PRE,      NOP,            TRY_PRE_CUR     }, // CONTINUE_NL
        { NOP,          END,            TRY_CUR         }, // IDENT_NL
        { NOP,          END,            TRY_CUR         }, // NEWLINE_BL
        { TRY_PRE,      NOP,            TRY_PRE_CUR     }, // CONTINUE_BL
        { NOP,          NOP,            TRY_CUR         }  // IDENT_BL
    };

    const char *const EMPTY_NAME = "";
    const char NULL_CHAR = *EMPTY_NAME;
    const char COMMENT_CHAR = '#';
    const char CONTINUE_CHAR = '\\';
    const char NEWLINE_CHAR = '\n';

    int numBadAliases = 0;
    int lineno = 1;

    idep_String componentName = EMPTY_NAME;
    idep_String lastToken = EMPTY_NAME;
    Input lastInput = IDENT;

    idep_TokenIter it(in);
    for (; it; ++it) {

        if (*it() == COMMENT_CHAR) {
            while (*it() != NEWLINE_CHAR) {
                ++it;                   // ignoruj wszystkie cigi a do znaku koca wiersza
	    }
        }
	
        // Okrelenie rodzaju informacji wejciowych:
	
        switch (*it()) {
          case CONTINUE_CHAR: {
            input = NULL_CHAR == it()[1] ? CONTINUE : IDENT;
          } break;
          case NEWLINE_CHAR: {
            input = NEWLINE;
          } break;
          default: {
            input = IDENT;
          } break;
        };

        // Wykonanie wybranej akcji:

        switch (actionTable[state][input]) {
          case NOP: {
          } break;
          case BEG_CUR: {
            componentName = it();
          } break;
          case BEG_PRE: {
            componentName = lastToken;
            warning(or, inputName, lineno) << '"' << lastToken
                << "\" << uyto jako nazw komponentu." << endl;
          } break;
          case BEG_PRE_CUR: {
            componentName = lastToken;
            numBadAliases += tryToAlias(table, or, inputName, lineno,
                                                        componentName, it());
            warning(or, inputName, lineno) << '"' << lastToken
                << "\" << uyto jako nazw komponentu." << endl;
          } break;
          case TRY_CUR: {
            numBadAliases += tryToAlias(table, or, inputName, lineno,
                                                        componentName, it());
          } break;
          case TRY_PRE: {
            numBadAliases += tryToAlias(table, or, inputName, lineno,
                                                   componentName, lastToken);
            warning(or, inputName, lineno) << '"' << lastToken
                << "\" << uyto jako nazw aliasu." << endl;
          } break;
          case TRY_PRE_CUR: {
            numBadAliases += tryToAlias(table, or, inputName, lineno,
                                                   componentName, lastToken);
            numBadAliases += tryToAlias(table, or, inputName, lineno,
                                                        componentName, it());
            warning(or, inputName, lineno) << '"' << lastToken
                << "\" << uyto jako nazw aliasu." << endl;
          } break;
          case END: {
            componentName = EMPTY_NAME;
          } break;
          default:
          case NUM_ACTIONS: {
            assert(0);
          }
        };

        // Przejcie do nastpnego stanu:

        if (NEWLINE == input) {
            ++lineno;                           // koniec wiersza
        }

        lastToken = it();
        lastInput = input;

        state = nextStateTable[state][input];
    }

    return numBadAliases;                       // 0 w przypadku sukcesu
}


int idep_AliasUtil::readAliases(idep_AliasTable *table, ostream& or,
							    const char *fileName)
{
    enum { IOERROR = -1 };
    ifstream in(fileName);
    if (!in) {
        return IOERROR;
    }
    return readAliases(table, or, in, fileName);
}
