// idep_adep.c
#include "idep_aliasdep.h"
#include "idep_namearray.h"
#include "idep_nameindexmap.h"
#include "idep_filedepiter.h"
#include "idep_tokeniter.h"
#include "idep_aliastable.h"
#include "idep_aliasutil.h"

#include <ctype.h>      // isascii() isspace()
#include <string.h>     // strlen() strrchr()
#include <fstream.h>    // ifstream
#include <memory.h>     // memcpy()
#include <iostream.h>
#include <assert.h>

                // -*-*-*- funkcje statyczne -*-*-*-

ostream& warn(ostream& ing)
{
    return ing << "Ostrzeenie: ";
}

static ostream& err(ostream& or)
{
    return or << "Bd: ";
}

static const char *stripDotSlash(const char *originalPath)
{
    if (originalPath) {
        while ('.' == originalPath[0] && '/' == originalPath[1]) {
            originalPath += 2;
        }
    }
    return originalPath;
}

static const char *stripDir(const char *s)
{
    if (s) {
        const char *slash = strrchr(s, '/');
        return slash ? slash + 1 : s;
    }
    return s;   // brak wejcia
}

static int isAsciiFile(const char *fileName)
{
    enum { NO = 0, YES = 1 };
    ifstream in(fileName);
    if (!in) {
        return NO;
    }

    // sprawdzenie, czy nazwa nie zawiera znakw spoza podstawowego kodu ASCI
    char c;
    while (in && !in.get(c).eof()) {
       if (!isascii(c)) {
          return NO;
       }
    }
    return YES;
}

typedef void (idep_AliasDep::*Func)(const char *);
static void loadFromStream(istream& in, idep_AliasDep *dep, Func add)
{
    assert(in);
    for (idep_TokenIter it(in); it; ++it) {
        if ('#' == *it()) {                     // obcicie komentarza
             while (it && '\n' != *it()) {
                ++it;
            }
            continue;                           // !it lub '\n'
        }
        if ('\n' != *it()) {
            (dep->*add)(it());
        }
    }
}

static int loadFromFile(const char *file, idep_AliasDep *dep, Func add)
{
    enum { BAD = -1, GOOD = 0 };
    if (!isAsciiFile(file)) {
        return BAD;
    }
    ifstream in(file);
    assert(in);
    loadFromStream(in, dep, add);
    return GOOD;
}

static char *newStrCpy(const char *name)
{
    int size = strlen(name) + 1;
    char *buf = new char[size];
    memcpy(buf, name, size);
    return buf;
}

static char *removeSuffix(char *dirPath)
{
    char *dot = strrchr(dirPath, '.');
    if (dot && !strchr(dot, '/')) {     // jeeli znaleziono '.' w segmencie cieki  
        dot[0] = '\0';                  // wyeliminowanie przyrostka ("a/.b" -> "a/") 
    }
    return dirPath;
}

                // -*-*-*- idep_AliasDepIntArray -*-*-*-

class idep_AliasDepIntArray {   // klasa pomocnicza do zarzdzania pamici tablicy
    int *d_array_p;
    int d_length;
    idep_AliasDepIntArray(const idep_AliasDepIntArray&);
    idep_AliasDepIntArray& operator=(const idep_AliasDepIntArray&);
  public:
    idep_AliasDepIntArray(int length) : // nie zeruje pamici!
			d_array_p(new int[length]), d_length(length) {}
    ~idep_AliasDepIntArray() { delete [] d_array_p; }
    int& operator[](int i) { return d_array_p[i]; }
    int length() const { return d_length; }
};

static void zero(idep_AliasDepIntArray *a) // nieelementarna operacja na tablicy
{
    for (int i = 0; i < a->length(); ++i) {
        (*a)[i] = 0;
    }
}

                // -*-*-*- idep_AliasDepString -*-*-*-

class idep_AliasDepString {     // pomocnicza klasa do zarzdzania modyfikowalnym char *
    char *d_string_p;
    idep_AliasDepString(const idep_AliasDepString&);
    idep_AliasDepString& operator=(const idep_AliasDepString&);
  public:
    idep_AliasDepString(const char *s) : d_string_p(newStrCpy(s)){}
    ~idep_AliasDepString() { delete [] d_string_p; }
    operator char *() { return d_string_p; }
};

                // -*-*-*- idep_AliasDep_i -*-*-*-

struct idep_AliasDep_i {
    idep_NameIndexMap d_ignoreNames;          // np., idep_compiledep.t.c
    idep_AliasTable d_aliases;                // mp., my_inta -> my_intarray
    idep_NameIndexMap d_fileNames;            // pliki do analizy
};

                // -*-*-*- idep_AliasDep -*-*-*-

idep_AliasDep::idep_AliasDep()
: d_this(new idep_AliasDep_i)
{
}

idep_AliasDep::~idep_AliasDep()
{
    delete d_this;
}

void idep_AliasDep::addIgnoreName(const char *fileName)
{
    d_this->d_ignoreNames.add(fileName);
}

int idep_AliasDep::readIgnoreNames(const char *file)
{
    return loadFromFile(file, this, &idep_AliasDep::addIgnoreName); 
}

const char *idep_AliasDep::addAlias(const char *alias, const char *component)
{
    return d_this->d_aliases.add(alias, component) < 0 ?
                                        d_this->d_aliases.lookup(alias) : 0;
}

int idep_AliasDep::readAliases(ostream& or, const char *file)
{
    return idep_AliasUtil::readAliases(&d_this->d_aliases, or, file);
}

void idep_AliasDep::addFileName(const char *fileName)
{
    d_this->d_fileNames.add(fileName);
}

int idep_AliasDep::readFileNames(const char *file)
{
    return loadFromFile(file, this, &idep_AliasDep::addFileName);
}

void idep_AliasDep::inputFileNames()
{
    if (cin) {
        loadFromStream(cin, this, &idep_AliasDep::addFileName);
        cin.clear(0);             // zerowanie - znak koca pliku standardowego wejcia
    }
}

int idep_AliasDep::unpaired(ostream& out, ostream& ing, int suffixFlag) const
{
    int maxLength = d_this->d_fileNames.length();
    idep_AliasDepIntArray hits(maxLength);  // liczba plikw w komponencie
    idep_AliasDepIntArray cmap(maxLength);  // odwzorowanie komponentu na (ostatni) plik
    zero(&hits);
    idep_NameIndexMap components;
    int numComponents = 0;

    idep_NameIndexMap printNames;   // Zmienna uywana do sortowania nazw w celu
                                    // uatwienia wykonywania operacji wytnij i wklej w
				    // edytorze.

    int i;
    for (i = 0; i < maxLength; ++i) {
        idep_AliasDepString s(d_this->d_fileNames[i]);

        if (d_this->d_ignoreNames.lookup(s) >= 0) {
            continue; // ignore this file
        }
        removeSuffix(s);

        const char *componentName = d_this->d_aliases.lookup(s);
        if (!componentName) {
            componentName = s;
        }

        int componentIndex = components.entry(componentName);
        if (components.length() > numComponents) {      // nowy komponent
            ++numComponents;

        }

        assert(components.length() == numComponents);

        ++hits[componentIndex];
        cmap[componentIndex] = i; // nadpisanie biecym indeksem
    }

    for (i = 0; i < numComponents; ++i) {
        assert(hits[i] > 0);
        if (1 == hits[i]) {
            printNames.add(suffixFlag ? d_this->d_fileNames[cmap[i]]
                                      : components[i]);
        }
        if (hits[i] > 2) {
            warn(ing) << "komponent \"" << components[i]
                      << "\" skada si z " << hits[i] << " plikw." << endl;
        }
    }

    // Z powodu ogranicze dugoci nazwy pliku .o w bibliotece
    // czsto zdarza si, e to wanie plik nagwkowy ma dusz -
    // waciw nazw komponentu. Jeeli zmienna suffixFlag wynosi 0, posortujemy nazwy    
    // w porzdku niemal leksykograficznym, poza tym, e krtsza z dwch nazw o identycznym pocztku
    // bdzie *po*, a nie przed dusz. Taki porzdek uatwi nam wykonywanie operacji
    // wytnij i wklej podczas rcznego tworzenia pliku aliasw za pomoc
    // edytora tekstu. 

    int numUnpaired = printNames.length();
    idep_AliasDepIntArray smap(numUnpaired);
    for (i = 0; i < numUnpaired; ++i) {
        smap[i] = i;                            // rozpoczynamy od odwzorowania                                                 
                                                // tosamociowego
}
    for (i = 1; i < numUnpaired; ++i) {
        for (int j = 0; j < numUnpaired; ++j) {
            int swap;
            if (suffixFlag) {
                swap = strcmp(printNames[smap[i]], printNames[smap[j]]) < 0;
            }
            else {
                int li = strlen(printNames[smap[i]]);
                int lj = strlen(printNames[smap[j]]);
                int len = li < lj ? li : lj;    // min dugo
                int cmp = strncmp(printNames[smap[i]],
                                  printNames[smap[j]], len);
                swap = cmp < 0 || 0 == cmp && li > lj;  // najpierw duszy
            }
            if (swap) {                                 // zamieniamy miejscami jeli trzeba
                int tmp = smap[i];
                smap[i] = smap[j];
                smap[j] = tmp;
            }
        }
    }

    // wywietlenie nazw w porzdku (prawie) leksykograficznym (jeeli suffixFlag ustawiono na 0)

    for (i = 0; i < numUnpaired; ++i) {
        out << printNames[smap[i]] << endl;
    }

    return printNames.length();
}

static char *th(int n)
{
    return (char *) (1 == n ? "szy" : 2 == n ? "gi" : 3 == n ? "ci" : "ty");
}

int idep_AliasDep::verify(ostream& or) const
{
    enum { IOERROR = -1, GOOD = 0 } status = GOOD;
    int errorCount = 0; // liczba plikw, ktrych nie dao si odczyta
    int length = d_this->d_fileNames.length();
    for (int i = 0; i < length; ++i) {
        const char *path = d_this->d_fileNames[i];
        idep_AliasDepString c(path);

        if (d_this->d_ignoreNames.lookup(c) >= 0) {
            continue; // ignorujemy plik
        }

        // ucinamy przyrostek i ciek z nazwy pliku komponentu i sprawdzamy aliasy
        removeSuffix(c);
        const char *actualComponent = stripDir(c);
        const char *compAlias = d_this->d_aliases.lookup(actualComponent);
        const char *component = compAlias ? compAlias : actualComponent;

        int directiveIndex = 0;
	idep_FileDepIter it(path);
        for (; it; ++it) {

            ++directiveIndex;

            // ucicie przyrostka i cieki z nazwy pliku nagwkowego i sprawdzenie aliasw
            idep_AliasDepString h(it());
            removeSuffix(h);
            const char *actualHeader = stripDir(h);
            const char *headerAlias = d_this->d_aliases.lookup(actualHeader);
            const char *header = headerAlias ? headerAlias : actualHeader;

            if (0 == strcmp(component, header)) { // jeli nazwy s te same             
                break;
            }
        }

        if (!it.isValidFile()) { // jeeli nazwa pliku jest nieprawidowa
        err(or) << "nie mona otworzy pliku \""
                    << path << "\" do odczytu." << endl;
            status = IOERROR;
        }
        else if (!it) {                         // nie znaleziono nagwka
            err(or) << "nie znaleziono odpowiadajcej dyrektywy include dla \"" << path << "\"."
                    << endl;
            ++errorCount;
        }
        else if (1 != directiveIndex) {         // znaleziono nagwek, ale nie jako pierwszy
             err(or) << '"' << path
                    << "\" zawiera odpowiadajcy mu plik include jako "
                    << directiveIndex << th(directiveIndex)
                    << " dyrektyw." << endl;
            ++errorCount;
        }

        // w innym przypadku wszystko jest ok
    }

    return status == GOOD ? errorCount : status;
}


int idep_AliasDep::extract(ostream& out, ostream& or) const
{
    enum { IOERROR = -1, GOOD = 0 } status = GOOD;
    enum { INVALID_INDEX = -1 };
    int errorCount = 0; // liczba plikw, ktrych nie dao si odczyta
    idep_NameIndexMap uniqueHeaders;       // zmienna do ledzenia kilku plikw .c
    int length = d_this->d_fileNames.length();
    idep_AliasDepIntArray hits(length);    // liczba plikw nagwkowych    zero(&hits);
    idep_AliasDepIntArray hmap(length);    // indeks pliku nagwkowego w tabeli
    idep_AliasDepIntArray verified(length);// sprawdzenie, czy odgadnito prawidowo    
    zero(&verified);

    int i;
    for (i = 0; i < length; ++i) {
        hmap[i] = INVALID_INDEX;   // ustawimy na VALID (prawidowy), po znalezieniu odpowiedniego nagwka
        const char *path = d_this->d_fileNames[i];
        idep_AliasDepString c(path);

        if (d_this->d_ignoreNames.lookup(c) >= 0) {
            continue; // zignorowanie tego pliku
        }

        // obcicie rozszerzenia i cieki z nazwy pliku komponentu i sprawdzenie aliasw
        removeSuffix(c);
        const char *actualComponent = stripDir(c);
        const char *compAlias = d_this->d_aliases.lookup(actualComponent);
        const char *component = compAlias ? compAlias : actualComponent;

        idep_FileDepIter it(path);      // prba otwarcia pliku wg pierwszej dyrektywy.

        if (!it.isValidFile()) {        // plik nie daje si czyta
            err(or) << "nie mona otworzy pliku \""
                    << path << "\" do odczytu." << endl;
            status = IOERROR;
            continue;                   // nic wicej nie mona zrobi
        }

        if (!it) {                      // brak dyrektyw include
            err(or) << '"' << path
                    << "\" nie zawiera dyrektyw include." << endl;
            ++errorCount;
            continue;                   // tu nie ma nic wicej do roboty
        }

        // obcicie rozszerzenia i cieki z nazwy pliku nagwka i sprawdzenie aliasw
        idep_AliasDepString h(it());
        removeSuffix(h);
        const char *actualHeader = stripDir(h);
        const char *headerAlias = d_this->d_aliases.lookup(actualHeader);
        const char *header = headerAlias ? headerAlias : actualHeader;

        if (0 == strcmp(component, header)) {

            // W tym miejscu mamy nazw komponentu i nagwka, ktre s ze sob zgodne
            // dlatego, bo ich nazwy zasadnicze byy identyczne
            // lub dlatego, bo znalelimy odpowiedni alias. Zapisujemy ten
            // fakt w tablicy verified (sprawdzono).

            verified[i] = 1;
        }
        else {

            // Podejrzewamy, e to moe by odpowiedni alias do pary, ale nie mamy
            // pewnoci. Sprawdzimy, czy ju istnieje alias obejmujcy ten plik .c.
            // Jeeli tak jest, aliast ten przesoni nasz nazw i nie bdziemy             
            // generowa aliasu na nowo.

            if (compAlias) {
                continue;               // nic wicej nie mamy do zrobienia
            }
        }

        // Pki co nie ma powodu, aby sdzi, e ta para jest nieprawidowa.
        // Zapisujemy fakt skojarzenia nagwka z biecym plikiem .c.

        int hIndex = uniqueHeaders.entry(header); // uzyskujemy indeks nagwka        
        ++hits[hIndex];                           // zapisujemy czstotliwo
        hmap[i] = hIndex;                         // ustawiamy indeks .c -> nagwek
    }

    const int FW = 25;
    const char *const ARROW = ">- prawdopodobnie waciwy -> ";

    // Wywietlamy ostrzeenie na standardowym urzdzeniu bdu dla kadego nagwka,
    // dla ktrego istnieje wicej ni jeden plik .c, w ktrym w pierwszej dyrektywie
    // include wymieniono nazw tego nagwka.

    for (i = 0; i < uniqueHeaders.length(); ++i) {
        if (hits[i] > 1) {
            warn(or) << hits[i] << " plikw wymienia \"" << uniqueHeaders[i]
                 << "\" w swojej pierwszej dyrektywie include:" << endl;
            for (int j = 0; j < length; ++j) {
                if (i ==  hmap[j]) {
                    or.width(FW);
                    or << (verified[j] ? ARROW : "");
                    or << '"' << stripDir(d_this->d_fileNames[j])
                       << '"' << endl;
                }
            }
            or << endl;
        }
    }

    // Wywietlenie nieredundantnych aliasw  plik nagwkowy / implementacyjny     // do okrelonego strumienia wyjciowego.

    for (i = 0; i < length; ++i) {
        if (hmap[i] >= 0 && !verified[i]) {
           // obcicie rozszerzenia i cieki z nazwy pliku komponentu
           idep_AliasDepString c(d_this->d_fileNames[i]);
           removeSuffix(c);
           out << uniqueHeaders[hmap[i]] << ' ' << c << endl;
        }
    }

    return status == GOOD ? errorCount : status;        
}

