/* The following code example is taken from the book
 * "The C++ Standard Library - A Tutorial and Reference"
 * by Nicolai M. Josuttis, Addison-Wesley, 1999
 *
 * (C) Copyright Nicolai M. Josuttis 1999.
 * Permission to copy, use, modify, sell and distribute this software
 * is granted provided this copyright notice appears in all copies.
 * This software is provided "as is" without express or implied
 * warranty, and with no claim as to its suitability for any purpose.
 */
#include <limits>
#include <iostream>

namespace MyLib {
   template <class T>
   class MyAlloc {
     public:
       // definicje typow
       typedef T        value_type;
       typedef T*       pointer;
       typedef const T* const_pointer;
       typedef T&       reference;
       typedef const T& const_reference;
       typedef std::size_t    size_type;
       typedef std::ptrdiff_t difference_type;

       // zwiaz alokator z typem U
       template <class U>
       struct rebind {
           typedef MyAlloc<U> other;
       };

       // zwroc adres wartosci
       pointer address (reference value) const {
           return &value;
       }
       const_pointer address (const_reference value) const {
           return &value;
       }

       /* konstruktory oraz destruktor
        * - niekonieczne jest wykonywanie zadnych czynnosci poniewaz alokator nie ma stanu
        */
       MyAlloc() throw() {
       }
       MyAlloc(const MyAlloc&) throw() {
       }
       template <class U>
         MyAlloc (const MyAlloc<U>&) throw() {
       }
       ~MyAlloc() throw() {
       }

       // zwroc maksymalna liczbe elementow, dla ktorych moze zostac zaalokowana pamiec
       size_type max_size () const throw() {
           return std::numeric_limits<std::size_t>::max() / sizeof(T);
       }

       // przydziel pamiec dla num elementow typu T lecz jej nie inicjalizuj
       pointer allocate (size_type num, const void* = 0) {
           // wyswietl komunikat oraz przydziel pamiec przy uzyciu globalnego operatora new
           std::cerr << "zostala przydzielona pamiec dla " << num << " elementow"
                     << " o rozmiarze " << sizeof(T) << std::endl;
           pointer ret = (pointer)(::operator new(num*sizeof(T)));
           std::cerr << " pamiec przydzielona pod adresem: " << (void*)ret << std::endl;
           return ret;
       }

       // zainicjalizuj elementy w przydzielonym obszarze pamieci p za pomoca wartosci value
       void construct (pointer p, const T& value) {
           // zainicjalizuj pamiec przy uzyciu operatora new
           new((void*)p)T(value);
       }

       // usun elementy z zainicjalizowanego obszaru pamieci p
       void destroy (pointer p) {
           // usun obiekty poprzez wywolanie ich destruktorow
           p->~T();
       }

       // zwolnij obszar pamieci p zajmowany przez usuniete elementy
       void deallocate (pointer p, size_type num) {
           // wyswietl komunikat oraz zwolnij pamiec przy uzyciu globalnego operatora delete
           std::cerr << "zostala zwolniona pamiec dla " << num << " elementow"
                     << " o rozmiarze " << sizeof(T)
                     << " pod adresem: " << (void*)p << std::endl;
           ::operator delete((void*)p);
       }
   };

   // zwroc wartosc wskazujaca, czy wszystkie konkretyzacje danego alokatora moga byc stosowane wymiennie
   template <class T1, class T2>
   bool operator== (const MyAlloc<T1>&,
                    const MyAlloc<T2>&) throw() {
       return true;
   }
   template <class T1, class T2>
   bool operator!= (const MyAlloc<T1>&,
                    const MyAlloc<T2>&) throw() {
       return false;
   }
}
