#include <iostream>

enum { kIsSmaller, kIsLarger, kIsSame};

// Każda klasa na tej liście musi obsługiwać dwie
// metody: show() wyświetlającą wartość oraz compare(), która
// zwraca relatywne położenie.
class Data
{
public:
    Data(int newVal):value(newVal) {}
    ~Data()
    {
        std::cout << "Usunięcie obiektu Data o wartości: ";
        std::cout << value << "\n";
    }
    int compare(const Data&);
    void show() { std::cout << value << "\n"; }
private:
    int value;
};

// Metoda decydująca, gdzie na liście zostanie
// umieszczony określony obiekt.
int Data::compare(const Data& otherObject)
{
    if (value < otherObject.value)
        return kIsSmaller;
    if (value > otherObject.value)
        return kIsLarger;
    else
        return kIsSame;
}

// Kolejna klasa, której obiekty mają być umieszczane na liście.
class Robot
{
public:
    Robot(int newAge): age(newAge) {}
    ~Robot()
    {
        std::cout << "Usunięcie robota w wieku ";
        std::cout << age << "lat.\n";
    }
    int compare(const Robot&);
    void show()
    {
        std::cout << "Ten robot ma ";
        std::cout << age << " lat.\n";
    }
private:
    int age;
};

// Ta klasa porównuje inne wartości niż Data.
int Robot::compare(const Robot& otherRobot)
{
    if (age < otherRobot.age)
        return kIsSmaller;
    if (age > otherRobot.age)
        return kIsLarger;
    else
        return kIsSame;
}

// To jest abstrakcyjny typ danych przedstawiający obiekt węzła na liście.
// Każda klasa potomna musi nadpisywać metody insert() i show().
template <class T>
class Node
{
public:
    Node(){}
    virtual ~Node() {}
    virtual Node* insert(T* object) = 0;
    virtual void show() = 0;
private:
};

template <class T>
class InternalNode: public Node<T>
{
public:
    InternalNode(T* object, Node<T>* next);
    ~InternalNode(){ delete next; delete object; }
    virtual Node<T> * insert(T * object);
    virtual void show()
    {
        object->show();
        next->show();
    } // Delegacja!
private:
    T* object;      // To jest obiekt.
    Node<T>* next; // Ten wskaźnik prowadzi do następnego węzła na liście.
};

// Prosty konstruktor.
template <class T>
InternalNode<T>::InternalNode(T* newObject, Node<T>* newNext):
object(newObject),next(newNext)
{
}

// Metoda przeznaczona do przechowywania nowego obiektu na liście.
// Obiekt zostaje przekazany do węzła, który ustala miejsce jego
// umieszczenia, a następnie faktycznie wstawia obiekt na liście.
template <class T>
Node<T>* InternalNode<T>::insert(T* newObject)
{
    // Czy nowy obiekt jest mniejszy, czy większy od bieżącego?
    int result = object->compare(*newObject);

    switch(result)
    {
    // Jeżeli są takie same, wtedy zgodnie z konwencją nowy będzie pierwszym.
    case kIsSame:   // Przejście dalej.
    case kIsLarger: // Nowe dane zostaną umieszczone przed bieżącymi.
        {
            InternalNode<T>* objectNode =
                new InternalNode<T>(newObject, this);
            return objectNode;
        }
    // iNowy jest większy, więc zostaje przekazany do następnego
    // węzła, gdzie powinien być obsłużony.
    case kIsSmaller:
        next = next->insert(newObject);
        return this;
    }
    return this;  // Uspokojenie kompilatora.
}

// Ostatni węzeł na liście.
template <class T>
class TailNode : public Node<T>
{
public:
    TailNode() {}
    virtual ~TailNode() {}
    virtual Node<T>* insert(T * object);
    virtual void show() { }
private:
};

// Jeżeli dane są przeznaczone dla tego węzła, muszą być wstawione przed nim,
// ponieważ nic nie może się znajdować po węźle końcowym.
template <class T>
Node<T>* TailNode<T>::insert(T * object)
{
    InternalNode<T>* objectNode =
        new InternalNode<T>(object, this);
    return objectNode;
}

// Węzeł początkowy nie przechowuje żadnych danych, ale
// wskazuje początek listy.
template <class T>
class HeadNode : public Node<T>
{
public:
    HeadNode();
    virtual ~HeadNode() { delete next; }
    virtual Node<T>* insert(T * object);
    virtual void show() { next->show(); }
private:
    Node<T> * next;
};

// Pierwszy węzeł na liście, powoduje utworzenie węzła końcowego.
template <class T>
HeadNode<T>::HeadNode()
{
    next = new TailNode<T>;
}

// Ponieważ nic nie może się znajdować przed węzłem początkowym,
// dane przekazujemy po prostu do następnego węzła.
template <class T>
Node<T> * HeadNode<T>::insert(T* object)
{
    next = next->insert(object);
    return this;
}

// Ta klasa otrzymuje uznanie, choć nie wykonuje żadnej pracy.
template <class T>
class LinkedList
{
public:
    LinkedList();
    ~LinkedList() { delete head; }
    void insert(T* object);
    void showAll() { head->show(); }
private:
    HeadNode<T> * head;
};

// W chwili powstania następuje utworzenie węzła początkowego,
// który z kolei tworzy węzeł końcowy.
template <class T>
LinkedList<T>::LinkedList()
{
    head = new HeadNode<T>;
}

// Delegacja do węzła początkowego.
template <class T>
void LinkedList<T>::insert(T* pObject)
{
    head->insert(pObject);
}

// Zebranie wszystkich klas w celu ich przetestowania.
int main()
{
    Robot* pRobot;
    Data* pData;
    int val;
    LinkedList<Robot> listOfRobots;
    LinkedList<Data> listOfData;

    // Wartości podane przez użytkownika są przechowywane na liście.
    while (true)
    {
        std::cout << "Podaj wartość (0 oznacza zakończenie programu): ";
        std::cin >> val;
        if (!val)
            break;
        pRobot = new Robot(val);
        pData = new Data(val);
        listOfRobots.insert(pRobot);
        listOfData.insert(pData);
    }

    // Wyświetlenie listy.
    std::cout << "\n";
    listOfRobots.showAll();
    std::cout << "\n";
    listOfData.showAll();
    std::cout << "\n ************ \n\n";
    return 0;
}
