#ifndef VEC_H
#define VEC_H

#include <algorithm>
#include <cstddef>
#include <memory>

#ifdef _MSC_VER
#include "../minmax.h"
#else
using std::max;
#endif

template <class T> class Vec {
public:
	typedef T* iterator;
	typedef const T* const_iterator;
	typedef size_t size_type;
	typedef T value_type;
	typedef T& reference;
	typedef const T& const_reference;

	Vec() { create(); }
	explicit Vec(size_type n, const T& t = T()) { create(n, t); }

	Vec(const Vec& v) { create(v.begin(), v.end()); }
	Vec& operator=(const Vec&);	// jak w 11.3.2
	~Vec() { uncreate(); }

	T& operator[](size_type i) { return data[i]; }
	const T& operator[](size_type i) const { return data[i]; }

	void push_back(const T& t) {
		if (avail == limit)
			grow();
		unchecked_append(t);
	}

	size_type size() const { return avail - data; }  // zmiana

	iterator begin() { return data; }
	const_iterator begin() const { return data; }

	iterator end() { return avail; }                 // zmiana
	const_iterator end() const { return avail; }     // zmiana
	void clear() { uncreate(); }
	bool empty() const { return data == avail; }

private:
	iterator data;	// pierwszy element kontenera `Vec'
	iterator avail;	// (nastpny za) otatni element kontenera `Vec'
	iterator limit;	// (nastpny za) element zainicjowany

	// przydzia pamici
	std::allocator<T> alloc;	// obiekt poredniczcy w zarzdzaniu pamici

	// przydzia i inicjalizacja pamici kontenera
	void create();
	void create(size_type, const T&);
	void create(const_iterator, const_iterator);

	// usuwanie elementw kontenera i zwalnianie pamici
	void uncreate();

	// funkcje pomocnicze metody push_back
	void grow();
	void unchecked_append(const T&);
};

template <class T> void Vec<T>::create()
{
	data = avail = limit = 0;
}

template <class T> void Vec<T>::create(size_type n, const T& val)
{
#ifdef _MSC_VER
	data = alloc.allocate(n, 0);
#else
	data = alloc.allocate(n);
#endif
	limit = avail = data + n;
	std::uninitialized_fill(data, limit, val);
}

template <class T>
void Vec<T>::create(const_iterator i, const_iterator j)
{
#ifdef _MSC_VER
	data = alloc.allocate(j - i, 0);
#else
	data = alloc.allocate(j - i);
#endif
	limit = avail = std::uninitialized_copy(i, j, data);
}

template <class T> void Vec<T>::uncreate()
{
	if (data) {
		// usu utworzone wczeniej elementy (w kolejnoci odwrotnej do kolejnoci tworzenia)
		iterator it = avail;
		while (it != data)
			alloc.destroy(--it);

		// zwolnij pami przydzielon wczeniej kontenerowi
		alloc.deallocate(data, limit - data);
	}
	// wyzeruj wskaniki aby zasygnalizowa brak elementw w kontenerze Vec
	data = limit = avail = 0;

}

template <class T> void Vec<T>::grow()
{
	// przy powikszaniu kontenera przydziel obszar dwukrotnie wikszy od dotychczasowego
	size_type new_size = max(2 * (limit - data), ptrdiff_t(1));

	// przydziel nowy obszar i skopiuj do niego elementy kontenera
#ifdef _MSC_VER
	iterator new_data = alloc.allocate(new_size, 0);
#else
	iterator new_data = alloc.allocate(new_size);
#endif
	iterator new_avail = std::uninitialized_copy(data, avail, new_data);

	// zwolnij dotychczasowy obszar pamici
	uncreate();

	// ustal wartoci wskanikw zgodnie z nowymi realiami
	data = new_data;
	avail = new_avail;
	limit = data + new_size;
}

// za, e avail wskazuje do obszaru przydzielonego ale niezainicjalizowanego 
template <class T> void Vec<T>::unchecked_append(const T& val)
{
	alloc.construct(avail++, val);
}

template <class T>
Vec<T>& Vec<T>::operator=(const Vec& rhs)
{
	// sprawd, czy nie zachodzi przypisanie obiektu do samego siebie
	if (&rhs != this) {

		// zwolnij tablic lewego operandu
		uncreate();

		// kopiuj elementy z operandu prawego do operandu lewego
		create(rhs.begin(), rhs.end());
	}
	return *this;
}

#endif
