/* The following code example is taken from the book
 * "The C++ Standard Library - A Tutorial and Reference"
 * by Nicolai M. Josuttis, Addison-Wesley, 1999
 *
 * (C) Copyright Nicolai M. Josuttis 1999.
 * Permission to copy, use, modify, sell and distribute this software
 * is granted provided this copyright notice appears in all copies.
 * This software is provided "as is" without express or implied
 * warranty, and with no claim as to its suitability for any purpose.
 */
#ifndef COUNTED_PTR_HPP
#define COUNTED_PTR_HPP

/* klasa realizujaca semantyke referencji ze zliczaniem
 * - usuwa obiekt, do ktrego si odnosi, gdy zniszczony zostanie ostatni
 *   odnoszacy sie do tego obiektu wskaznik CountedPtr
 */
template <class T>
class CountedPtr {
  private:
    T* ptr;        // wskaznik do wartosci
    long* count;   // dzielona liczba wlascicieli

  public:
    // inicjalizacja wskaznika istniejacym wskaznikiem
    // - wymaga, aby wskaznik p byl wartoscia zwrocona przez operator new
    explicit CountedPtr (T* p=0)
     : ptr(p), count(new long(1)) {
    }

    // kopiowanie wskaznika (jeszcze jeden wlasciciel)
    CountedPtr (const CountedPtr<T>& p) throw()
     : ptr(p.ptr), count(p.count) {
        ++*count;
    }

    // destruktor (usuwa wartosc, jesli to byl ostatni wlasciciel)
    ~CountedPtr () throw() {
        dispose();
    }

    // operacja przypisania (zwolnia stara i wspoldzieli nowa wartosc)
    CountedPtr<T>& operator= (const CountedPtr<T>& p) throw() {
        if (this != &p) {
            dispose();
            ptr = p.ptr;
            count = p.count;
            ++*count;
        }
        return *this;
    }

    // dostep do wartosci, do ktorej odnosi sie wskaznik
    T& operator*() const throw() {
        return *ptr;
    }
    T* operator->() const throw() {
        return ptr;
    }

  private:
    void dispose() {
        if (--*count == 0) {
             delete count;
             delete ptr;
        }
    }
    
};

#endif /*COUNTED_PTR_HPP*/
