/* The following code example is taken from the book
 * "The C++ Standard Library - A Tutorial and Reference"
 * by Nicolai M. Josuttis, Addison-Wesley, 1999
 *
 * (C) Copyright Nicolai M. Josuttis 1999.
 * Permission to copy, use, modify, sell and distribute this software
 * is granted provided this copyright notice appears in all copies.
 * This software is provided "as is" without express or implied
 * warranty, and with no claim as to its suitability for any purpose.
 */
#include <iostream>
#include <iomanip>
#include <map>
#include <string>
#include <algorithm>
using namespace std;

/* obiekt funkcyjny do porownywania lancuchow
 * - umozliwia ustawianie kryterium porownania podczas wykonywania
 * - umozliwia porownywanie bez rozrozniania wielkosci liter
 */
class RuntimeStringCmp {
  public:
    // stale dla kryterium porownania
    enum cmp_mode {normal, nocase};
  private:
    // aktualny tryb porownywania
    const cmp_mode mode;

    // funkcja pomocnicza do porownywania bez rozrozniania wielkosci liter
    static bool nocase_compare (char c1, char c2)
    {
        return toupper(c1) < toupper(c2);
    }

  public:  
    // konstruktor: inicjalizuje kryterium porownania
    RuntimeStringCmp (cmp_mode m=normal) : mode(m) {
    }

    // operacja porownania
    bool operator() (const string& s1, const string& s2) const {
        if (mode == normal) {
            return s1<s2;
        }
        else {
            return lexicographical_compare (s1.begin(), s1.end(),
                                            s2.begin(), s2.end(),
                                            nocase_compare);
        }
    }
};

/* typ kontenera:
 * - mapa zawierajaca
 *       - klucze typu string
 *       - wartosci typu string
 *       - specjalny typ obiektu realizujacego porownania
 */
typedef map<string,string,RuntimeStringCmp> StringStringMap;

// funkcja wypelniajaca i wyswietlajaca zawartosc takich kontenerow
void fillAndPrint(StringStringMap& coll);

int main()
{
    // utworz kontener o domyslnym kryterium porownania
    StringStringMap coll1;
    fillAndPrint(coll1);

    // utworz obiekt realizujacy porownania bez rozrozniania wielkosci liter
    RuntimeStringCmp ignorecase(RuntimeStringCmp::nocase);

    // utworz kontener o kryterium porownania nierozrozniajacym wielkosci liter
    StringStringMap coll2(ignorecase);
    fillAndPrint(coll2);
}

void fillAndPrint(StringStringMap& coll)
{
    // wstaw elementy w przypadkowej kolejnosci
    coll["Deutschland"] = "Niemcy";
    coll["deutsch"] = "niemiecki";
    coll["Haken"] = "haczyk";
    coll["arbeiten"] = "pracowac";
    coll["Hund"] = "pies";
    coll["gehen"] = "isc";
    coll["Unternehmen"] = "przedsiebiorstwo";
    coll["unternehmen"] = "podjac sie";
    coll["gehen"] = "isc";
    coll["Bestatter"] = "grabarz";

    // wypisz elementy
    StringStringMap::iterator pos;
    cout.setf(ios::left, ios::adjustfield);
    for (pos=coll.begin(); pos!=coll.end(); ++pos) {
        cout << setw(15) << pos->first.c_str() << " "
             << pos->second << endl;
    }
    cout << endl;
}
