/* The following code example is taken from the book
 * "The C++ Standard Library - A Tutorial and Reference, 2nd Edition"
 * by Nicolai M. Josuttis, Addison-Wesley, 2012
 *
 * (C) Copyright Nicolai M. Josuttis 2012.
 * Permission to copy, use, modify, sell and distribute this software
 * is granted provided this copyright notice appears in all copies.
 * This software is provided "as is" without express or implied
 * warranty, and with no claim as to its suitability for any purpose.
 */
#include <cstddef>
#include <memory>
#include <limits>

template <typename T>
class MyAlloc {
  public:
    // definicje typów
    typedef std::size_t    size_type;
    typedef std::ptrdiff_t difference_type;
    typedef T*             pointer;
    typedef const T*       const_pointer;
    typedef T&             reference;
    typedef const T&       const_reference;
    typedef T              value_type;

    // konstruktory i destruktor
    // - puste, bo alokator jest bezstanowy
    MyAlloc() throw() {
    }
    MyAlloc(const MyAlloc&) throw() {
    }
    template <typename U>
      MyAlloc (const MyAlloc<U>&) throw() {
    }
    ~MyAlloc() throw() {
    }

    // przydział (ale bez inicjalizacji) num elementów typu T
    T* allocate (std::size_t num, const void* hint = 0) {
        // przydział pamięci globalną wersją operatora new
        return static_cast<T*>(::operator new(num*sizeof(T)));
    }

    // zwolnienie obszaru pamięci p po num usuniętych elementach
    void deallocate (T* p, std::size_t num) {
        // zwolnienie pamięci globalną wersją operatora delete
        ::operator delete(p);
    }

    // zwraca adres wartości
    T* address (T& value) const {
        return &value;
    }
    const T* address (const T& value) const {
        return &value;
    }

    // zwraca maksymalną liczbę elementów alokacji
    std::size_t max_size () const throw() {
        return std::numeric_limits<std::size_t>::max() / sizeof(T);
    }

    // inicjalizuje elementy przydzielonej pamięci p wartością value
    void construct (T* p, const T& value) {
        // inicjalizacja pamięci miejscową wersją operatora new
        ::new((void*)p)T(value);
    }

    // usuwa elementy zainicjalizowanej pamięci p
    void destroy (T* p) {
        // usunięcie obiektów przez wywołanie ich destruktorów
        p->~T();
    }

    // przewiązanie alokatora do typu U
    template <typename U>
    struct rebind {
        typedef MyAlloc<U> other;
    };
};

// wszystkie specjalizacje alokatora mogą być stosowane zamiennie
template <typename T1, typename T2>
bool operator== (const MyAlloc<T1>&,
                 const MyAlloc<T2>&) throw() {
    return true;
}
template <typename T1, typename T2>
bool operator!= (const MyAlloc<T1>&,
                 const MyAlloc<T2>&) throw() {
    return false;
}

