/* The following code example is taken from the book
 * "The C++ Standard Library - A Tutorial and Reference, 2nd Edition"
 * by Nicolai M. Josuttis, Addison-Wesley, 2012
 *
 * (C) Copyright Nicolai M. Josuttis 2012.
 * Permission to copy, use, modify, sell and distribute this software
 * is granted provided this copyright notice appears in all copies.
 * This software is provided "as is" without express or implied
 * warranty, and with no claim as to its suitability for any purpose.
 */
#include <cstdio>
#include <streambuf>

// funkcja write():
#ifdef _MSC_VER
# include <io.h>
#else
# include <unistd.h>
#endif

class outbuf : public std::streambuf {
  protected:
    static const int bufferSize = 10;   // rozmiar bufora danych
    char buffer[bufferSize];            // bufor danych

  public:
    // konstruktor
    // - zainicjalizuj bufor danych
    // - jeden znak mniej w celu umożliwienia wywolania
    //   funkcji overflow() przez znak o indeksie bufferSize
    outbuf() {
        setp (buffer, buffer+(bufferSize-1));
    }

    // destruktor
    // - opróżnij bufor danych
    virtual ~outbuf() {
        sync();
    }

  protected:
    // opróżnij znaki z bufora
    int flushBuffer () {
        int num = pptr()-pbase();
        if (write (1, buffer, num) != num) {
            return EOF;
        }
        pbump (-num);    // odpowiednio wyzeruj wskaźnik zapisu
        return num;
    }

    // bufor pełny
    // - zapisz znak c oraz wszystkie poprzednie znaki
    virtual int_type overflow (int_type c) {
        if (c != EOF) {
            // wstaw znak do bufora
            *pptr() = c;
            pbump(1);
        }
        // opróźnij bufor
        if (flushBuffer() == EOF) {
            // BŁĄD
            return EOF;
        }
        return c;
    }

    // zsynchronizuj dane z plikiem 
    // - opróżnij dane z bufora
    virtual int sync () {
        if (flushBuffer() == EOF) {
            // BŁĄD
            return -1;
        }
        return 0;
    }
};
