﻿namespace AddisonWesley.Michaelis.EssentialCSharp.Chapter20.Listing20_20
{
    using System;
    using System.Runtime.InteropServices;
    using System.Windows.Forms;

    class Program
    {
        public unsafe static int Main()
        {
            // Przypisanie wartości do tablicy redpill 
            byte[] redpill = { 
          0x0f, 0x01, 0x0d,       // Asemblerowa instrukcja SIDT
          0x00, 0x00, 0x00, 0x00, // Miejsce na adres.
          0xc3};                  // Asemblerowa instrukcja return.

            unsafe
            {
                fixed(byte* matrix = new byte[6],
                    redpillPtr = redpill)
                {
                    // Przeniesienie adresu ze wskaźnika matrix bezpośrednio
                    // za instrukcję SIDT.
                    *(uint*)&redpillPtr[3] = (uint)&matrix[0];

                    using(VirtualMemoryPtr codeBytesPtr =
                        new VirtualMemoryPtr(redpill.Length))
                    {
                        Marshal.Copy(
                            redpill, 0,
                            codeBytesPtr, redpill.Length);

                        MethodInvoker method =
                            (MethodInvoker)Marshal.GetDelegateForFunctionPointer(
                            codeBytesPtr, typeof(MethodInvoker));

                        method();
                    }
                    if(matrix[5] > 0xd0)
                    {
                        Console.WriteLine("Jesteśmy w matriksie!\n");
                        return 1;
                    }
                    else
                    {
                        Console.WriteLine("Nie jesteśmy w matriksie.\n");
                        return 0;
                    }
                } // Koniec instrukcji fixed.
            } // Koniec instrukcji unsafe.
        }
    }

    public class VirtualMemoryPtr : System.Runtime.InteropServices.SafeHandle
    {
        public VirtualMemoryPtr(int memorySize) :
            base(IntPtr.Zero, true)
        {
            ProcessHandle =
                VirtualMemoryManager.GetCurrentProcessHandle();
            MemorySize = (IntPtr)memorySize;
            AllocatedPointer =
                VirtualMemoryManager.AllocExecutionBlock(
                memorySize, ProcessHandle);
            Disposed = false;
        }

        public readonly IntPtr AllocatedPointer;
        readonly IntPtr ProcessHandle;
        readonly IntPtr MemorySize;
        bool Disposed;

        public static implicit operator IntPtr(
            VirtualMemoryPtr virtualMemoryPointer)
        {
            return virtualMemoryPointer.AllocatedPointer;
        }

        // Abstrakcyjna składowa SafeHandle
        public override bool IsInvalid
        {
            get
            {
                return Disposed;
            }
        }

        // Abstrakcyjna składowa SafeHandle
        protected override bool ReleaseHandle()
        {
            if(!Disposed)
            {
                Disposed = true;
                GC.SuppressFinalize(this);
                VirtualMemoryManager.VirtualFreeEx(ProcessHandle,
                    AllocatedPointer, MemorySize);
            }
            return true;
        }
    }

    class VirtualMemoryManager
    {
        [DllImport("kernel32.dll", SetLastError = true)]
        private static extern IntPtr VirtualAllocEx(
            IntPtr hProcess,
            IntPtr lpAddress,
            IntPtr dwSize,
            AllocationType flAllocationType,
            uint flProtect);

        [DllImport("kernel32.dll", SetLastError = true)]
        static extern bool VirtualProtectEx(
            IntPtr hProcess, IntPtr lpAddress,
            IntPtr dwSize, uint flNewProtect,
            ref uint lpflOldProtect);

        [DllImport("kernel32.dll", EntryPoint = "GetCurrentProcess")]
        internal static extern IntPtr GetCurrentProcessHandle();

        [DllImport("kernel32.dll", SetLastError = true)]
        static extern bool VirtualFreeEx(
            IntPtr hProcess, IntPtr lpAddress,
            IntPtr dwSize, IntPtr dwFreeType);

        public static bool VirtualFreeEx(
            IntPtr hProcess, IntPtr lpAddress,
            IntPtr dwSize)
        {
            bool result = VirtualFreeEx(
                hProcess, lpAddress, dwSize,
                (IntPtr)MemoryFreeType.Decommit);
            if(!result)
            {
                throw new System.ComponentModel.Win32Exception();
            }
            return result;
        }

        public static bool VirtualFreeEx(
            IntPtr lpAddress, IntPtr dwSize)
        {
            return VirtualFreeEx(
                GetCurrentProcessHandle(), lpAddress, dwSize);
        }


        /// <summary>
        /// Typ alokacji pamięci. Ten parametr musi
        /// mieć jedną z określonych dalej wartości.
        /// </summary>
        [Flags]
        private enum AllocationType : uint
        {
            /// <summary>
            /// Alokuje fizycznie obszar na określone zarezerwowane 
            /// strony w pamięci lub w pliku stronicowania na dysku.
            /// Inicjuje tę pamięć wartością zero. 
            /// </summary>
            Commit = 0x1000,
            /// <summary>
            /// Rezerwuje zakres przestrzeni adresów wirtualnych 
            /// procesu bez alokowania fizycznego obszaru w
            /// pamięci lub w pliku stronicowania na dysku. 
            /// </summary>
            Reserve = 0x2000,
            /// <summary>
            /// Informuje, że dane z obszaru pamięci określanego przez
            /// parametry lpAddress i dwSize nie są już istotne. Tych 
            /// stron nie należy wczytywać ani zapisywać w pliku 
            /// stronicowania. Jednak dany blok pamięci może być 
            /// jeszcze używany, dlatego nie należy go zwalniać. Tej
            /// wartości nie można łączyć z innymi.
            /// </summary>
            Reset = 0x80000,
            /// <summary>
            /// Alokuje fizyczną pamięć z dostępem do odczytu i do zapisu. 
            /// Ta wartość jest dostępna tylko dla pamięci 
            /// AWE (ang. Address Windowing Extensions).
            /// </summary>
            Physical = 0x400000,
            /// <summary>
            /// Alokuje pamięć o najwyższym możliwym adresie. 
            /// </summary>
            TopDown = 0x100000,
        }

        /// <summary>
        /// Określa poziom ochrony alokowanego obszaru pamięci ze stronami.
        /// </summary>
        [Flags]
        private enum ProtectionOptions : uint
        {
            /// <summary>
            /// Umożliwia wykonywanie zajętego obszaru ze stronami. 
            /// Próba odczytu lub zapisu w tym obszarze
            /// spowoduje błąd naruszenia poziomu dostępu.
            /// </summary>
            Execute = 0x10,
            /// <summary>
            /// Zapewnia możliwość wykonywania i odczytu zajętego
            /// obszaru ze stronami. Próba zapisu w tym
            /// obszarze spowoduje błąd naruszenia poziomu dostępu.
            /// </summary>
            PageExecuteRead = 0x20,
            /// <summary>
            /// Umożliwia wykonywanie, odczyt i zapis w
            /// zajętym obszarze ze stronami.
            /// </summary>
            PageExecuteReadWrite = 0x40,
            // ...
        }

        /// <summary>
        /// Typ operacji zwalniającej pamięć.
        /// </summary>
        [Flags]
        private enum MemoryFreeType : uint
        {
            /// <summary>
            /// Zwalnia określony zajęty obszar stron. 
            /// Po tej operacji strony są uznawane za zarezerwowane (stan Reserved). 
            /// </summary>
            Decommit = 0x4000,
            /// <summary>
            /// Zwalnia określony obszar ze stronami. Po wykonaniu 
            /// tej operacji strony są uznawane za zwolnione (stan Free). 
            /// </summary>
            Release = 0x8000
        }

        public static IntPtr AllocExecutionBlock(
           int size, IntPtr hProcess)
        {
            IntPtr codeBytesPtr;
            codeBytesPtr = VirtualAllocEx(
                hProcess, IntPtr.Zero,
                (IntPtr)size,
                AllocationType.Reserve | AllocationType.Commit,
                (uint)ProtectionOptions.PageExecuteReadWrite);

            if(codeBytesPtr == IntPtr.Zero)
            {
                throw new System.ComponentModel.Win32Exception();
            }

            uint lpflOldProtect = 0;
            if(!VirtualProtectEx(
                hProcess, codeBytesPtr,
                (IntPtr)size,
                (uint)ProtectionOptions.PageExecuteReadWrite,
                ref lpflOldProtect))
            {
                throw new System.ComponentModel.Win32Exception();
            }
            return codeBytesPtr;
        }

        public static IntPtr AllocExecutionBlock(int size)
        {
            return AllocExecutionBlock(
                size, GetCurrentProcessHandle());
        }
    }
}