//----------------------------------------------------------------------------
// COMTHROW.H 
//
// Deklaracja klasy ECOMError i makr obsugujcych wyjtki COM. 
// Do uycia w programach tworzonych w Visual C++ i C++Builderze.
//
// Nazwy i typy parametrw podano wycznie dla inustracji.
//
// HCHECK(HRESULT hResult): 
//  Sprawdza poprawno przekazanego parametru HRESULT.
//  Generuje wyjtek ECOMError jeli parametr jest niepoprawny.
//  Struktura wyjtku zawiera standardowy komunikat systemowy.
//
// BCHECK(BOOL bCondition, HRESULT hDesiredResult): 
//  Sprawdza warto parametru boolowskiego.
//  Generuje wyjtek ECOMError w razie wartoci FALSE, uywajc w tym celu
//  wartoci parametru HRESULT.
//  Struktura wyjtku zawiera standardowy komunikat systemowy.
//
// WINAPI_CHECK(DWORD dwWin32ErrorCode):
//  Sprawdza warto przekazanego kodu bdu.
//  Generuje wyjtek ECOMError dla wartoci rnych od ERROR_SUCCESS.
//  Struktura wyjtku zawiera standardowy komunikat systemowy odpowiadajcy
//  kodowi bdu.
//  Warto HRESULT jest generowana na podstawie parametru.
//
// WINAPI_BCHECK(BOOL bCondition): 
//  Sprawdza warto parametru boolowskiego.
//  Generuje wyjtek ECOMError dla wartoci FALSE.
//  Struktura wyjtku zawiera standardowy komunikat systemowy odpowiadajcy
//  wartoci zwracanej przez funkcj GetLastError().
//  Warto HRESULT jest generowana na podstawie kodu bdu.
//
// HCHECK_FMT(HRESULT hResult, LPCTSTR szFormat, ...):
//  Sprawdza poprawno przekazanego parametru HRESULT.
//  Generuje wyjtek ECOMError jeli parametr jest niepoprawny.
//  Struktura wyjtku zawiera komunikat zdefiniowany przez uytkownika.
//
// BCHECK_FMT(BOOL bCondition, HRESULT hDesiredResult, 
//            LPCTSTR szFormat, ...):
//  Sprawdza warto parametru boolowskiego.
//  Generuje wyjtek ECOMError w razie wartoci FALSE, uywajc w tym celu
//  wartoci parametru HRESULT.
//  Struktura wyjtku zawiera komunikat zdefiniowany przez uytkownika.
//
// HCHECK_FMT_RES(HRESULT hResult, UINT uResourceID, ...):
//  Sprawdza poprawno przekazanego parametru HRESULT.
//  Generuje wyjtek ECOMError jeli parametr jest niepoprawny.
//  Struktura wyjtku zawiera komunikat zdefiniowany przez uytkownika,
//  pobrany z zasobw biecego moduu.
//
// BCHECK_FMT_RES(BOOL bCondition, HRESULT hDesiredResult, 
//                UINT uResourceID, ... ):
//  Sprawdza warto parametru boolowskiego.
//  Generuje wyjtek ECOMError w razie wartoci FALSE, uywajc w tym celu
//  wartoci parametru HRESULT.
//  Struktura wyjtku zawiera komunikat zdefiniowany przez uytkownika,
//  pobrany z zasobw biecego moduu.
//
// HCHECK_FMT_INSTANCE_RES(HRESULT hResult, 
//                         HINSTANCE hInstance, UINT uResourceID, ...): 
//  Sprawdza poprawno przekazanego parametru HRESULT.
//  Generuje wyjtek ECOMError jeli parametr jest niepoprawny.
//  Struktura wyjtku zawiera komunikat zdefiniowany przez uytkownika,
//  pobrany z zasobw zadanego moduu.
//
// BCHECK_FMT_INSTANCE_RES(BOOL bCondition, HRESULT hDesiredResult, 
//                         HINSTANCE hInstance, UINT uResourceID, ...):
//  Sprawdza warto parametru boolowskiego.
//  Generuje wyjtek ECOMError w razie wartoci FALSE, uywajc w tym celu
//  wartoci parametru HRESULT.
//  Struktura wyjtku zawiera komunikat zdefiniowany przez uytkownika,
//  pobrany z zasobw zadanego moduu.
//
// COMCHECK(HRESULT hResult, IUnknown* pUnk, RIID riid)
//  Sprawdza poprawno przekazanego parametru HRESULT.
//  W razie niepowodzenia generowanyn jest wyjtek z komunikatem pobranym
//  z obiektu ErrorInfo zwizanego z argumentem riid i interfejsem danym
//  przez pUnk.
//  Przydatny w aplikacjach-klientach pisanych w C++Builderze.
//
//----------------------------------------------------------------------------
#if !defined(COMTHROW_H__)
#define COMTHROW_H__

/*
// Typowa implemetnacja metod interfejsu COM z uyciem ww. makrodefinicji

STDMETHODIMP IMyInterface::DoSomething()
{
   HRESULT hResult = S_OK;

   try
   {
      CComPtr<IWhatever> ptrWhatever;
      HCHECK(ptrWhatever.CoCreateInstance(__uuidof(Whatever)));

      BOOL bFlag = FALSE;
      HCHECK_FMT(ptrWhatever->Method1(&bFlag), _T("Bd metody"));

      BCHECK(bFlag, E_FAILED);

      OFSTRUCT ofs;
      WINAPI_BCHECK(OpenFile(_T("Nie ma mnie!"), &ofs, OF_EXIST) 
                    != HFILE_ERROR);

      HKEY hKey = NULL; 
      WINAPI_CHECK(RegOpenKeyEx(HKEY_CURRENT_USER, "Klucz nie istnieje", 
                   0, KEY_ALL_ACCESS, &hKey));
      // etc...

   }
   catch(ECOMError& e) // Nasze wasne wyjtki
   {
      hResult = Error(e.Message, IID_IMyInterface, e.ErrorCode);
   }
   catch(std::exception& e) // Wyjtki standardowe
   {
      hResult = Error(e.what(), IID_IMyInterface, E_FAIL);
   }
   catch(Exception& e) // Inne wyjtki (np. VCL)
   {
      hResult = Error(e.Message, IID_IMyInterface, E_FAIL);
   }
   catch(...) // inne...
   {
      Error(_T("Wystpi nieoczekiwany bd!"), IID_IMyInterface, E_UNEXPECTED);
      hResult = E_UNEXPECTED;
   }

   return hResult;
}

// Tylko dla C++Buildera:
// Poniej zilustrowano typowe wykorzystanie makra COMCHECK w kliencie pisanym
// w C++Builderze (otoczki klientw generowane przez Visual C++ s wystarczajco
// "inteligentne"). W wygenerowanym automatycznie pliku "nazwa_TLB" zamiast:
template <class T> BSTR __fastcall
TCOMINazwarT<T>::GetString()
{
  BSTR String = 0;
  OLECHECK(this->GetString(Number, (BSTR*)&String));
  return String;
}

// ...wstaw definicj:
template <class T> BSTR __fastcall
TCOMINazwaT<T>::GetString()
{
  BSTR String = 0;
  COMCHECK(this->GetString(Number, (BSTR*)&String), (*this), iid);
  return String;
}

// Podobnie mona "opakowa" inne metody, co pozwala skorzysta z obsugi
// wyjtkw strukturalnych. Zamiast:
template <class T> HRESULT __fastcall
TCOMINazwaT<T>::DoSomething(void)
{
  return (*this)->DoSomething();
}

//...zapisz definicj:
template <class T> void __fastcall
TCOMINazwaT<T>::DoSomething(void)
{
  COMCHECK((*this)->DoSomething(), (*this), iid);
}

// A oto przykadowy kod klienta:
void __fastcall TForm1::Button1Click(TObject *Sender);
{
   try
   {
      TCOMINazwa FNazwa = CoNazwa::Create();
      FNazwa.DoSomething();
      WideString MyStr = FNazwa.GetString();
      // etc.
   }
   catch(ECOMError& e) // Nasze wyjtki
   {
      ShowMessage(e.Message);
   }
   catch(...) // pozostae
   {
      // ...
   }
}
*/

// Uyte pliki nagwkowe:
#include <tchar.h>
#include <stdio.h>

// Makra definiujce waciwoci:
#if (__BORLANDC__ >= 0x0530)
   #if !defined(PROPERTY)
      #define PROPERTY(aType, aName, aGet, aPut) \
         __property aType aName = {read = aGet, write = aPut}
   #endif
   #if !defined(READ_ONLY_PROPERTY)
      #define READ_ONLY_PROPERTY(aType, aName, aGet) \
         __property aType aName = {read = aGet}
   #endif
#elif (_MSC_VER >= 1100)
   #if !defined(PROPERTY)
         #define PROPERTY(aType, aName, aGet, aPut) \
            _declspec (property(get = aGet, put = aPut)) aType aName
   #endif
   #if !defined(READ_ONLY_PROPERTY)
         #define READ_ONLY_PROPERTY(aType, aName, aGet) \
            _declspec (property(get = aGet)) aType aName
   #endif
#else
   #error C++Builder 3 or higher and Visual C++ 5 or higher are the supported compilers.
#endif

#include <string>
#include <exception>

// Tylko C++Builder: klasy bazowej std::exception uyto nie tylko ze wzgldu na przenono,
// ale take poniewa wyjtki VCL nie daj si zgasza wielokrotnie.
class ECustomError : public std::exception
{
   typedef std::basic_string<TCHAR> tstrng;
private:
   tstrng FMessage;
public:
   ECustomError(LPCTSTR Msg)
   {
      FMessage = Msg;
   }

   PROPERTY(LPCTSTR, Message, GetMessage, SetMessage);
   LPCTSTR GetMessage() const { return FMessage.c_str(); }
   void SetMessage(LPCTSTR Msg) { FMessage = Msg; }

   // TCHAR nieobsugiwany (to le...)
   virtual const char* what () const
   {
      return FMessage.c_str();
   };

   virtual ~ECustomError()
   {
   }
};

// Klasa ECOMError 
class ECOMError : public ECustomError
{
protected:
   HRESULT FErrorCode;
   IErrorInfo* FErrorInfo;

   void PickStandardMessage()
   {
      LPTSTR lpszMsgBuf = NULL;

      FormatMessage(
          FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM,
          NULL,
          FErrorCode,
          MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), // Jzyk domylny
          reinterpret_cast<LPTSTR>(&lpszMsgBuf),
          0,
          NULL
      );

      Message = lpszMsgBuf;

      // Free the buffer.
      LocalFree(lpszMsgBuf);
   }

public:
   ECOMError(HRESULT HError = E_FAIL,
             LPCTSTR Msg = NULL) :
      ECustomError(_T("")),
      FErrorCode(HError),
      FErrorInfo(NULL)
   {
      if (Msg == NULL)
      {
         PickStandardMessage();
      }
      else
      {
         Message = Msg;
      }
   }

   ECOMError(HRESULT HError, IUnknown* pUnk, REFIID riid) :
      ECustomError(_T("")),
      FErrorCode(HError),
      FErrorInfo(NULL)
   {
      ISupportErrorInfo* pSupportEI = NULL;
      if ((pUnk != NULL) && 
          SUCCEEDED(pUnk->QueryInterface(IID_ISupportErrorInfo,
             reinterpret_cast<LPVOID*>(&pSupportEI))))
      {
         if (pSupportEI->InterfaceSupportsErrorInfo(riid) == S_OK)
         {
            GetErrorInfo(0, &FErrorInfo);
            if (FErrorInfo == NULL)
            {
               PickStandardMessage();
            }
            else
            {
               BSTR bstrDescription = NULL;
               FErrorInfo->GetDescription(&bstrDescription);
               #if defined(__BORLANDC__)
               AnsiString bstrTemp(bstrDescription);
               Message = bstrTemp.c_str();
               #else
               _bstr_t bstrTemp(bstrDescription);
               Message = bstrTemp;
               #endif
               SysFreeString(bstrDescription);
            }
            
            pSupportEI->Release();
         }
         else
         {
            PickStandardMessage();
         }
      }
      else
      {
         PickStandardMessage();
      }
   }

   ECOMError(ECOMError& obj) :
      ECustomError(obj.Message)
   {
      FErrorCode = obj.FErrorCode;
      FErrorInfo = obj.FErrorInfo;
      if (FErrorInfo != NULL)
         FErrorInfo->AddRef();
   }

   virtual ~ECOMError()
   {
      if (FErrorInfo != NULL)
      {
         FErrorInfo->Release();
         FErrorInfo = NULL;
      }
   }

   READ_ONLY_PROPERTY(HRESULT, ErrorCode, GetErrorCode);
   HRESULT GetErrorCode() const { return FErrorCode; }
};

typedef ECOMError CCOMException; // Dla Visual C++

//----------------------------------------------------------------------------
// Obsuga makra TRACE
//
// W Visual C++ wynik makra trafia do oknw Debug.
//
// W C++Builderze wynik trafia do okna Event Log (jeli uywamy ATL). 
// Projekt naley skompilowa z wczon opcj ATL "General Tracing" 
// (Project|Options, karta ATL). Do wywietlenia okna Event Log suy
// polecenie View|Debug Windows|Event Log.
//
//----------------------------------------------------------------------------
#if !defined(TRACE)
#if defined(ATLTRACE)
#define TRACE ATLTRACE
#else
// Brak ledzenia
inline void _cdecl FakeTrace(LPCTSTR , ...){}
#define TRACE 1 ? (void)0 : FakeTrace
#endif
#endif

//----------------------------------------------------------------------------
// Wyjtki COM
//
//----------------------------------------------------------------------------

// Zwyka informacja o bdzie
#define HCHECK(x) \
   {  \
      HRESULT hResult = (x);  \
      if (FAILED(hResult)) \
      {  \
         TRACE(_T("'%s' bd w pliku ='%s', w wierszu=%d, kod=0x%X\n"), #x, __FILE__, __LINE__, hResult); \
         throw (ECOMError((HRESULT)(hResult))); \
      }  \
   }

#define BCHECK(cond,hError)   \
   if (!(cond))   \
   {  \
      TRACE(_T("'%s' bd w pliku='%s', wiersz=%d, kod=0x%X\n"), #cond, __FILE__, __LINE__, hError); \
      throw (ECOMError((HRESULT)(hError)));  \
   }

// Bd Win32 API (wicej informacji od systemu)
#define WINAPI_CHECK(x) \
   {  \
      DWORD dwError = (x); \
      if (dwError != ERROR_SUCCESS) \
      {  \
         HRESULT hError = HRESULT_FROM_WIN32(dwError); \
         TRACE(_T("'%s' bd w pliku='%s', wiersz=%d, kod=0x%X\n"), #x, __FILE__, __LINE__, hError); \
         throw ECOMError(hError);   \
      }  \
   }

#define WINAPI_BCHECK(cond)   \
   if (!(cond))   \
   {  \
      HRESULT hError = HRESULT_FROM_WIN32(GetLastError()); \
      TRACE(_T("'%s' bd w pliku='%s', wiersz=%d, kod=0x%X\n"), #cond, __FILE__, __LINE__, hError); \
      throw ECOMError(hError);   \
   }

// Komunikaty wykorzystujce dodatkowe informacje i formatowanie
inline void __cdecl HcheckFmt(HRESULT hResult, LPCTSTR szFormat, ... )
{
   if (FAILED(hResult))
   {
      va_list args;
      va_start(args, szFormat);

      TCHAR szBuffer[1024];
      _vstprintf(szBuffer, szFormat, args);

      va_end(args);

      throw ECOMError(hResult, szBuffer);
   }
}

inline void __cdecl HcheckFmtRes(HRESULT hResult, UINT uID, ... )
{
   if (FAILED(hResult))
   {
      TCHAR szFormat[1024];
      const int nBufferMax = sizeof(szFormat)/sizeof(szFormat[0]);

      #if defined(_ATL)
      HINSTANCE hInstance = _Module.GetResourceInstance();
      #elif defined(_MFC_VER)
      HINSTANCE hInstance = AfxGetResourceHandle();
      #elif defined(VCL_H)
      HINSTANCE hInstance = HInstance;
      #else
      #error Wymagana biblioteka ATL, MFC lub VCL.
      #endif

      LoadString(hInstance, uID, szFormat, nBufferMax);

      va_list args;
      va_start(args, uID);

      TCHAR szBuffer[1024];
      _vstprintf(szBuffer, szFormat, args);

      va_end(args);

      throw ECOMError(hResult, szBuffer);
   }
}

inline void __cdecl HcheckFmtInstanceRes(HRESULT hResult, HINSTANCE hInstance,
                                         UINT uID, ... )
{
   if (FAILED(hResult))
   {
      TCHAR szFormat[1024];
      const int nBufferMax = sizeof(szFormat)/sizeof(szFormat[0]);

      LoadString(hInstance, uID, szFormat, nBufferMax);

      va_list args;
      va_start(args, uID);

      TCHAR szBuffer[1024];
      _vstprintf(szBuffer, szFormat, args);

      va_end(args);

      throw ECOMError(hResult, szBuffer);
   }
}

// Komunikaty wykorzystujce dodatkowe informacje i formatowanie
inline void __cdecl BcheckFmt(BOOL bCond, HRESULT hResult,
                              LPCTSTR szFormat, ... )
{
   if (!(bCond))
   {
      va_list args;
      va_start(args, szFormat);

      TCHAR szBuffer[1024];
      _vstprintf(szBuffer, szFormat, args);

      va_end(args);

      throw ECOMError(hResult, szBuffer);
   }
}

inline void __cdecl BcheckFmtRes(BOOL bCond, HRESULT hResult, UINT uID, ...)
{
   if (!(bCond))
   {
      TCHAR szFormat[1024];
      const int nBufferMax = sizeof(szFormat)/sizeof(szFormat[0]);

      #if defined(_ATL)
      HINSTANCE hInstance = _Module.GetResourceInstance();
      #elif defined(_MFC_VER)
      HINSTANCE hInstance = AfxGetResourceHandle();
      #elif defined(VCL_H)
      HINSTANCE hInstance = HInstance;
      #else
      #error ATL, MFC or VCL required.
      #endif

      LoadString(hInstance, uID, szFormat, nBufferMax);

      va_list args;
      va_start(args, uID);

      TCHAR szBuffer[1024];
      _vstprintf(szBuffer, szFormat, args);

      va_end(args);

      throw ECOMError(hResult, szBuffer);
   }
}

inline void __cdecl BcheckFmtInstanceRes(BOOL bCond, HRESULT hResult,
                                         HINSTANCE hInstance,
                                         UINT uID, ...)
{
   if (!(bCond))
   {
      TCHAR szFormat[1024];
      const int nBufferMax = sizeof(szFormat)/sizeof(szFormat[0]);

      LoadString(hInstance, uID, szFormat, nBufferMax);

      va_list args;
      va_start(args, uID);

      TCHAR szBuffer[1024];
      _vstprintf(szBuffer, szFormat, args);

      va_end(args);

      throw ECOMError(hResult, szBuffer);
   }
}

#define HCHECK_FMT HcheckFmt

#define HCHECK_FMT_RES HcheckFmtRes

#define HCHECK_FMT_INSTANCE_RES HcheckFmtInstanceRes

#define BCHECK_FMT BcheckFmt

#define BCHECK_FMT_RES BcheckFmtRes

#define BCHECK_FMT_INSTANCE_RES BcheckFmtInstanceRes

//----------------------------------------------------------------------------
// Obsuga bdw po stronie klienta
// Ponisze makro lepiej nadaje si do uycia w C++Builderze ni OLECHECK.
// (zob. przykad na pocztku pliku)
#define COMCHECK(x,pUnk,riid) \
   {  \
      HRESULT hResult = (x);  \
      if (FAILED(hResult)) \
      {  \
         TRACE(_T("'%s' bd w pliku='%s', wiersz=%d, kod=0x%X\n"), #x, __FILE__, __LINE__, hResult); \
         throw (ECOMError(hResult, pUnk, riid));   \
      }  \
   }

#endif //COMTHROW_H__

