//***************************  HuffmanCoding.h  **************************

#include <vector>
#include <algorithm>

using namespace std;

class HuffmanNode {
public:
    char symbol;
    unsigned long codeword, freq;
    unsigned int runLen, codewordLen;
    HuffmanNode *left, *right;
    HuffmanNode() {
        left = right = 0;
    }
    HuffmanNode(char s, unsigned long f, unsigned int r, 
        HuffmanNode *lt = 0, HuffmanNode *rt = 0) {
        symbol = s; freq = f; runLen = r; left = lt; right = rt;
    }
};

class ListNode {
public:
    HuffmanNode *tree;
    ListNode *next, *prev;
    ListNode() {
        next = prev = 0;
    }
    ListNode(ListNode *p, ListNode *n) {
        prev = p; next = n;
    }
};

class DataRec {
public:
    char symbol;
    unsigned int runLen;
    unsigned long freq;
    DataRec() {
    }
    bool operator== (const DataRec& dr) const { // uywany przez find();
        return symbol == dr.symbol && runLen == dr.runLen;
    }
    bool operator< (const DataRec& dr) const {  // uywany przez sort();
        return freq < dr.freq;
    }
};

class HuffmanCoding {
public:
    HuffmanCoding() : mask(0xff), bytes(4), bits(8), ASCII(256) {
        chars = new HuffmanNode*[ASCII+1];
    }
    void compress(char*,ifstream&);
    void decompress(char*,ifstream&);
private:
    const unsigned int bytes, bits, ASCII;
    unsigned int dataSize;
    const unsigned long mask;
    unsigned long charCnt;
    ofstream fOut;
    HuffmanNode *HuffmanTree, **chars;
    vector<DataRec> data;
    void error(char *s) {
        cerr << s << endl; exit(1);
    }
    void output(unsigned long pack);
    void garnerData(ifstream&);
    void outputFrequencies(ifstream&);
    void read2ByteNum(unsigned int&,ifstream&);
    void read4ByteNum(unsigned long&,ifstream&);
    void inputFrequencies(ifstream&);
    void createHuffmanTree();
    void createCodewords(HuffmanNode*,unsigned long,int);
    void transformTreeToArrayOfLists(HuffmanNode*);
    void encode(ifstream&);
    void decode(ifstream&);
};

void HuffmanCoding::output(unsigned long pack) {
    char *s = new char[bytes];
    int i;
    for (i = bytes - 1; i >= 0; i--) {
        s[i] = pack & mask;
        pack >>= bits;
    }
    for (i = 0; i < bytes; i++)
        fOut.put(s[i]);
}

void HuffmanCoding::garnerData(ifstream& fIn) {
    char ch, ch2;
    DataRec r;
    vector<DataRec>::iterator i;
    r.freq = 1;
    for (fIn.get(ch); !fIn.eof(); ch = ch2) {
        for (r.runLen = 1, fIn.get(ch2); !fIn.eof() && ch2 == ch; r.runLen++)
            fIn.get(ch2);
        r.symbol = ch;
        if ((i = find(data.begin(),data.end(),r)) == data.end())
             data.push_back(r);
        else i->freq++;
    }
    sort(data.begin(),data.end());
}

void HuffmanCoding::outputFrequencies(ifstream& fIn) {
    unsigned long temp4;
    char ch = data.size(); 
    unsigned int temp2 = data.size();
    temp2 >>= bits;
    fOut.put(char(temp2)).put(ch);
    fIn.clear();
    output((unsigned long)fIn.tellg());
    for (int j = 0; j < data.size(); j++) {
        fOut.put(data[j].symbol);
        ch = temp2 = data[j].runLen;
        temp2 >>= bits;
        fOut.put(char(temp2)).put(ch);
        temp4 = data[j].freq;
        output(temp4);
    }
}

void HuffmanCoding::read2ByteNum(unsigned int& num, ifstream& fIn) {
    num = fIn.get();
    num <<= bits;
    num |= fIn.get();
}

void HuffmanCoding::read4ByteNum(unsigned long& num, ifstream& fIn) {
    num = (unsigned long) fIn.get();
    for (int i = 1; i < 4; i++) {
        num <<= bits;
        num |= (unsigned long) fIn.get();
    }
}

void HuffmanCoding::inputFrequencies(ifstream& fIn) {
    DataRec r;
    read2ByteNum(dataSize,fIn);
    read4ByteNum(charCnt,fIn);
    data.reserve(dataSize);
    for (int j = 0; !fIn.eof() && j < dataSize; j++) {
        r.symbol = fIn.get();
        read2ByteNum(r.runLen,fIn);
        read4ByteNum(r.freq,fIn);
        data.push_back(r);
    }
}

void HuffmanCoding::createHuffmanTree() {
    ListNode *p, *newNode, *head, *tail;
    unsigned long newFreq;   
    head = tail = new ListNode;             // inicjalizacja wskanikw list;
    head->tree = new HuffmanNode(data[0].symbol,data[0].freq,data[0].runLen);
    for (int i = 1; i < data.size(); i++) { // stworzenie pozostaej czci list;
        tail->next = new ListNode(tail,0);
        tail = tail->next;
        tail->tree =
            new HuffmanNode(data[i].symbol,data[i].freq,data[i].runLen);
    }
    while (head != tail) {                  // stworzenie jednego drzewa Huffmana;
        newFreq = head->tree->freq + head->next->tree->freq; // dwie najmniejsze czstoci
        for (p = tail; p != 0 && p->tree->freq > newFreq; p = p->prev);
        newNode = new ListNode(p,p->next);
        p->next = newNode;
        if (p == tail)
             tail = newNode;
        else newNode->next->prev = newNode;
        newNode->tree =
            new HuffmanNode('\0',newFreq,0,head->tree,head->next->tree);
        head = head->next->next;
        delete head->prev->prev;
        delete head->prev;
        head->prev = 0;
    }
    HuffmanTree = head->tree;
    delete head;
}

void HuffmanCoding::createCodewords(HuffmanNode *p, unsigned long codeword, int level) {
    if (p->left == 0 && p->right == 0) {         // p jest liciem,
         p->codeword    = codeword;              // zapisanie sowa kodowego
         p->codewordLen = level;                 // oraz jego dugoci,
    }
    else {                                       // w przeciwnym przypadku - dodanie 0
         createCodewords(p->left,  codeword<<1,   level+1); // dla lewej krawdzi
         createCodewords(p->right,(codeword<<1)+1,level+1); // i 1 dla prawej krawdzi;
    }
}

void HuffmanCoding::transformTreeToArrayOfLists(HuffmanNode *p) {
    if (p->left == 0 && p->right == 0) {       // jeli p jest liciem,
         p->right = chars[(unsigned char)p->symbol]; // doczamy p do 
         chars[(unsigned char)p->symbol] = p;  // listy skojarzonej z 
    }                                          // symbolem odnalezionym w p;
    else {
         transformTreeToArrayOfLists(p->left);
         transformTreeToArrayOfLists(p->right);
    }
}

void HuffmanCoding::encode(ifstream& fIn) {
    unsigned long packCnt = 0, hold, maxPack = bytes*bits, pack = 0;
    char ch, ch2;
    int bitsLeft, runLength;
    HuffmanNode *p;
    for (fIn.get(ch); !fIn.eof(); ) {
        for (runLength = 1, fIn.get(ch2); !fIn.eof() && ch2 == ch; runLength++)
            fIn.get(ch2);
        for (p = chars[(unsigned char) ch];
             p != 0 && runLength != p->runLen; p = p->right)
            ;
        if (p == 0)
             error("Problem w metodzie encode()");
        if (p->codewordLen < maxPack - packCnt) {   // jeli w pack jest wystarczajco
             pack = (pack << p->codewordLen) | p->codeword; // duo miejsca na zapisanie nowego
             packCnt += p->codewordLen;             // sowa kodowego, naley przesun 
        }                                           // zawarto pack w lewo
                                                    // i doczy nowe sowo kodowe;
        else {                                      // w przeciwnym razie naley przesun
             bitsLeft = maxPack - packCnt;          // zawarto pack 
             pack <<= bitsLeft;                     // w lewo o ilo
             if (bitsLeft != p->codewordLen) {      // wolnych bitw
                  hold = p->codeword;               // i jeli nowe sowo kodowe 
                  hold >>= p->codewordLen - bitsLeft;// jest dusze od iloci 
                  pack |= hold;                     // wolnego miejsca, przekaza tylko
             }                                      // tyle bitw, ile mona zapisa
                                                    // w pack;
             else pack |= p->codeword;              // jeli nowe sowo kodowe 
                                                    // dokadnie pasuje do 
                                                    // pack, naley je przekaza;
             output(pack);                          // generacja pack jako 
                                                    // cigu czterech znakw ;
             if (bitsLeft != p->codewordLen) {      // zapisanie nieprzetworzonych
                  pack = p->codeword;               // bitw nowego 
                  packCnt = maxPack - (p->codewordLen - bitsLeft); // sowa kodowego
                  packCnt = p->codewordLen - bitsLeft; // w pack;
             }
             else packCnt = 0;
        }
        ch = ch2;
    }
    if (packCnt != 0) {
        pack <<= maxPack - packCnt; // przekazanie pozostaych sw kodowych 
        output(pack);               // i kilku 0.
    }
}

void HuffmanCoding::compress(char *inFileName, ifstream& fIn) {
    char outFileName[30];
    strcpy(outFileName,inFileName);
    if (strchr(outFileName,'.'))               // jeli jest rozszerzenie
         strcpy(strchr(outFileName,'.')+1,"z");// zostaje zmienione na 'z'
    else strcat(outFileName,".z");             // dodanie rozszerzenia '.z';
    fOut.open(outFileName,ios::out);
//wiersz wykorzystywany na komputerach PC:
//    fOut.open(outFileName,ios::out|ios::binary);
    garnerData(fIn);
    outputFrequencies(fIn);
    createHuffmanTree();
    createCodewords(HuffmanTree,0,0);
    for (int i = 0; i <= ASCII; i++)
        chars[i] = 0;
    transformTreeToArrayOfLists(HuffmanTree);
    fIn.clear();      // wyczyszczenie flagi koca pliku;
    fIn.seekg(0,ios::beg);
    encode(fIn);
    cout.precision(2);
    cout << "Stopie kompresji = " <<
            100.0*(fIn.tellg()-fOut.tellp())/fIn.tellg() << "%\n"
         << "Stopie kompresji bez tablicy = " <<
            100.0*(fIn.tellg()-fOut.tellp()+data.size()*(2+4))/fIn.tellg() << "%\n";
    fOut.close();
}

void HuffmanCoding::decode(ifstream& fIn) {
    unsigned long chars;
    char ch, bitCnt = 1, mask = 1;
    mask <<= bits - 1;        // zamiana 00000001 na 100000000;
    for (chars = 0, fIn.get(ch); !fIn.eof() && chars < charCnt; ) {
        for (HuffmanNode *p = HuffmanTree; ; ) {
            if (p->left == 0 && p->right == 0) {
                for (int j = 0; j < p->runLen; j++)
                    fOut.put(p->symbol);
                chars += p->runLen;
                break;
            }
            else if ((ch & mask) == 0)
                 p = p->left;
            else p = p->right;
            if (bitCnt++ == bits) { // odczyt kolejnego znaku z fIn,
                 fIn.get(ch);       // jeli wszystkie bity w ch zostay sprawdzone;
                 bitCnt = 1;
            }                       // w przeciwnym razie wszystkie bity w ch
            else ch <<= 1;          // s przesuwane w lewo o jedn pozycj;
        }
    }
}

void HuffmanCoding::decompress(char *inFileName, ifstream& fIn) {
    char outFileName[30];
    strcpy(outFileName,inFileName);
    if (strchr(outFileName,'.'))                  // jeli jest rozszerzenie 
         strcpy(strchr(outFileName,'.')+1,"dec"); // zostaje zmienione na 'z'
    else strcat(outFileName,".dec");              // dodanie rozszerzenia '.z';
    fOut.open(outFileName,ios::out);
//wiersz wykorzystywany na komputerach PC:
//    fOut.open(outFileName,ios::out|ios::binary);
    inputFrequencies(fIn);
    createHuffmanTree();
    createCodewords(HuffmanTree,0,0);
    decode(fIn);
    fOut.close();
}
