﻿using System;

namespace AddisonWesley.Michaelis.EssentialCSharp.Chapter08.Table08_01
{
    // 1.
    namespace StaticMembers
    {
        public interface ISampleInterface
        {
            static string? _Field;
            public static string? Field { get => _Field; private set => _Field = value; }
            static ISampleInterface() => Field = "Nelson Mandela";
            public static string? GetField() => Field;
        }
    }

    // 2.
    namespace ImplementedPropertiesAndMethods
    {
        public interface IPerson
        {
            // Definicje standadowych właściwości abstrakcyjnych
            string FirstName { get; set; }
            string LastName { get; set; }

            // Implementacja właściwości i metod instancji
            public string Name { get => GetName(); }
            public string GetName() => $"{FirstName} {LastName}";
        }
        public class Person : IPerson
        {
            public Person(string firstName, string lastName)
            {
                (FirstName, LastName) = (firstName, lastName);
            }
            private string? _FirstName;
            public string FirstName
            {
                get => _FirstName!;
                set => _FirstName = value ?? throw new ArgumentNullException(nameof(value));
            }
            private string? _LastName;
            public string LastName
            {
                get => _LastName!;
                set => _LastName = value ?? throw new ArgumentNullException(nameof(value));
            }
        }
        public class Employee : Person
        {
            public Employee(string firstName, string lastName)
                : base(firstName, lastName) { }
            public string Name => $"{LastName}, {FirstName}";
        }
        public class Program
        {
            static public void Main()
            {
                Person inigo = new Person("Inigo", "Montoya");
                Console.WriteLine(
                    ((IPerson)inigo).Name);
                Employee employee = new Employee("Mark", "Michaelis");
                Console.WriteLine(
                    ((IPerson)employee).Name);
                Console.WriteLine(
                    employee.Name);
            }
        }
    }

    // 3.
    namespace PublicAccessModifiers
    {
        public interface IPerson
        {
            // Wszystkie składowe domyślnie są publiczne
            string FirstName { get; set; }
            public string LastName { get; set; }
            string Initials => $"{FirstName[0]}{LastName[0]}";
            public string Name => GetName();
            public string GetName() => $"{FirstName} {LastName}";
        }
    }

    // 4.
    namespace ProtectedAccessModifiers
    {
        
    }

    // 5. 
    namespace ProvateAccessModifiers
    {
        public interface IPerson
        {
            string FirstName { get; set; }
            string LastName { get; set; }
            string Name => GetName();
            private string GetName() =>
                $"{FirstName} {LastName}";
        }
    }

    // 6. 
    namespace InternalAccessModifiers
    {
        public interface IPerson
        {
            string FirstName { get; set; }
            string LastName { get; set; }
            string Name => GetName();
            internal string GetName() =>
                $"{FirstName} {LastName}";
        }
    }

    // 7. 
    namespace ProtectedInternalAccessModifiers
    {
        public interface IPerson
        {
            string FirstName { get; set; }
            string LastName { get; set; }
            string Name => GetName();
            protected internal string GetName() =>
                $"{FirstName} {LastName}";
        }
    }

    // 8. 
    namespace PrivateProtectedAccessModifiers
    {
    class Program
    {
        static void Main()
        {
            IPerson? person = null;
                // W klasach niepochodnych nie można wywoływać
                // składowej z modyfikatorem private protected.
                // person?.GetName();
                Console.WriteLine(person);
        }
    }
    public interface IPerson
    {
        string FirstName { get; }
        string LastName { get; }
        string Name => GetName();
        private protected string GetName() =>
            $"{FirstName} {LastName}";
    }
    public interface IEmployee: IPerson
    {
        int EmpoyeeId => GetName().GetHashCode();
    }
    public class Person : IPerson
    {
        public Person(string firstName, string lastName)
        {
            FirstName = firstName ??
                throw new ArgumentNullException(nameof(firstName));
            LastName = lastName ?? 
                throw new ArgumentNullException(nameof(lastName));
        }
        public string FirstName { get; }
        public string LastName { get; }

            // Składowe interfejsu mające modyfikator private protected
            // są niedostępne w klasach pochodnych.

            // public int PersonTitle => 
            //      GetName().ToUpper();
        }
    }

    // 9. 
    namespace VirtualMembers
    {
        public interface IPerson
        {
            // Modyfikator virtual jest niedozwolony
            // dla składowych bez implementacji.
            /* virtual */
            string FirstName { get; set; }
            string LastName { get; set; }
            virtual string Name => GetName();
            private string GetName() =>
                $"{FirstName} {LastName}";
        }
    }

    // 10.
    namespace SealedMembers
    {
        public interface IWorkflowActivity
        {
            // Prywatna, a więc niewirtualna
            private void Start() =>
                Console.WriteLine(
                    "IWorkflowActivity.Start()...");

            // Modyfikator sealed, aby uniemożliwić przesłanianie
            sealed void Run()
            {
                try
                {
                    Start();
                    InternalRun();
                }
                finally
                {
                    Stop();
                }
            }

            protected void InternalRun();

            // Prywatna, a więc niewirtualna
            private void Stop() =>
                Console.WriteLine(
                    "IWorkflowActivity.Stop()..");
        }
    }

    // 11. 
    namespace AbstractMembers
    {
        public interface IPerson
        {
            // Modyfikator virtual jest niedozwolony
            // dla składowych bez implementacji
            /* virtual */
            abstract string FirstName { get; set; }
            string LastName { get; set; }
            // Modyfikator abstract jest niedozwolony
            // dla składowych bez implementacji
            /* abstract */ string Name => GetName();
            private string GetName() =>
                $"{FirstName} {LastName}";
        }
    }

    // 12. 
    namespace PartialMembers
    {
        public partial interface IThing
        {
            string Value { get; protected set; }
            void SetValue(string value)
            {
                AssertValueIsValid(value);
                Value = value;
            }

            partial void AssertValueIsValid(string value);
        }

        public partial interface IThing
        {
            partial void AssertValueIsValid(string value)
            {
                // Błędna wartość powoduje zgłoszenie wyjątku.
                switch (value)
                {
                    case null:
                        throw new ArgumentNullException(
                            nameof(value));
                    case "":
                        throw new ArgumentException(
                            "Pusty łańcuch jest niedozwolony", nameof(value));
                    case string _ when string.IsNullOrWhiteSpace(value):
                        throw new ArgumentException(
                            "Spacje są niedozwolone", nameof(value));
                };
            }
        }
    }
}
