﻿using Microsoft.EntityFrameworkCore; // ExecuteUpdate, ExecuteDelete
using Microsoft.EntityFrameworkCore.ChangeTracking; // EntityEntry<T>
using Biblioteka.ModeleEncji; // Northwind, Product

partial class Program
{
   static void WypiszProdukty(int[]? idProduktowDoWyroznienia = null)
   {
      using Northwind db = new();

      if ((db.Products is null) || (!db.Products.Any()))
      {
         Blad("Nie znaleziono produktów.");
         return;
      }

      WriteLine("| {0,-3} | {1,-35} | {2,8} | {3,5} | {4} |",
        "Id", "Nazwa produktu", "Cena", "Sztuk", "Nieprod.");

      foreach (Product p in db.Products)
      {
         ConsoleColor poprzedniKolor = ForegroundColor;

         if ((idProduktowDoWyroznienia is not null) &&
           idProduktowDoWyroznienia.Contains(p.ProductID))
         {
            ForegroundColor = ConsoleColor.Green;
         }

         WriteLine("| {0:000} | {1,-35} | {2,8:$#,##0.00} | {3,5} | {4} |",
           p.ProductID, p.ProductName, p.Koszt, p.WMagazynie, p.Discontinued);
         ForegroundColor = poprzedniKolor;
      }
   }

   private static (int zmienionych, int IdProduktu) DodajProdukt(
      int idKategorii, string nazwaProduktu, decimal? cena, short? wMagazynie)
   {
      using Northwind db = new();
      if (db.Products is null) return (0, 0);

      Product p = new()
      {
         CategoryID = idKategorii,
         ProductName = nazwaProduktu,
         Koszt = cena,
         WMagazynie = wMagazynie
      };

      // Zaznacz produkt jako dodany do śledzenia zmian.
      EntityEntry<Product> encja = db.Products.Add(p);
      WriteLine($"Stan: {encja.State}, ID produktu: {p.ProductID}");
      // Save tracked change to database.
      int zmienionych = db.SaveChanges();
      WriteLine($"Stan: {encja.State}, ID produktu: {p.ProductID}");

      return (zmienionych, p.ProductID);
   }

   static (int zmienione, int idProduktu) ZwiekszCeneProduktu(
      string poczatekNazwy, decimal kwota)
   {
      using (Northwind db = new())
      {
         if (db.Products is null) return (0, 0);

         // pobierz pierwszy produkt, którego nazwa zaczyna się od wartości 
         // parametru nazwa
         Product produktDoAktualizacji = db.Products.First(
            p => p.ProductName.StartsWith(poczatekNazwy));

         produktDoAktualizacji.Koszt += kwota;

         int zmienione = db.SaveChanges();

         return (zmienione, produktDoAktualizacji.ProductID);
      }
   }


   static int UsunProdukty(string poczatekNazwy)
   {
      using (Northwind db = new())
      {
         IQueryable<Product>? produkty = db.Products?.Where(
            p => p.ProductName.StartsWith(poczatekNazwy));

         if ((produkty is null) || (!produkty.Any()))
         {
            WriteLine("Nie znaleziono produktów do usunięcia.");
            return 0;
         }
         else
         {
            if (db.Products is null) return 0;

            db.Products.RemoveRange(produkty);
         }

         int zmienione = db.SaveChanges();
         return zmienione;
      }
   }


   static (int zmienione, int[]? idProduktow) SzybkszeZwiekszanieCenProduktow(
      string poczatekNazwy, decimal kwota)
   {
      using Northwind db = new();

      if (db.Products is null) return (0, null);

      // Pobieranie produktów, których nazwy zaczynają się podanym ciągiem znaków
      IQueryable<Product>? produkty = db.Products.Where(
        p => p.ProductName.StartsWith(poczatekNazwy));

      int zmienione = produkty.ExecuteUpdate(s => s.SetProperty(
        p => p.Koszt, // Wyrażenie lambda wybierające właściwość
        p => p.Koszt + kwota)); // Wyrażenie lambda z wartością modyfikującą

      int[] idProduktow = produkty.Select(p => p.ProductID).ToArray();

      return (zmienione, idProduktow);

   }


   static int LepszeUsuwanieProduktow(string poczatekNazwy)
   {
      using (Northwind db = new())
      {
         int zmienione = 0;

         IQueryable<Product>? produkty = db.Products?.Where(
           p => p.ProductName.StartsWith(poczatekNazwy));

         if ((produkty is null) || (!produkty.Any()))
         {
            WriteLine("Nie znaleziono produktów do usunięcia.");
            return 0;
         }
         else
         {
            zmienione = produkty.ExecuteDelete();
         }
         return zmienione;
      }
   }

}
