import javax.swing.JTable;
import javax.swing.JFrame;
import javax.swing.JScrollPane;
import javax.swing.JPanel;
import javax.swing.JButton;

import java.awt.BorderLayout;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowEvent;
import java.awt.event.WindowAdapter;

import java.util.ArrayList;

import java.sql.DriverManager;
import java.sql.Connection;
import java.sql.Statement;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;

public class CustomerApp extends JFrame implements ActionListener {

//An instance of the model that feeds the table that lists the customer details:

	private CustomerTableModel model;

//Table in which the customer details are listed:

	private JTable table;
//Connection to the PostgreSQL database:

	private Connection con;

//Panel that contains the form for entering the customer data:

	private CustomerPanel customerPanel;
	
	public CustomerApp(String[] arg) throws Exception {
		
		super("Customer Management System");

//Get a connection to the database:

		getConnection(arg[0],arg[1],arg[2]);

//Create an instance of the table model and populate it with the data:

		model = new CustomerTableModel();
		refreshData();

//Create the table using the model and add the table to a scroll pane. The pane is then added to the parent frame:

		table = new JTable(model);
		table.setAutoCreateColumnsFromModel(true);
		JScrollPane pane = new JScrollPane(table);
		
		getContentPane().setLayout(new BorderLayout());
		
		getContentPane().add(pane,BorderLayout.CENTER);

//Add two buttons, one for adding new customers and one for deleting selected customers. The current class is registered as the action listener for the two buttons:

		JPanel buttonPanel = new JPanel();
		JButton newButton = new JButton("Add Customer");
		newButton.addActionListener(this);
		buttonPanel.add(newButton);
		JButton deleteButton = new JButton("Remove Customer");
		deleteButton.addActionListener(this);
		buttonPanel.add(deleteButton);
		getContentPane().add(buttonPanel,BorderLayout.SOUTH);

//Create an instance of the customer panel and add it to the frame:

		customerPanel = new CustomerPanel();
		getContentPane().add(customerPanel,BorderLayout.WEST);



//Display the frame:
		pack();
		show();
		setLocation(50,50);
		setSize(800,375);
		setResizable(false);
		
		validate();

//Add an inner class that listens to the window closing events and closes the database connections:

		addWindowListener(new WindowAdapter() {
			
			public void windowClosing(WindowEvent e) {
				try {
					con.close();
				}catch(SQLException ex) {
					ex.printStackTrace();
				}
				System.exit(0);
			}
			
		});
		
	}
	 
	public static void main(String args[]) throws Exception {

//The main method expects the JDBC URL, user name and password for the database as commandline arguments:
		if(args == null || args.length != 3) {
			System.out.print("Usage");
			System.out.println("java EmployeeApp 
				<jdbcURL> <user> <passwd>");
			return;
		}
		CustomerApp app = new CustomerApp(args);		
		
	}

//Callback method for listening to the button actions:

	public void actionPerformed(ActionEvent e) {
		
		JButton button = (JButton)e.getSource();
		
		try {

//If the button for adding customers is clicked call the method to add customers:

			if("Add Customer".equals(button.getText())) {
				addCustomer();

//If the button for removing customers is clicked call the method to remove customers:

			}else if("Remove Customer".equals(button.getText())) {
				removeCustomer();
			}
		}catch(SQLException ex) {
			ex.printStackTrace();
		}
		
		validate();
		
	}

//This method gets a connection to the database:

	private void getConnection(String url,String user,String passwd) 
		throws Exception {
		
		Class.forName("org.postgresql.Driver");
		con = DriverManager.getConnection(url,user,passwd);
		
	}

//This method refreshes the table model data:

	private void refreshData() throws SQLException {
		
		String sql = "select * from customer";
		Statement stmt = con.createStatement();
		ResultSet res = stmt.executeQuery(sql);
		
		ArrayList list = new ArrayList();
		while(res.next()) {
			Customer cust = new Customer(res.getInt(1),
				res.getString(2),res.getString(3),
				res.getString(4),res.getString(5),
				res.getString(6),res.getString(7),
				res.getString(8));
			list.add(cust);
		}
		
		model.setCustomerList(list);
		
		res.close();
		stmt.close();
	}

//This method inserts a record to the database:

	private void addCustomer() throws SQLException {
		
		String sql = "insert into customer(" + 
					"title,fname,lname, " + 
					"addressline,town,zipcode,phone)" +
					  "values(" +
					  "?,?,?,?,?,?,?)";

//Prepare the statement:

		PreparedStatement stmt = con.prepareStatement(sql);
		
		Customer cust = customerPanel.getCustomer();

//Set the SQL IN parameters:

		stmt.setString(1,cust.getTitle());
		stmt.setString(2,cust.getFname());
		stmt.setString(3,cust.getLname());
		stmt.setString(4,cust.getAddressLine());
		stmt.setString(5,cust.getTown());
		stmt.setString(6,cust.getZipCode());
		stmt.setString(7,cust.getPhone());

//Execute the SQL:

		stmt.executeUpdate();		
		
		stmt.close();

//Refresh the table:

		refreshData();
		customerPanel.reset();
		
	}
//This method deletes a record from the database:

	private void removeCustomer() throws SQLException {
		
		String sql = "delete from customer where customer_id = ?";
		PreparedStatement stmt = con.prepareStatement(sql);

//Get the selected customer:

		int selectedRow = table.getSelectedRow();
		
		Customer cust = model.getCustomer(selectedRow);
		if(cust == null) {
			return;
		}

//Set the customer id:

		stmt.setInt(1,cust.getCustomerId());  

//Execute the delete SQL:
		stmt.executeUpdate();	 
		
		stmt.close();		

//Remove the customer from the model:

		model.removeCustomer(selectedRow);
		
	}
}
