/*
 * Video1.c
 *
 * Created: 2012-03-26 15:39:35
 *  Author: tmf
 */ 


#include <avr/io.h>
#include <avr/interrupt.h>
#include <stdbool.h>
#include <util/delay.h>
#include <string.h>
#include "graphics.h"

#include "Font8x8.h"                             //Tablica czionek 8x8

#define VIDEO_Scan_line    31.78                 //Czas trwania caej linii
#define VIDEO_Sync_Len      3.81                 //Czas synchronizacji
#define VIDEO_Back_porch    1.80 //1.907                //Czas od koca synchronizacji do wywietlania pikseli
#define VIDEO_Front_porch   0.63                 //Czas od koca wywietlania pikseli do impulsu synchronizacji

#define VIDEO_HLINES        524                  //Liczba linii poziomych (wszystkich)
#define VIDEO_VSYNC           2                  //Czas trwania VSYNC - 2 linie
#define VIDEO_VBack_porch    (40+30)                  //Odstp do wywietlania pierwszej linii obrazu
#define VIDEO_VFront_porch   (40+12)                  //Odstp pomidzy ostatni lini obrazu a VSync

uint8_t framebuf[VIDEO_LINES*VIDEO_COLUMNS];     //Bufor ekranu

bool OSC_wait_for_rdy(uint8_t clk)
{
	uint8_t czas=255;
	while ((!(OSC.STATUS & clk)) && (--czas)) // Czekaj na ustabilizowanie si generatora
	 _delay_ms(1);
	return czas;   //false jeli generator nie wystartowa, true jeli jest ok
}

bool RC32M_en()
{
	OSC.CTRL |= OSC_RC32MEN_bm; //Wcz generator RC 32 MHz
	return OSC_wait_for_rdy(OSC_RC32MEN_bm); //Zaczekaj na jego poprawny start
}

ISR(TCC0_CCB_vect)
{
	static uint8_t *bufaddr;             //Wskanik do bufora ramki
	
	uint16_t HLine=TCC1_CNT;             //Numer linii do wywietlenia
	uint8_t HLineL=HLine>>1;             //Dziki temu czasami moemy operowa tylko na 8 bitach
	if((HLineL >= ((VIDEO_VSYNC + VIDEO_VBack_porch) / 2)) && (HLineL < ((VIDEO_HLINES-VIDEO_VFront_porch) / 2)))
	{   //Aktywuj transfer DMA tylko dla aktywnych linii obrazu
		DMA_CH0_SRCADDR2=0;                          //Przepisz adres z rejestrw tymczasowych do SRCADDR
		
		asm volatile (                               //Poniszy kod likwiduje jitter zwizany z wejciem do funkcji obsugi przerwania (0-4 cykli)
                       "lds r16, %A0"   "\n\t"
	                   "sbrs r16, 1"    "\n\t"
	                   "lpm"            "\n\t"
	                   "sbrs r16, 0"    "\n\t"
	                   "rjmp .+0"       "\n\t"
					   "sbrc r16, 2"    "\n\t"
					   "rjmp L1%="      "\n\t"
	                   "lpm"            "\n\t"
					   "nop"            "\n\t"
					   "nop"            "\n\t"
					   "L1%=:"            "\n\t"
                       :: "m" (TCC0_CNT) : "r16");

        USARTC0_DATA=255; USARTC0_DATA=255;                //Wypenij wstpnie bufor - unikamy jitteru zwizanego ze startem DMA
		DMA_CH0_CTRLA=DMA_CH_ENABLE_bm | DMA_CH_SINGLE_bm | DMA_CH_BURSTLEN_1BYTE_gc;  //Odblokuj transfer kolejnej linii danych

		HLine=HLine-(VIDEO_VSYNC + VIDEO_VBack_porch) + 1; //Adres kolejnej linii po odjciu linii niewidocznych
		if((HLine & 1)==0)
		{
			bufaddr+=VIDEO_COLUMNS;                      //Adres kolejnego wiersza danych
			DMA_CH0_SRCADDR0=(uint16_t)bufaddr & 0xFF;   //Umie w rejestrach tymczasowych DMA adres pocztku kolejnego transferu
			DMA_CH0_SRCADDR1=(uint16_t)bufaddr >> 8;
		}
	} else
	{
		bufaddr=(uint8_t*)&framebuf;
		DMA_CH0_SRCADDR0=(uint16_t)bufaddr & 0xFF;   //Umie w rejestrach tymczasowych DMA adres pocztku kolejnego transferu
		DMA_CH0_SRCADDR1=(uint16_t)bufaddr >> 8;
	}		
}

void DotClockTimer_init()      //Timer okrelajcy czas trwania pikseli
{
	PORTC.DIRSET=1;                                //PC0 - wyjcie
	PORTC.PIN0CTRL=PORT_INVEN_bm;                  //z inwersj, co zapewnia odpowiedni ksztat impulsu synchronizacji H-Sync
	TCC0.PER=(VIDEO_Scan_line*F_CPU)/1000000UL;    //Czas trwania linii w taktach timera
	TCC0.CCA=(VIDEO_Sync_Len*F_CPU)/1000000UL;     //Czas trwania impulsu synchronizacji
	TCC0.CCB=((VIDEO_Sync_Len+VIDEO_Back_porch)*F_CPU)/1000000UL; //Czas do rozpoczcia wywietlania pikseli - aktywacji DMA
	TCC0.CTRLB=TC0_CCAEN_bm | TC_WGMODE_SS_gc;     //Single slope mode
	TCC0.INTCTRLB=TC_CCBINTLVL_HI_gc;              //Odblokuj przerwania pocztku aktywnego obszaru linii
	TCC0.CTRLA=TC_CLKSEL_DIV1_gc;                  //Timer taktowany z CLKPER
	EVSYS_CH0MUX=EVSYS_CHMUX_TCC0_OVF_gc;          //Kada linia pozioma generuje zdarzenie
}

void HLine_counter()          //Timer zliczajcy linie poziome
{
	PORTC_DIRSET|=1<<4;                            //PC4 jest wyjciem na ktrym bdzie V-Sync
	PORTC.PIN4CTRL=PORT_INVEN_bm;                  //Inwersja zapewni odpowiedni ksztat impulsu
	TCC1.PER=VIDEO_HLINES;                         //Cakowita liczba linii obrazu
	TCC1.CCA=VIDEO_VSYNC;                          //Impuls synchronizacji pionowej
	TCC1.CTRLB=TC1_CCAEN_bm | TC_WGMODE_SS_gc;     //Wcz porwnanie w kanale A
	TCC1.CTRLA=TC_CLKSEL_EVCH0_gc;                 //Zliczaj zdarzenia z evsys CH0 - czyli linie poziome z TCC0
}

void DMA_init()
{
	DMA.CTRL=DMA_ENABLE_bm | DMA_PRIMODE_CH0RR123_gc;  //Odblokuj kontroler DMA, kana 0 ma najwyszy priorytet, reszta round robin
	
	DMA.CH0.ADDRCTRL=DMA_CH_SRCRELOAD_BLOCK_gc | DMA_CH_SRCDIR_INC_gc | DMA_CH_DESTDIR_FIXED_gc | DMA_CH_DESTRELOAD_BURST_gc; //Zwiekszamy adres rda i przeznaczenia, reload adresu co blok
	DMA.CH0.TRFCNT=VIDEO_COLUMNS;                          //Blok ma dugo jednej linii obrazu
	DMA.CH0.DESTADDR0=(uint16_t)&USARTC0.DATA & 0xFF;      //Dane bd wysyane do USART
	DMA.CH0.DESTADDR1=(uint16_t)&USARTC0.DATA >> 8;
	DMA.CH0.DESTADDR2=0;    
	DMA.CH0.TRIGSRC=DMA_CH_TRIGSRC_USARTC0_DRE_gc;         //Pusty rejestr nadajnika inicjuje transfer DMA
	DMA.CH0.CTRLA=DMA_CH_ENABLE_bm | DMA_CH_SINGLE_bm | DMA_CH_BURSTLEN_1BYTE_gc;  //Kana 0 dugo transferu 1 bajt
}

void USART_SPI_init()
{
	PORTC.DIRSET=(1<<3);                            //MOSI (Txd) jest wyjciem
	PORTC.PIN3CTRL=PORT_INVEN_bm;                   //Musimy znaegowa sygna, aby port spoczynkowo mia stan 0
	USARTC0.CTRLB=USART_CLK2X_bm | USART_TXEN_bm;
	USARTC0.CTRLC=USART_CMODE_MSPI_gc;              //Tryb SPI, LSB first		
}

void INT_init()
{
	PMIC_CTRL|=PMIC_HILVLEN_bm;     //Odblokuj przerwania niskiego poziomu
	sei();
}

int main(void)
{
	RC32M_en();

	CPU_CCP=CCP_IOREG_gc;            //Odblokuj zmian konfiguracji
	CLK.CTRL=CLK_SCLKSEL_RC32M_gc;   //Wybierz generator RC 32MHz
	
	memset(framebuf, 0xFF, sizeof(framebuf));   //Wyczy ekran
	for(uint8_t y=0; y<VIDEO_LINES; y+=10) GLCD_Line(0, 0, 319, y);
	for(uint8_t y=0; y<(VIDEO_LINES / 2); y+=10) GLCD_Circle(160, 100, y);
	
	DotClockTimer_init();            //Zainicjuj timer zliczajcy piksele
	HLine_counter();                 //Zainicjuj timer zliczajcy linie
	USART_SPI_init();
	DMA_init();
	INT_init();
	
    while(1)
    {
        //TODO:: Please write your application code 
    }
}