/*
 * DCF77.c
 *
 * Created: 2012-03-06 19:21:08
 *  Author: tmf
 */ 

#include "DCF77.h"
#include <avr/io.h>
#include <string.h>
#include <avr/interrupt.h>
#include <util/atomic.h>
#include <calendar.h>

#define F_CPU 2000000UL    //Tylko dla debuggera

#define DCF77_HI_beg (140*F_CPU/1024000ul)
#define DCF77_HI_end (250*F_CPU/1024000ul)
#define DCF77_LO_beg (40*F_CPU/1024000ul)
#define DCF77_LO_end (130*F_CPU/1024000ul)
#define DCF77_Interval (1730*F_CPU/1024000ul)

volatile uint8_t DCF77_bitcounter;
volatile DCF77_t DCF77Record[2];
volatile uint32_t Timestamp;
volatile bool TimestampUpdate;

static volatile uint8_t DCF77RecNo;   //Numer rekordu DCF77_t do ktrego si odwoujemy

//Funkcja obsugi przerwania capture z kanau A - odczytywanie kolejnych bitw
// Pole DCF77Record[DCF77RecNo].begin przechowuje take wskanik poprawnoci rekordu 0 - ok, 1 - bdny
ISR(TCC1_CCA_vect)
{
	uint16_t PW=TCC1.CCA;

	if((PW >= DCF77_LO_beg) && (PW <= DCF77_LO_end)) DCF77Record[DCF77RecNo].Byte[DCF77_bitcounter / 8] &= ~(1<<DCF77_bitcounter % 8);  //Ustaw 0 na bicie odpowiadajcym aktualnie odbieranemu polu
     else if((PW < DCF77_HI_beg) || ( PW > DCF77_HI_end)) DCF77Record[DCF77RecNo].begin=1;   //Sprawd timingi dla bitu o wartoci 1

	DCF77_bitcounter++;
	if(DCF77_bitcounter == 59)
	{  //Caa struktura odebrana - sprawdzamy jej poprawno
		DCF77Record[DCF77RecNo].begin=!DCF77_Data_Valid((DCF77_t*)&DCF77Record[DCF77RecNo]);  //Rzutowanie typw jest konieczne, gdy porzucamy atrybut volatile
		if((DCF77Record[DCF77RecNo].begin==0) && (TimestampUpdate==true))
		{
			struct calendar_date cd={.second=58, .minute=BCDtoBIN(DCF77Record[DCF77RecNo].Minute), .hour=BCDtoBIN(DCF77Record[DCF77RecNo].Hour),
			                         .date=BCDtoBIN(DCF77Record[DCF77RecNo].Day)-1, .month=BCDtoBIN(DCF77Record[DCF77RecNo].Month)-1, .year=2000 + BCDtoBIN(DCF77Record[DCF77RecNo].Year)};
			//uint8_t hourtz=2;
			//if(DCF77Record[DCF77RecNo].TimeZone==2) hourtz=1;
		    //Timestamp=calendar_date_to_timestamp_tz(&cd,hourtz, 0);
			uint32_t tmp=calendar_date_to_timestamp(&cd);
			if(tmp)     //Nie uaktualniaj czasu jeli jest on bdny
			{
				Timestamp=tmp;
				TimestampUpdate=false;
			}				
		}
	}
	
	if(DCF77_bitcounter > 59)
	{
		DCF77Record[DCF77RecNo].begin=1; //Bd - za duo bitw
	}
}

//Funkcja obsugi przerwania nadmiaru - wykrywanie przerwy synchronizacyjnej
ISR(TCC1_OVF_vect)
{
	DCF77_bitcounter = 0;
	DCF77RecNo ^= 1;  //Zmieniamy bufor
	memset((void*)&DCF77Record[DCF77RecNo], 0xff, sizeof(DCF77_t));  //Zainicjuj struktur zawierajc dane DCF77
}

void DCF77_Rec_Init()
{
//Konfiguracja struktur danych
	memset((void*)&DCF77Record[DCF77RecNo], 0xff, sizeof(DCF77_t));  //Zainicjuj struktur zawierajc dane DCF77
	
//Konfiguracja portu IO
	PORTC.DIRSET=0b00000001;  //Pin PC0 jest wyjciem (steruje pinem P1 odbiornika)
	DCF77_Rec_En();           //Odblokuj odbiornik - pin P1
	PORTC.PIN1CTRL=PORT_OPC_TOTEM_gc | PORT_ISC_BOTHEDGES_gc;
	//PORTC.PIN1CTRL=PORT_INVEN_bm | PORT_OPC_TOTEM_gc | PORT_ISC_BOTHEDGES_gc;  //Pin w konfiguracji totempole, z inwersj, generujcy zdarzenia na obu zboczach, dla odbiornika ze stanem spoczynkowym wynoszcym 1

//Konfiguracja kanau 0 event system
	EVSYS_CH0MUX=EVSYS_CHMUX_PORTC_PIN1_gc;   //Zdarzenie zwizane z PC1

//Konfiguracja timera 1 dla portu C
	TCC1.CTRLB=TC1_CCAEN_bm;         //Odblokuj zdarzenia capture, tryb timera Normal
	TCC1.CTRLD=TC_EVACT_PW_gc | TC_EVSEL_CH0_gc;     //Pulse width capture, event channel 0
	TCC1.INTCTRLB=TC_CCAINTLVL_LO_gc;    //Odblokuj przerwania capture z poziomem niskim
	
	TCC1.INTCTRLA=TC_OVFINTLVL_LO_gc; //Odblokuj przerwanie nadmiaru licznika
	TCC1.PER=DCF77_Interval;      //Okres licznika rwny jest najduszej przerwie synchronizacyjnej

    TCC1.CTRLA=TC_CLKSEL_DIV1024_gc; //Preskaler 1024
	
//Konfiguracja podsystemu przerwa
	PMIC_CTRL|=PMIC_LOLVLEN_bm;     //Odblokuj przerwania niskiego poziomu
}

void DCF77_Rec_Dis()
{
	PORTC_OUTSET=0b00000001; //Zablokuj odbiornik (stan wysoki na pinie P1)
}

void DCF77_Rec_En()
{
	PORTC_OUTCLR=0b00000001; //Odblokuj odbiornik (stan niski na pinie P1)
}

bool GetDCF77Record(DCF77_t *dcf)
{
	bool ret=true;
	ATOMIC_BLOCK(ATOMIC_RESTORESTATE)
	{
		uint8_t tmp=DCF77RecNo^1;       //Bdziemy kopiowa ostatni aktualny rekord
		if(DCF77Record[tmp].begin==0)
		  memcpy(dcf, (void *)&DCF77Record[tmp], sizeof(DCF77_t));  //Kopiowanie danych musi przebiega w sposb atomowy
		   else ret=false;
	}
	return ret;
}

bool DCF77_Data_Valid(DCF77_t *data)
{
	if(data->begin != 0) return false;    //Ramka nie zaczyna si od bitu 0
	if(data->TimeSep != 1) return false;  //Ramka nie ma separatora pola czasu rwnego 1
	if(data->LastBit != 1) return false;  //Ostatni bit ramki musi by rwny 1
	
	return true;
}

uint8_t BCDtoBIN(uint8_t aBCD)
{
	return ((aBCD & 0xF0)>>4)*10 + (aBCD & 0x0F);
}