/*
 * EEPROMToken.c
 *
 * Created: 2012-09-16 15:59:37
 *  Author: tmf
 */ 

#include "EEPROMToken.h"
#include <avr/io.h>
#include <avr/eeprom.h>
#include <avr/pgmspace.h>

static void *EEPROMReg_GetFirstAvailPos(); //Pierwsza wolna komrka pamici EEPROM
static void *EEPROMReg_GetRegItem(const char *regname);  //Sprawd czy rekord o podanej nazwie istnieje
static uint8_t EEPROMReg_GetLastStrAddr(uint8_t **addr);
static void EEPROMReg_MoveEEPROMBlock(const void *src, void *dst, size_t len);  //Przenie blok pamici EEPROM z src do dst
static bool EEPROMReg_CreateRegEntry(const char *regname, uint8_t size, uint8_t **addr); //Dodaj rekord do koca danych
static void EEPROMReg_CmdExec(NVM_CMD_t cmd);  //Wykonaj polecenie kontrolera NVM

bool EEPROMReg_AddRegEntry(const char *regname, void *regvalue, uint8_t size)
{
	uint8_t *addr=EEPROMReg_GetRegItem(regname);
	if(addr==(void*)-1)
	    if(EEPROMReg_CreateRegEntry(regname, size, &addr)==false) //Nie ma podanego wpisu, wic tworzymy nowy
		   return false;  //Jeli bd to wracamy

	//Wpis znaleziony/stworzony wic "tylko" go uaktualniamy
	uint8_t recsize=*addr;	//Odczytaj dugo rekordu
	uint8_t actsize=EEPROMReg_GetLastStrAddr(&addr)+1;
	uint8_t howmuch, dat;

    if((actsize+size)!=recsize) 		//Czy zmienia si dugo rekordu?
	{
		uint8_t prevsize=recsize-actsize;   //Poprzednia dugo pola danych
		uint8_t *lastaddr=EEPROMReg_GetFirstAvailPos()-1;
		size_t ile=lastaddr-addr-prevsize+1; //Dugo przesuwanego bloku
		if(prevsize<size)
		{
			EEPROMReg_MoveEEPROMBlock(lastaddr, lastaddr+size-prevsize, ile);
		} else
		{
			EEPROMReg_MoveEEPROMBlock(addr+prevsize, addr+prevsize-size-1, ile);
			eeprom_write_byte(lastaddr+size-prevsize+1, 0xFF); //Zapisz znacznik koca listy
		}
		eeprom_write_byte(addr-actsize,actsize+size);  //Zapisz now dugo rekordu
	}
	eeprom_write_block(regvalue, addr, size);
	while(NVM.STATUS & NVM_NVMBUSY_bm); //Zaczekaj na koniec poprzednich operacji
	NVM_CTRLB|=NVM_EEMAPEN_bm;  //Wcz mapowanie EEPROM
	return true;
}

void *EEPROMReg_GetRegEntry(const char *regname, uint8_t *size)
{
	uint8_t *addr=EEPROMReg_GetRegItem(regname); //Znajd rekord
	if((size_t)addr==-1) return NULL;         //Nie ma rekordu
	*size=*addr - EEPROMReg_GetLastStrAddr(&addr)-1;     //Znajd pole danych
	return addr;
}

void EEPROMReg_DelRegEntry(const char *regname)
{
	uint8_t *addr=EEPROMReg_GetRegItem(regname);
	if(addr==(void*)-1) return;					//Nie znaleziono podanego wpisu
	uint8_t recsize=*addr;
	uint8_t *endofreg=(uint8_t*)((int)EEPROMReg_GetFirstAvailPos()-recsize);	//Pierwsza dostpna pozycja (koniec danych)
	EEPROMReg_MoveEEPROMBlock(addr + recsize, addr, endofreg-addr+1);       //Usu dziur po rekordzie
}

bool EEPROMReg_CreateRegEntry(const char *regname, uint8_t size, uint8_t **addr)
{
	*addr=EEPROMReg_GetFirstAvailPos()+1;
	if((*addr==(void*)-1) || ((int)(*addr+size)>=MAPPED_EEPROM_END)) return false;	//Bd - brak miejsca w pamici
	uint8_t index=0;
	uint8_t c;
	(*addr)-=MAPPED_EEPROM_START; //Ponisze funkcje uywaj adresowania w przestrzeni EEPROM
	do
	{
		c=pgm_read_byte_near(&regname[index]);  //Odczytaj token z FLASH
		eeprom_write_byte(*addr+index,c);        //Zapisz go w EEPROM
		index++;
	} while(c!=0);
	(*addr)--;
	eeprom_write_byte(*addr,size + index + 1);
	eeprom_write_byte(*addr + size + index +1, 0xFF);	//Znacznik koca listy
	while(NVM.STATUS & NVM_NVMBUSY_bm); //Zaczekaj na koniec poprzednich operacji
	NVM_CTRLB|=NVM_EEMAPEN_bm;  //Wcz mapowanie EEPROM
	(*addr)+=MAPPED_EEPROM_START;  //Dalej korzystamy z adresu w zmapowanej przestrzeni
	return true;
}

void *EEPROMReg_GetRegItem(const char *regname)
{
	uint8_t tmpEE;
	char *addr= (char*)MAPPED_EEPROM_START;  //Pocztek EEPROM
	do
	{
		tmpEE=*addr;
		if(tmpEE==0xFF) return (void*)-1;		//Ta komrka jest wolna, a rekordu nie znaleziono
		if(strcmp_P(addr+1, regname)==0) return addr; //Rekord znaleziony
		addr+=tmpEE;                    //Przejd do kolejnego rekordu
	}while((int)addr<MAPPED_EEPROM_END);            //Koniec pamici EEPROM
	return (void*)-1;
}

void *EEPROMReg_GetFirstAvailPos()
{
	
	uint8_t *addr=(uint8_t*)MAPPED_EEPROM_START;  //Pocztek EEPROM
	uint8_t EEChar;
	do
	{
		EEChar=*addr;
		if(EEChar!=255) addr+=EEChar;
	} while(((int)addr<MAPPED_EEPROM_END) && (EEChar!=255));
	if(EEChar!=255) addr= (void*)-1;	//Brak miejsca w pamici
	return addr;		                //Pierwsza wolna komrka pamici EEPROM
}

uint8_t EEPROMReg_GetLastStrAddr(uint8_t **addr)
{
	uint8_t EEChar;
	uint8_t len=0;
	(*addr)++;
	do
	{
		EEChar=*((*addr)++); len++;

	} while((EEChar!=0) && ((int)*addr<MAPPED_EEPROM_END));
	return len;
}

void EEPROMReg_MoveEEPROMBlock(const void *src, void *dst, size_t len)
{
	void EEPROMReg_MovePage_inc(const void **src, void **dst, uint8_t len)
	{
		while(len-->0) *(uint8_t*)(*dst)++=*(uint8_t*)(*src)++;
	}
	
	void EEPROMReg_MovePage_dec(const void **src, void **dst, uint8_t len)
	{
		while(len-->0) *(uint8_t*)(*dst)--=*(uint8_t*)(*src)--;
	}

	void (*MovePage)(const void **, void **, uint8_t);
	MovePage=EEPROMReg_MovePage_inc;
	if(src<dst) MovePage=EEPROMReg_MovePage_dec; //Zapewnij prawidowy przesy blokw
	
	EEPROMReg_CmdExec(NVM_CMD_ERASE_EEPROM_BUFFER_gc); //Skasuj bufor EEPROM - ewentualne wczeniejsze zapisy/odczyty bd bez znaczenia
	while(len)
	{
		int8_t tmplen=((size_t)dst % EEPROM_PAGE_SIZE)+1; //Ile bajtw moemy przesa
		if(src>dst) tmplen=EEPROM_PAGE_SIZE - tmplen;
		if(tmplen>len) tmplen=len;  //Jeli moemy wicej ni chcemy
		NVM.ADDR0=(size_t)dst;
		NVM.ADDR1=(size_t)dst >> 8;
		len-=tmplen;
		MovePage(&src, &dst, tmplen);
		EEPROMReg_CmdExec(NVM_CMD_ERASE_WRITE_EEPROM_PAGE_gc); //Zapisz stron
	}
}

void EEPROMReg_CmdExec(NVM_CMD_t cmd)
{
	NVM.CMD=cmd;                //Polecenie kontrolera
	CPU_CCP=CCP_IOREG_gc;       //Zezwl na jego wykonanie wykonanie
	NVM.CTRLA=NVM_CMDEX_bm;     //Wykonaj polecenie
	while(NVM.STATUS & NVM_NVMBUSY_bm); //Zaczekaj a si zakoczy
}