/*
 * OpenBench.c
 *
 * Created: 2013-01-10 19:40:15
 *  Author: tmf
 */ 


#include <avr/io.h>
#include <stdbool.h>
#include <avr/pgmspace.h>
#include <util/delay.h>
#include <avr/interrupt.h>
#include <string.h>

#include "usart.h"

#define OLS_SAMPLE_BUF 6144 //Wielko bufora prbek - 6 kB
#define OLS_MAX_SRATE 8000000UL  //Maksymalna szybko prbkowania
#define OLS_Ref_CLK 100000000UL  //Zegar referencyjny 100 MHz

enum OLS_CMDs {OLS_Reset=0x00, OLS_Run=0x01, OLS_ID=0x02, OLS_XON=0x11, OLS_XOFF=0x13, OLS_MetaData=0x04,
	           OLS_Set_Trg_Mask0=0xC0, OLS_Set_Trg_Mask1=0xC4, OLS_Set_Trg_Mask2=0xC8, OLS_Set_Trg_Mask3=0xCC, 
	           OLS_Set_Trg_Value0=0xC1, OLS_Set_Trg_Value1=0xC5, OLS_Set_Trg_Value2=0xC9, OLS_Set_Trg_Value3=0xCD,
			   OLS_Set_Trg_Cfg0=0xC2, OLS_Set_Trg_Cfg1=0xC6, OLS_Set_Trg_Cfg2=0xCA, OLS_Set_Trg_Cfg3=0xCE, 
	           OLS_SetDivider=0x80, OLS_Set_ReadDelay_Count=0x81};

enum OLS_States {OLS_St_Waiting, OLS_St_Rec5byteCmd, OLS_St_Cmd_Received, OLS_Transmitting}; 

const uint8_t MetaData[] PROGMEM = {0x01, 'X','M','E','G','A','T','M','F',0,  //Nazwa sterownika
	                                0x02, '1','.','0',0,                      //Wersja firmware 1.0
	                                0x21, ((uint32_t)OLS_SAMPLE_BUF)>>24, (uint8_t)(((uint32_t)OLS_SAMPLE_BUF)>>16), (uint8_t)(OLS_SAMPLE_BUF>>8), (uint8_t)OLS_SAMPLE_BUF, //Pami prbek - 4096 bajtw - big-endian
                                    0x23, ((uint32_t)OLS_MAX_SRATE)>>24, (uint8_t)(OLS_MAX_SRATE>>16), (uint8_t)(OLS_MAX_SRATE>>8), (uint8_t)OLS_MAX_SRATE,     //Szybko samplowania 1 MHz - big-endian
									0x25, 0x00, 0x00, 0x00, 0x01,             //Opis moduu - dysponujemy tylko podstawowymi wyzwalaczami
									0x40, 0x08,                               //Liczba kanaw - 8
									0x41, 0x02,                               //Wersja protokou 2
									0x00};									  //Koniec metadanych

volatile enum OLS_States State;
uint8_t Sample_buf[OLS_SAMPLE_BUF];  //Bufor na prbki

typedef union
{
	uint32_t value;
	struct  
	{
		uint8_t byte0;
		uint8_t byte1;
		uint8_t byte2;
		uint8_t byte3;
	};
} probe;

struct 
{
	uint32_t SampleRate;      //Czstotliwo prbkowania
	uint16_t ReadCount;       //Liczba prbek
	uint16_t DelayCount;      //Opnienie prbkowania
	probe TriggerMask[4];     //Maska dla triggera
	probe TriggerValue[4];    //Warto bitw triggera
	probe TriggerCfg[4];      //Konfiguracja triggera
} OLS_Data;

ISR(PORTF_INT0_vect) //Przerwanie bdce triggerem
{
	PORTF_INT0MASK=0;              //Wyczamy dalsze przerwania
	PORTF_INTFLAGS=PORT_INT0IF_bm; //Usuwamy eventualne inne przerwania portu, ktre w midzyczasie zostay zarejestrowane
	TCF1.CNT=0;
	TCF1.CCA=OLS_Data.DelayCount - 1;  //Zlicz odpowiedni liczb prbek
	TCF1.INTFLAGS=TC0_CCAIF_bm;         //Skasuj flag CCA
	TCF1.INTCTRLB=TC_CCAINTLVL_MED_gc;  //Wcz przerwanie porwania - koca samplowania
    
	//SLEEP();
}

ISR(TCF1_CCA_vect) //Koczymy prbkowanie i wysyamy wyniki
{
	TCF0_CTRLA=TC_CLKSEL_OFF_gc;      //Wyczamy prbkowanie
	TCF1_INTCTRLB&=~TC1_CCAINTLVL_gm; //Wycz przerwania CCA
    for(uint16_t indeks=0; indeks<OLS_Data.ReadCount; indeks++)
     {
	   int16_t addr=DMA.CH0.DESTADDR0 + (DMA.CH0.DESTADDR1<<8) - (uint16_t)&Sample_buf;
	   addr-=indeks;
	   if(addr<0) addr+=OLS_SAMPLE_BUF;
	   USART_putchar(&USARTC0, Sample_buf[addr]);  //Wylij dane
     }
    TCF0_CTRLA=TC_CLKSEL_DIV1_gc; //Ponownie wcz prbkowanie
	State=OLS_St_Waiting;        //Czekamy na kolejne polecenia
}

ISR(USARTC0_RXC_vect)
{
	static uint8_t cmd[5];    //Odebrane polecenie
	static uint8_t index;     //Indeks do bajtu polecenia
	uint8_t rec=USARTC0_DATA;
	
	switch(State)
	{
		case OLS_St_Waiting: 
		                      index=1;
							  cmd[0]=rec;
							  if(rec>OLS_XOFF) State=OLS_St_Rec5byteCmd; //Komendy o kodach >0x11 s 5-bajtowe
							    else State=OLS_St_Cmd_Received;          //Cae polecenie odebrane
							  break;
		case OLS_St_Rec5byteCmd:
		                      if(index<5) cmd[index++]=rec;            //Kolejna cz polecenia
							  if(index==5) State=OLS_St_Cmd_Received;  //Cae polecenie odebrane
							  break;
		default: ;       //W pozostaych stanach nic nie robimy
	}
	
	if(State==OLS_St_Cmd_Received)
	{
		switch (cmd[0])
		{
			case OLS_Reset:
			          PORTF_INT0MASK=0;                 //Wyczamy przerwania od pinw IO
	                  TCF1_INTCTRLB&=~TC1_CCAINTLVL_gm; //Wycz przerwania CCA
			          State=OLS_St_Waiting;
			          break;
			case OLS_ID:  
		              USART_send_F(&USARTC0, PSTR("1ALS"));  //Wersja protokou
		              State=OLS_St_Waiting;
				      break; 

		    case OLS_MetaData:
		              USART_send_block_F(&USARTC0, MetaData, sizeof(MetaData));
					  State=OLS_St_Waiting;
		              break; 
		
		    case OLS_Run:                       //Uzbrj trigger
					  {
					   uint8_t mask=1;
					   for(uint8_t probe=0; probe<8; probe++)  //Skonfiguruj odpowiednio triggery na poszczeglnych pinach
					    {
						  uint8_t cfg=(OLS_Data.TriggerValue->byte0 & mask)? PORT_ISC_RISING_gc : PORT_ISC_FALLING_gc;
						  ((PORT_t*)((uint16_t)&PORTF + probe))->PIN0CTRL=cfg;
						  mask<<=1;
					    }
					  }					  
					  PORTF.INT0MASK=OLS_Data.TriggerMask[0].byte0;  //Zaaduj mask triggerw dla sond 0-7
					  if(OLS_Data.TriggerMask[0].byte0==0) PORTF_INT0_vect(); //Jeli nie ustawiono triggerw to po prostu prbkujemy
					  break;
					  
		     case OLS_SetDivider:
			          OLS_Data.SampleRate=cmd[3];
					  OLS_Data.SampleRate<<=8; OLS_Data.SampleRate|=cmd[2];
					  OLS_Data.SampleRate<<=8; OLS_Data.SampleRate|=cmd[1];
					  OLS_Data.SampleRate=OLS_Ref_CLK/(OLS_Data.SampleRate+1); //Szybko samplowania to CLK/(skaler+1)
					  if(OLS_Data.SampleRate>OLS_MAX_SRATE) OLS_Data.SampleRate=OLS_MAX_SRATE; //Nie powinnimy przekracza maksymalnej czstotliwoci prbowania
					  TCF0.PER=(F_CPU/OLS_Data.SampleRate) - 1;     //Zmie samplowanie w timerze
					  State=OLS_St_Waiting;
		              break;
			 
		     case OLS_Set_ReadDelay_Count:
					  OLS_Data.ReadCount=(cmd[2] << 8) + cmd[1] + 1; 
					  OLS_Data.ReadCount<<=2;    //Mnoymy razy 4
					  OLS_Data.DelayCount=(cmd[4] << 8) + cmd[3];
					  OLS_Data.DelayCount<<=2;
		              State=OLS_St_Waiting;
		              break;
					  
			  case OLS_Set_Trg_Mask0:
			  case OLS_Set_Trg_Mask1:
			  case OLS_Set_Trg_Mask2:
			  case OLS_Set_Trg_Mask3:
			          {
						  uint8_t trgno=(cmd[0]>>2) & 0b11; //Numer triggera na podstawie nr polecenia
						  memcpy(&OLS_Data.TriggerMask[trgno], &cmd[1], sizeof(OLS_Data.TriggerMask[0])); //Skopiuj mask triggera
					  }
					  State=OLS_St_Waiting;			  
					  break;
			  
			  case OLS_Set_Trg_Value0:
			  case OLS_Set_Trg_Value1:
			  case OLS_Set_Trg_Value2:
			  case OLS_Set_Trg_Value3:
			          {
				          uint8_t trgno=(cmd[0]>>2) & 0b11; //Numer triggera na podstawie nr polecenia
				          memcpy(&OLS_Data.TriggerValue[trgno], &cmd[1], sizeof(OLS_Data.TriggerValue[0])); //Skopiuj warto triggera
			          }
					  State=OLS_St_Waiting;		          
			          break;

			  case OLS_Set_Trg_Cfg0:
			  case OLS_Set_Trg_Cfg1:
			  case OLS_Set_Trg_Cfg2:
			  case OLS_Set_Trg_Cfg3:
			          {
				          uint8_t trgno=(cmd[0]>>2) & 0b11; //Numer triggera na podstawie nr polecenia
				          memcpy(&OLS_Data.TriggerCfg[trgno], &cmd[1], sizeof(OLS_Data.TriggerCfg[0])); //Skopiuj konfiguracj triggera
			          }
					  State=OLS_St_Waiting;
			          break;
			  
			 default: State=OLS_St_Waiting;
	    }
	}		
}

void USART_init()
{
	USARTC0.CTRLB=USART_TXEN_bm | USART_RXEN_bm;   //Wcz nadajnik i odbiornik USART
	USARTC0.CTRLC=USART_CHSIZE_8BIT_gc;  //Ramka 8 bitw, bez parzystoci, 1 bit stopu
	usart_set_baudrate(&USARTC0, 115200, F_CPU); 

    PORTC_OUTSET=PIN3_bm;
	PORTC_DIRSET=PIN3_bm;  //Pin TxD musi by wyjciem w stanie wysokim
	
	USARTC0.CTRLA=USART_RXCINTLVL_LO_gc; //Wcz przerwanie odbiornika USART
	PMIC_CTRL|=PMIC_LOLVLEN_bm;          //Wcz przerwania niskiego poziomu
}

bool OSC_wait_for_rdy(uint8_t clk)
{
	uint8_t czas=255;
	while ((!(OSC.STATUS & clk)) && (--czas)) // Czekaj na ustabilizowanie si generatora
	 _delay_ms(1);
	return czas;   //false jeli generator nie wystartowa, true jeli jest ok
}

bool RC32M_en()
{
	OSC.CTRL |= OSC_RC32MEN_bm; //Wcz generator RC 32 MHz
	return OSC_wait_for_rdy(OSC_RC32MEN_bm); //Zaczekaj na jego poprawny start
}

void Sample_init()
{
	//Zainicjuj kontroler DMA
	DMA.CTRL=DMA_ENABLE_bm | DMA_PRIMODE_CH0123_gc; //Wcz DMA i priorytet kanaw 0-1-2-3
	DMA.CH0.ADDRCTRL=DMA_CH_SRCRELOAD_NONE_gc | DMA_CH_DESTRELOAD_TRANSACTION_gc | DMA_CH_SRCDIR_FIXED_gc | DMA_CH_DESTDIR_INC_gc; //Wzikszaj wycznie adres docelowy
	DMA.CH0.TRIGSRC=DMA_CH_TRIGSRC_EVSYS_CH0_gc; //DMA_CH_TRIGSRC_TCF0_OVF_gc;  //Transfer wyzwala zdarzenie przepenienia timera
	DMA.CH0.TRFCNT=OLS_SAMPLE_BUF;   //Liczba transferowanych bajtw
	DMA.CH0.REPCNT=0;                //Nieskoczona liczba powtrze
	DMA.CH0.SRCADDR0=(uint8_t)((uint16_t)&PORTF_IN);
	DMA.CH0.SRCADDR1=(uint8_t)(((uint16_t)&PORTF_IN) >> 8);
	DMA.CH0.SRCADDR2=0;   //Wpisz adres rdowy - PORTF - rdo danych
	DMA.CH0.DESTADDR0=(uint8_t)((uint16_t)&Sample_buf);
	DMA.CH0.DESTADDR1=(uint8_t)(((uint16_t)&Sample_buf) >> 8);
	DMA.CH0.DESTADDR2=0;  //Wpisz adres docelowy - bufor prbek
	DMA.CH0.CTRLA=DMA_CH_ENABLE_bm | DMA_CH_SINGLE_bm | DMA_CH_BURSTLEN_1BYTE_gc | DMA_CH_REPEAT_bm; //Wcz DMA
	
    //Zainicjuj timer sterujcy prbkowaniem
	TCF0.CTRLB=TC_WGMODE_NORMAL_gc; //Tryb normalny
	TCF0.PER=4;                     //Samplowanie CLKper/4
	EVSYS_CH0MUX=EVSYS_CHMUX_TCF0_OVF_gc; //Nadmiar wywoa zdarzenie w EVCH0
	//Zainicjuj timer zliczajcy prbki
	TCF1.CTRLB=TC_WGMODE_NORMAL_gc;
	TCF1.PER=OLS_SAMPLE_BUF - 1;    //Zliczamy do pojemnoci bufora
	TCF1.CTRLA=TC_CLKSEL_EVCH0_gc;  //Impulsy taktujce pochodz z EVCH0
	
	TCF0.CTRLA=TC_CLKSEL_DIV1_gc;   //Timer taktowany bezporednio z CLKper, wczenie samplowania TCF0 i TCF1 i DMA
	
	PORTF.INTCTRL=PORT_INT0LVL_MED_gc; //Wcz przerwania, ktre wyzwol prbkowanie - na razie INT0MASK jest rwny 0 - brak przerwa
	PMIC_CTRL=PMIC_MEDLVLEN_bm;        //Zezwl na przerwania
}

int main(void)
{
	RC32M_en();
	CPU_CCP=CCP_IOREG_gc;            //Odblokuj zmian konfiguracji
	CLK.CTRL=CLK_SCLKSEL_RC32M_gc;   //Wybierz generator RC 32MHz
	
	Sample_init();
	USART_init();
	sei();

    while(1)
    {
        //TODO:: Please write your application code 
    }
}