using System;
using System.Data;

using TwoLayer.DataAccess;

namespace ThreeLayer.Business
{
   /// <summary>
   /// Obiekt biznesowy na informacje o wydawcy
   /// </summary>
   public class PublisherBO: AbstractBO 
   {
      // dane skadowe
      private int _id = DEFAULT_ID;
      private string _name = "";

      // konstruktory
      public PublisherBO() { }
      public PublisherBO(int id, string name)
      {
         _id = id;
         _name = name;
         IsNew = false;
         IsModified = true;
      }

      public static PublisherBO CreatePublisher(int pubId)
      {
         PublisherBO pub = new PublisherBO();
         if (pub.Load(pubId))
            return pub;
         else
            return null;
      }

      public static void UpdatePublisher(PublisherBO pub)
      {
         pub.Update();
      }

      /// <summary>
      /// Zapenia obiekt danymi na podstawie przekazanego identyfikatora
      /// </summary>
      /// <returns>
      /// true w przypadku pomlnego zapenienia danymi, w przeciwnym razie false
      /// </returns>
      public bool Load(int id)
      {
         PublisherDA da = new PublisherDA();
         DataTable table = da.GetById(id);
         // jeli w tabeli nie ma danych, zapenienie nie powidoo si
         if (table.Rows.Count == 0)
         {
            AddBrokenRule("Wydawca o numerze id=" + id + " nie zopsta odnaleziony");
            return false;
         }
         // ustaw dane skadowe zgodnie z pobranymi danymi
         Id = (int)table.Rows[0]["PublisherId"];
         Name = (string)table.Rows[0]["PublisherName"];
         IsNew = false;
         // upewnij si, e zaadowane dane s zgodne z reguami biznesowymi
         return IsValid;
      }

      /// <summary>
      /// Zapisuje dane obiektu. Jeli obiekt jest nowy, wstaw dane,
      /// w przeciwnym razie aktualizuj dane, jeeli ulegy zmianie
      /// </summary>
      public void Save()
      {
         if (IsNew)
            Insert();
         else 
            Update();
      }

      /// <summary>
      /// Tylko aktualizuje dane
      /// </summary>
      public void Update()
      {
         if (IsValid)
         {
            PublisherDA da = new PublisherDA();
            if (IsModified)
            {
               da.UpdatePublisher(Id, Name);
               IsModified = false;
            }
         }
      }

      /// <summary>
      /// Tylko wstawia dane
      /// </summary>
      public void Insert()
      {
         IsNew = true;
         if (IsValid)
         {
            
            PublisherDA da = new PublisherDA();
            da.InsertPublisher(Name);
         }
      }

      /// <summary>
      /// Usuwa dane
      /// </summary>
      public void Delete()
      {
         PublisherDA da = new PublisherDA();
         da.DeletePublisher(Id);
      }

      /// <summary>
      /// Sprawdza, czy stan wewntrzny obiektu biznesowego jest prawidowy
      /// </summary>
      protected override bool CheckIfSubClassStateIsValid
      {
         get
         {
            bool valid = true;
            if (Name.Length == 0)
            {
               AddBrokenRule("Nazwa wydawcy nie moe byc pusta");
               valid = false;
            }
            if (Id < 0)
            {
               AddBrokenRule("Id wydawcy nie moe by mniejsze ni zero");
               valid = false;
            }
            if (IsNew)
            {
               // sprawd, czy wydawca ju nie istnieje
               PublisherDA da = new PublisherDA();
               DataTable dt = da.GetByName(Name);
               if (dt.Rows.Count > 0)
               {
                  AddBrokenRule("Nazwa wydawcy ju istnieje");
                  valid = false;
               }
            }
            return valid;
         }
      }

      // waciwoci

      public int Id
      {
         get { return _id; }
         set { 
            _id = value;
            IsModified = true;
         }
      }
      public string Name
      {
         get { return _name; }
         set { 
            _name = value;
            IsModified = true;
         }
      }
      public PublisherBO[] WrapAsArray
      {
         get
         {
            PublisherBO[] pubArray = {
               this
            };
            return pubArray;
         }
      }
   }
}
