using System;
using System.Data;
using System.Configuration;
using System.Data.Common;
using System.Web.Configuration;

namespace TwoLayer.DataAccess
{
   /// <summary>
   /// Hermetyzuje wikszo funkcji rodowiska ADO.NEt potrzebnych
   /// przez klasy dostpu do danych
   /// </summary>
   public abstract class AbstractDA
   {
      /// <summary>
      /// Definiuje podstawow instrukcj pobierania danych
      /// bez kryteriw.
      /// 
      /// Ta waciwo jest zaimplementowana przez konkretne podklasy.
      /// </summary>
      protected abstract string SelectStatement
      {
         get;
      }

      /// <summary>
      /// Zwraca tabel z danymi zawierajc informacje o tabeli wydawcy dla okrelonego identyfikatora.
      /// Ten zbir danych bdzie zawiera 0 lub 1 wiersz danych.
      /// </summary>
      public abstract DataTable GetById(object oId);

      /// <summary>
      /// Zwraca wszystkie dane dla rda danych
      /// </summary>
      public DataTable GetAll()
      {
         string sql = SelectStatement;
         return GetDataTable(sql, null);
      }

      /// <summary>
      /// Utwrz niezaley od dostawcy obiekt DbParameter
      /// </summary>
      protected DbParameter MakeParameter(string name, object value, DbType paramType)
      {
         DbParameter param = Factory.CreateParameter();
         param.ParameterName = name;
         param.Value = value;
         param.DbType = paramType;
         return param;
      }

      /// <summary>
      /// Zwraca obiekt DataTable wypeniony danymi okrelonymi przez sql i parametry
      /// </summary>
      protected DataTable GetDataTable(string sql, DbParameter[] parameters)
      {
         return GetDataTable(sql, CommandType.Text, parameters);
      }

      /// <summary>
      /// Zwraca obiekt DataTable wypeniony danymi okrelonymi przez nazw i parametry
      /// </summary>
      protected DataTable GetDataTable(string name, CommandType cmdType, DbParameter[] parameters)
      {
         DataTable dt = null;
         try
         {
            // utwrz niezalene od dostawcy poczenie
            DbConnection conn = Factory.CreateConnection();
            conn.ConnectionString = ConnectionSetting.ConnectionString;

            // utwrz niezaleny od dostawcy adapter danych
            DbDataAdapter adapter = Factory.CreateDataAdapter();

            DbCommand cmd = Factory.CreateCommand();
            cmd.Connection = conn;
            cmd.CommandText = name;
            cmd.CommandType = cmdType;
            adapter.SelectCommand = cmd; ;

            // dodaj parametry
            if (parameters != null)
            {
               foreach (DbParameter p in parameters)
                  adapter.SelectCommand.Parameters.Add(p);
            }

            // utwrz tabel danych
            dt = new DataTable();

            // wypenij now tabel danymi pochodzcymi z adaptera
            adapter.Fill(dt);

         }
         catch (Exception ex)
         {
            // wszystkie bdy bd obsugiwane przez niestandardow procedur obsugi wyjtkw
            DataAccessExceptionHandler.HandleException(ex.Message);
         }

         // Zwr obiekt DataTable 
         return dt;
      }

      /// <summary>
      /// Uruchamia zapytanie SQL, ktre nie zwraca danych
      /// </summary>
      protected void RunNonQuery(string sql, DbParameter[] parameters)
      {
         // utwrz niezalene od dostawcy poczenie
         using (DbConnection conn = Factory.CreateConnection())
         {
            conn.ConnectionString = ConnectionSetting.ConnectionString;

            // utwrz niezalen od dostawcy instrukcj
            DbCommand cmd = Factory.CreateCommand();
            cmd.Connection = conn;
            cmd.CommandText = sql;

            // dodaj parametry
            if (parameters != null)
            {
               foreach (DbParameter p in parameters)
                  cmd.Parameters.Add(p);
            }

            try
            {
               conn.Open();
               cmd.ExecuteNonQuery();
            }
            catch (Exception ex)
            {
               // wszystkie bdy bd obsugiwane przez niestandardow procedur obsugi wyjtkw
               DataAccessExceptionHandler.HandleException(ex.Message);
            }
         }
      }

      /// <summary>
      /// Uruchamia skalar zwracajcy zapytanie SQL
      /// </summary>
      protected object RunScalar(string sql, DbParameter[] parameters)
      {
         object scalar = null;

         // utwrz niezalene od dostawcy poczenie
         using (DbConnection conn = Factory.CreateConnection())
         {
            conn.ConnectionString = ConnectionSetting.ConnectionString;

            // utwrz niezalen od dostawcy instrukcj
            DbCommand cmd = Factory.CreateCommand();
            cmd.Connection = conn;
            cmd.CommandText = sql;

            // dodaj parametry
            if (parameters != null)
            {
               foreach (DbParameter p in parameters)
                  cmd.Parameters.Add(p);
            }

            try
            {
               conn.Open();
               scalar = cmd.ExecuteScalar();
            }
            catch (Exception ex)
            {
               // wszystkie bdy bd obsugiwane przez niestandardow procedur obsugi wyjtkw
               DataAccessExceptionHandler.HandleException(ex.Message);
            }
            return scalar;
         }
      }

      /// <summary>
      /// Waciwo prywatna, ktra zwraca ustawienia acucha poczenia z pliku web.config
      /// </summary>
      private ConnectionStringSettings ConnectionSetting
      {
         get
         {
            return WebConfigurationManager.ConnectionStrings["BookCatalog"];

         }
      }

      /// <summary>
      /// Waciwo prywatna uruchamiajca odpowiedni DbProviderFactory
      /// </summary>
      private DbProviderFactory Factory
      {
         get
         {
            string connString = ConnectionSetting.ConnectionString;
            string invariantName = ConnectionSetting.ProviderName;

            // sprawd, czy nazwa tego dostawcy jest obsugiwana
            DataTable providers = DbProviderFactories.GetFactoryClasses();
            DataRow[] foundArray = providers.Select("InvariantName='" + invariantName + "'");
            if (foundArray.Length == 0)
            {
               // wszystkie bdy bd obsugiwane przez niestandardow procedur obsugi wyjtkw
               string msg = "[AbstractDA] Data Provider " + invariantName + " not found";
               DataAccessExceptionHandler.HandleException(msg);
            }

            // pobierz odbowiedni fabryk dostawcy
            return DbProviderFactories.GetFactory(invariantName);
         }
      }
   }
}