using System;
using System.Data;
using System.Configuration;
using System.Data.Common;
using System.Web.Configuration;


/// <summary>
/// Hermetyzuje wikszo funkcji ADO.NEt potrzebnych
/// przez klasy dostpu do danych
/// </summary>
public abstract class AbstractDA
{

   /// <summary>
   /// Zwraca wszystkie dane dla okrelonego rda danych
   /// </summary>
   public DataTable GetAll()
   {
      string sql = SelectStatement;
      return GetDataTable(sql, null);
   }

   /// <summary>
   /// Definiuje podstawow instrukcj wybierajc, ktra suy do pobierania danych
   /// bez adnych kryteriw.
   /// 
   /// Ta waciwoc jest implementowana przez konkretne podklasy.
   /// </summary>
   protected abstract string SelectStatement
   {
      get;
   }

   /// <summary>
   /// Tworzy obiekt klasy DbParameter niezaleny od dostawcy
   /// </summary>
   protected DbParameter MakeParameter(string name, object value, DbType paramType)
   {
      DbParameter param = Factory.CreateParameter();
      param.ParameterName = name;
      param.Value = value;
      param.DbType = paramType;
      return param;
   }

   /// <summary>
   /// Zwraca obiekt klasy DataTable wypeniony danymi okrelonymi przez sql i parametry
   /// </summary>
   protected DataTable GetDataTable(string sql, DbParameter[] parameters)
   {
      return GetDataTable(sql, CommandType.Text, parameters);
   }

   /// <summary>
   /// Zwraca obiekt klasy DataTable wypeniony danymi okrelonymi przez nazw i parametry
   /// </summary>
   protected DataTable GetDataTable(string name, CommandType cmdType, DbParameter[] parameters)
   {
      DataTable dt = null;
      try
      {
         // utwrz poczenie niezalene od dostawcy
         DbConnection conn = Factory.CreateConnection();
         conn.ConnectionString = ConnectionSetting.ConnectionString;

         // utwrz dapter danych niezaleny od dostawcy
         DbDataAdapter adapter = Factory.CreateDataAdapter();

         DbCommand cmd = Factory.CreateCommand();
         cmd.Connection = conn;
         cmd.CommandText = name;
         cmd.CommandType = cmdType;
         adapter.SelectCommand = cmd; ;

         // dodaj parametry
         if (parameters != null)
         {
            foreach (DbParameter p in parameters)
               adapter.SelectCommand.Parameters.Add(p);
         }

         // utwrz now tabel z danymi
         dt = new DataTable();

         // wypenij now tabel danymi pochodzcymi z adaptera
         adapter.Fill(dt);

      }
      catch (Exception ex)
      {
         // kady bd bdzie obsugiwany przez niestandardow funkcj obsugi wyjtkw
         DataAccessExceptionHandler.HandleException(ex.Message);
      }

      // Zwr obiekt klasy DataTable 
      return dt;
   }

   /// <summary>
   /// Uruchamia zapytanie SQL, ktre nie zwraca danych
   /// </summary>
   protected void RunNonQuery(string sql, DbParameter[] parameters)
   {
      // utwrz poczenie niezalene od dostawcy
      using (DbConnection conn = Factory.CreateConnection())
      {
         conn.ConnectionString = ConnectionSetting.ConnectionString;

         // utwrz polecenie niezalene od dostawcy
         DbCommand cmd = Factory.CreateCommand();
         cmd.Connection = conn;
         cmd.CommandText = sql;

         // dodaj parametry
         if (parameters != null)
         {
            foreach (DbParameter p in parameters)
               cmd.Parameters.Add(p);
         }

         try
         {
            conn.Open();
            cmd.ExecuteNonQuery();
         }
         catch (Exception ex)
         {
            // kady bd bdzie obsugiwany przez niestandardow funkcj obsugi wyjtkw
            DataAccessExceptionHandler.HandleException(ex.Message);
         }
      }
   }

   /// <summary>
   /// Uruchamia skalar zwracajcy zapytanie SQL
   /// </summary>
   protected object RunScalar(string sql, DbParameter[] parameters)
   {
      object scalar = null;

      // utwrz poczenie niezalene od dostawcy
      using (DbConnection conn = Factory.CreateConnection())
      {
         conn.ConnectionString = ConnectionSetting.ConnectionString;

         // utwrz polecenie niezalene od dostawcy
         DbCommand cmd = Factory.CreateCommand();
         cmd.Connection = conn;
         cmd.CommandText = sql;

         // dodaj parametry
         if (parameters != null)
         {
            foreach (DbParameter p in parameters)
               cmd.Parameters.Add(p);
         }

         try
         {
            conn.Open();
            scalar = cmd.ExecuteScalar();
         }
         catch (Exception ex)
         {
            // kady bd bdzie obsugiwany przez niestandardow funkcj obsugi wyjtkw
            DataAccessExceptionHandler.HandleException(ex.Message);
         }
         return scalar;
      }
   }

   /// <summary>
   /// Waciwoc prywatna, ktra zwraca konfiguracj acucha poczenia z pliku web.config
   /// </summary>
   private ConnectionStringSettings ConnectionSetting
   {
      get
      {
         return WebConfigurationManager.ConnectionStrings["BookCatalog"];
         
      }
   }

   /// <summary>
   /// Waciwoc prywatna, ktra zwraca odpowiedni obiekt klasy DbProviderFactory
   /// </summary>
   private DbProviderFactory Factory
   {
      get
      {
         string connString = ConnectionSetting.ConnectionString;
         string invariantName = ConnectionSetting.ProviderName;

         // sprawd, czy nazwa dostawcy jest obsugiwana
         DataTable providers = DbProviderFactories.GetFactoryClasses();
         DataRow[] foundArray = providers.Select("InvariantName='" + invariantName + "'");
         if (foundArray.Length == 0)
         {
            // kady bd bdzie obsugiwany przez niestandardow funkcj obsugi wyjtkw
            string msg = "[AbstractDA] Data Provider " + invariantName + " not found";
            DataAccessExceptionHandler.HandleException(msg);
         }

         // pobierz odpowiedni fabryk dostawcw
         return DbProviderFactories.GetFactory(invariantName);
      }
   }
}
