﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;
using Microsoft.AspNetCore.Mvc;
using Microsoft.AspNetCore.Mvc.Rendering;
using Microsoft.EntityFrameworkCore;
using TriathlonRaceTracking.Models;
using TriathlonRaceTracking.Data;

namespace TriathlonRaceTracking.Controllers
{
    public class TimingPointsController : Controller
    {
        private readonly TriathlonRaceTrackingContext _context;

        public TimingPointsController(TriathlonRaceTrackingContext context)
        {
            _context = context;
        }

        // GET: TimingPoints
        public async Task<IActionResult> Index()
        {
            var triathlonRaceTrackingContext = _context.TimingPoints.Include(t => t.Race);
            return View(await triathlonRaceTrackingContext.ToListAsync());
        }

        // GET: TimingPoints/Details/5
        public async Task<IActionResult> Details(int? id)
        {
            if (id == null)
            {
                return NotFound();
            }

            var timingPoint = await _context.TimingPoints
                .Include(t => t.Race)
                .SingleOrDefaultAsync(m => m.ID == id);
            if (timingPoint == null)
            {
                return NotFound();
            }

            return View(timingPoint);
        }

        // GET: TimingPoints/Create
        public IActionResult Create()
        {
            ViewData["RaceID"] = new SelectList(_context.Races, "ID", "Name");
            return View();
        }

        // POST: TimingPoints/Create
        // Aby chronić przed atakami należy włączyć konkretne właściwości, z którymi ma następować łączenie. 
        // Więcej informacji na ten temat znajdziesz na stronie http://go.microsoft.com/fwlink/?LinkId=317598.
        [HttpPost]
        [ValidateAntiForgeryToken]
        public async Task<IActionResult> Create([Bind("ID,RaceID,Name,Type")] TimingPoint timingPoint)
        {
            if (ModelState.IsValid)
            {
                _context.Add(timingPoint);
                await _context.SaveChangesAsync();
                return RedirectToAction(nameof(Index));
            }
            ViewData["RaceID"] = new SelectList(_context.Races, "ID", "Name", timingPoint.RaceID);
            return View(timingPoint);
        }

        // GET: TimingPoints/Edit/5
        public async Task<IActionResult> Edit(int? id)
        {
            if (id == null)
            {
                return NotFound();
            }

            var timingPoint = await _context.TimingPoints.SingleOrDefaultAsync(m => m.ID == id);
            if (timingPoint == null)
            {
                return NotFound();
            }
            ViewData["RaceID"] = new SelectList(_context.Races, "ID", "Name", timingPoint.RaceID);
            return View(timingPoint);
        }

        // POST: TimingPoints/Edit/5
        // Aby chronić przed atakami należy włączyć konkretne właściwości, z którymi ma następować łączenie. 
        // Więcej informacji na ten temat znajdziesz na stronie http://go.microsoft.com/fwlink/?LinkId=317598.
        [HttpPost]
        [ValidateAntiForgeryToken]
        public async Task<IActionResult> Edit(int id, [Bind("ID,RaceID,Name,Type")] TimingPoint timingPoint)
        {
            if (id != timingPoint.ID)
            {
                return NotFound();
            }

            if (ModelState.IsValid)
            {
                try
                {
                    _context.Update(timingPoint);
                    await _context.SaveChangesAsync();
                }
                catch (DbUpdateConcurrencyException)
                {
                    if (!TimingPointExists(timingPoint.ID))
                    {
                        return NotFound();
                    }
                    else
                    {
                        throw;
                    }
                }
                return RedirectToAction(nameof(Index));
            }
            ViewData["RaceID"] = new SelectList(_context.Races, "ID", "Name", timingPoint.RaceID);
            return View(timingPoint);
        }

        // GET: TimingPoints/Delete/5
        public async Task<IActionResult> Delete(int? id)
        {
            if (id == null)
            {
                return NotFound();
            }

            var timingPoint = await _context.TimingPoints
                .Include(t => t.Race)
                .SingleOrDefaultAsync(m => m.ID == id);
            if (timingPoint == null)
            {
                return NotFound();
            }

            return View(timingPoint);
        }

        // POST: TimingPoints/Delete/5
        [HttpPost, ActionName("Delete")]
        [ValidateAntiForgeryToken]
        public async Task<IActionResult> DeleteConfirmed(int id)
        {
            var timingPoint = await _context.TimingPoints.SingleOrDefaultAsync(m => m.ID == id);
            _context.TimingPoints.Remove(timingPoint);
            await _context.SaveChangesAsync();
            return RedirectToAction(nameof(Index));
        }

        private bool TimingPointExists(int id)
        {
            return _context.TimingPoints.Any(e => e.ID == id);
        }
    }
}
