﻿using System;
using System.ComponentModel;
using System.Linq;
using System.Web.Mvc;
using System.Web.Security;
using MvcContrib.Pagination;
using MvcContrib.Sorting;
using MvcContrib.UI.Grid;
using PortalUslug.Models;
using PortalUslug.Repositories;
using WebMatrix.WebData;

namespace PortalUslug.Controllers
{
    public class ServiceProviderController : Controller
    {
        /// <summary>
        /// Repozytorium usługodawcy.
        /// </summary>
        private ServiceProviderRepository _providerRepo;

        /// <summary>
        /// Repozytorium usług.
        /// </summary>
        private ServiceRepository _serviceRepo;

        /// <summary>
        /// Repozytorium komentarzy.
        /// </summary>
        private CommentRepository _commentRepo;

        /// <summary>
        /// Konstruktor kontrolera usługodawcy.
        /// </summary>
        public ServiceProviderController()
        {
            // Inicjalizacja repozytoriów 
            _providerRepo = new ServiceProviderRepository();
            _serviceRepo = new ServiceRepository();
            _commentRepo = new CommentRepository();
        }

        //
        // GET: /ServiceProvider/

        [Authorize]
        public ActionResult Index(string name, string city, string street, GridSortOptions gridSortOptions, [DefaultValue(1)]int page)
        {
            // Pobranie listy usługodawców
            var providersList = _providerRepo.GetAllServiceProviders();

            // Ograniczenie listy usługodawców do potwierdzonych dla użytkowników zalogowanych innych niż administrator
            if (!Roles.GetRolesForUser(WebSecurity.CurrentUserName).Contains("Administrator"))
            {
                providersList = providersList.Where(a => a.IsConfirmed);
            }

            // Ustawienie domyślnej kolumny sortowania
            if (string.IsNullOrWhiteSpace(gridSortOptions.Column))
            {
                gridSortOptions.Column = "Id";
            }

            // Filtrowanie po nazwie
            if (!string.IsNullOrWhiteSpace(name))
            {
                providersList = providersList.Where(a => a.Name.Contains(name));
            }

            // Filtrowanie po mieście
            if (!string.IsNullOrWhiteSpace(city))
            {
                providersList = providersList.Where(a => a.City.Contains(city));
            }

            // Filtrowanie po ulicy
            if (!string.IsNullOrWhiteSpace(street))
            {
                providersList = providersList.Where(a => a.Street.Contains(street));
            }

            var providerFilterViewModel = new ServiceProviderFilterViewModel();

            // Sortowanie listy usługodawców oraz stronicowanie
            var providerPagedList = providersList.OrderBy(gridSortOptions.Column, gridSortOptions.Direction)
                   .AsPagination(page, 5);

            var providerListContainer = new ServiceProviderListContainerViewModel
            {
                ServiceProviderPagedList = providerPagedList,
                ServiceProviderFilterViewModel = providerFilterViewModel,
                GridSortOptions = gridSortOptions
            };

            return View(providerListContainer);
        }

        //
        // GET: /ServiceProvider/Details/5

        [Authorize]
        public ActionResult Details(int id, string name, string content, int? categoryId, GridSortOptions gridSortOptions,
           [DefaultValue(1)] int page)
        {
            // Pobranie usługodawcy po identyfikatorze
            var provider = _providerRepo.GetServiceProviderById(id);

            // Pobranie listy usług danego usługodawcy
            var servicesList = _serviceRepo.GetActiveServicesByUserId(provider.UserId);

            // Ustawienie domyślnej kolumny sortowania
            if (string.IsNullOrWhiteSpace(gridSortOptions.Column))
            {
                gridSortOptions.Column = "Id";
            }

            // Filtrowanie po kategorii usługi
            if (categoryId.HasValue)
            {
                servicesList = servicesList.Where(a => a.CategoryId == categoryId);
            }

            // Filtrowanie po nazwie usługi
            if (!string.IsNullOrWhiteSpace(name))
            {
                servicesList = servicesList.Where(a => a.Name.Contains(name));
            }

            // Filtrowanie po opisie usługi
            if (!string.IsNullOrWhiteSpace(content))
            {
                servicesList = servicesList.Where(a => a.Content.Contains(content));
            }

            // Utworzenie modelu do filtrowania usług
            var servicesFilterViewModel = new ServiceFilterViewModel();
            servicesFilterViewModel.SelectedCategoryId = categoryId ?? -1;
            servicesFilterViewModel.Fill();

            // Stronicowanie i sortowanie listy usług
            var servicesPagedList = servicesList.OrderBy(gridSortOptions.Column, gridSortOptions.Direction)
                   .AsPagination(page, 10);

            var servicesListContainer = new ServiceListContainerViewModel
            {
                ServicePagedList = servicesPagedList,
                ServiceFilterViewModel = servicesFilterViewModel,
                GridSortOptions = gridSortOptions
            };

            var serviceProviderServicesListContainer = new ServiceProviderServicesListContainerViewModel
            {
                Services = servicesListContainer,
                ServiceProvider = provider,
            };

            return View(serviceProviderServicesListContainer);
        }

        [Authorize(Roles = "ServiceProvider")]
        public ActionResult Change()
        {
            // Pobranie aktualnie zalogowanego usługodawcy i przekierowanie do akcji Edit
            var provider = _providerRepo.GetServiceProviderByUserId(WebSecurity.CurrentUserId);
            return RedirectToAction("Edit", new { id = provider.Id });
        }

        [Authorize(Roles = "Administrator, ServiceProvider")]
        public ActionResult Edit(int id)
        {
            // Pobranie usługodawcy po identyfikatorze
            var provider = _providerRepo.GetServiceProviderById(id);

            // Sprawdzenie, czy użytkownik próbuje zmienić swoje własne dane
            if (provider.UserId == WebSecurity.CurrentUserId)
            {
                return View(provider);
            }
            // Jeżeli użytkownik próbuje modyfikować cudze dane, to może jest administratorem.
            else if (Roles.GetRolesForUser(WebSecurity.CurrentUserName).Contains("Administrator"))
            {
                return View(provider);
            }
            else
            {
                TempData["Error"] = "Nie masz uprawnień do edytowania tego usługodawcy.";
                return View("Index");
            }
        }

        //
        // POST: /ServiceProvider/Edit/5
        [Authorize(Roles = "Administrator, ServiceProvider")]
        [HttpPost]
        public ActionResult Edit(int id, FormCollection formValues)
        {
            // Pobranie usługobiorcy po identyfikatorze
            var provider = _providerRepo.GetServiceProviderById(id);

            // Uaktualnienie modelu o nowe dane
            if (TryUpdateModel(provider))
            {
                // Zapisanie zmian w bazie danych
                _providerRepo.SaveChanges();
                TempData["Message"] = "Pomyślnie zmodyfikowano usługodawcę!";
                return RedirectToAction("Details", new { id = provider.Id });
            }
            else
            {
                TempData["Error"] = "Wystąpił błąd!";
                return View(provider);
            }
        }

        //
        // GET: /ServiceProvider/Delete/5

        [Authorize(Roles = "Administrator")]
        public ActionResult Delete(int id)
        {
            // Pobranie usługodawcy po identyfikatorze
            var provider = _providerRepo.GetServiceProviderById(id);

            if (provider != null)
            {
                // Sprawdzenie, czy użytkownik dodał usługę
                bool hasServices = _serviceRepo.HasUserServices(provider.UserId);

                // Sprawdzenie, czy użytkownik dodał komentarze
                bool hasComments = _commentRepo.HasUserComment(provider.UserId);

                if (hasServices || hasComments)
                {
                    // Jeżeli użytkownik dodał usługę lub komentarz nie może zostać usunięty
                    TempData["Error"] = "Nie można usunąć tego usługodawcy.";
                    return RedirectToAction("Index");
                }
                else
                {
                    return View(provider);
                }
            }
            else
            {
                TempData["Error"] = "Brak usługodawcy o podanym id!";
                return RedirectToAction("Index");
            }
        }

        //
        // POST: /ServiceProvider/Delete/5

        [Authorize(Roles = "Administrator")]
        [HttpPost]
        public ActionResult Delete(int id, FormCollection collection)
        {
            // Pobranie usługodawcy po identyfikatorze
            var provider = _providerRepo.GetServiceProviderById(id);

            try
            {
                // Usunięcie usługodawcy i zapisanie zmian
                _providerRepo.Delete(provider);
                _providerRepo.SaveChanges();
            }
            catch (Exception)
            {
                TempData["Error"] = "Nie można usunąć tego usługodawcy";
                return View(provider);
            }

            TempData["Message"] = "Usługodawca został usunięty.";
            return RedirectToAction("Index");
        }
    }
}
