/// <reference path="workout.spec.js" />
describe("Kontrolery", function () {

    beforeEach(module('app'));

    beforeEach(module(function ($provide) {
        $provide.factory("WorkoutService", function ($q, WorkoutPlan, Exercise) {
            var mock = {};
            mock.sampleWorkout = new WorkoutPlan({
                name: "treningtestowy",
                title: "Trening testowy",
                description: "To jest trening testowy",
                restBetweenExercise: "40",
                exercises: [{ details: new Exercise({ name: "ćwiczenie1", title: "Ćwiczenie 1", description: "Opis ćwiczenia 1", image: "/image1/path", nameSound: "audio1/path" }), duration: 50 },
                            { details: new Exercise({ name: "ćwiczenie2", title: "Ćwiczenie 2", description: "Opis ćwiczenia 2", image: "/image2/path", nameSound: "audio2/path" }), duration: 30 },
                            { details: new Exercise({ name: "ćwiczenie3", title: "Ćwiczenie 3", description: "Opis ćwiczenia 3", image: "/image3/path", nameSound: "audio3/path" }), duration: 20 }, ]
            });
            mock.getWorkout = function (name) {
                return $q.when(mock.sampleWorkout);
            }
            mock.totalWorkoutDuration = 180;
            return mock;
        });
    }));

    describe("WorkoutController", function () {
        var ctrl, $scope;

        beforeEach(function () {
            module(function ($provide) {
                $provide.value("workoutHistoryTracker", { startTracking: function () { }, endTracking: function () { } });
            });
        });

        beforeEach(inject(function ($rootScope, $controller, $interval, $location, $timeout, workoutHistoryTracker, WorkoutService, appEvents, Exercise) {
            $scope = $rootScope.$new();
            $scope.carousel = { next: function () { } };
            ctrl = $controller('WorkoutController', {
                $scope: $scope,
                $interval: $interval,
                $location: $location,
                $timeout: $timeout,
                workoutHistoryTracker: workoutHistoryTracker,
                appEvents: appEvents,
                WorkoutService: WorkoutService,
                $routeParams: { id: "TreningTestowy" },
                Exercise: Exercise
            });

            spyOn(workoutHistoryTracker, 'startTracking');
            spyOn(workoutHistoryTracker, 'endTracking');
            spyOn($scope, "$emit");
            spyOn($scope.carousel, "next");

            $scope.$digest();
        }));

        it("powinien wczytać kontroler workoutController", function () {
            expect(ctrl).toBeDefined();
        });

        it("powinien uruchomić trening", inject(function (WorkoutService) {
            expect($scope.workoutPlan).toEqual(WorkoutService.sampleWorkout);
            expect($scope.workoutTimeRemaining).toEqual(WorkoutService.totalWorkoutDuration);
            expect($scope.workoutPaused).toBeFalsy();
        }));

        it("powinien uruchomić pierwsze ćwiczenie", inject(function (WorkoutService, appEvents) {
            expect($scope.currentExercise).toEqual(WorkoutService.sampleWorkout.exercises[0]);
            expect($scope.currentExerciseIndex).toEqual(0);
            expect($scope.$emit).toHaveBeenCalledWith(appEvents.workout.exerciseStarted, WorkoutService.sampleWorkout.exercises[0].details);
        }));

        it("powinien prawidłowo przygotować obrazki ćwiczeń i odpoczynków", function () {
            expect($scope.exerciseImages.length).toEqual(5);
            expect($scope.exerciseImages[1]).toEqual("img/rest.png");
            expect($scope.exerciseImages[3]).toEqual("img/rest.png");
        });

        it("powinien uruchomić rejestrację historii treningów", inject(function (workoutHistoryTracker) {
            expect(workoutHistoryTracker.startTracking).toHaveBeenCalled();
        }));

        it("powinien wraz z upływem czasu zwiększać czas wykonywania ćwiczenia", inject(function ($interval) {
            expect($scope.currentExerciseDuration).toBe(0);
            $interval.flush(1000);
            expect($scope.currentExerciseDuration).toBe(1);
            $interval.flush(1000);
            expect($scope.currentExerciseDuration).toBe(2);
            $interval.flush(8000);
            expect($scope.currentExerciseDuration).toBe(10);
        }));

        it("powinien wraz z upływem czasu zmniejszać pozostały czas trwania treningu", inject(function (WorkoutService, $interval) {
            expect($scope.workoutTimeRemaining).toBe(WorkoutService.totalWorkoutDuration);
            $interval.flush(1000);
            expect($scope.workoutTimeRemaining).toBe(WorkoutService.totalWorkoutDuration - 1);
            $interval.flush(1000);
            expect($scope.workoutTimeRemaining).toBe(WorkoutService.totalWorkoutDuration - 2);
        }));

        it("po zakończeniu jednego ćwiczenia powinien przejść do następnego", inject(function (WorkoutService, $interval) {
            $interval.flush(WorkoutService.sampleWorkout.exercises[0].duration * 1000);
            expect($scope.currentExercise.details.name).toBe('rest');
            expect($scope.currentExercise.duration).toBe(WorkoutService.sampleWorkout.restBetweenExercise);
        }));

        it("powinien naprzemiennie zmieniać ćwiczenie i odpoczynek", inject(function (WorkoutService, $interval) {
            // pierwsze ćwiczenie
            expect($scope.currentExercise).toBe(WorkoutService.sampleWorkout.exercises[0]);
            $interval.flush(WorkoutService.sampleWorkout.exercises[0].duration * 1000);
            // ćwiczenie - odpoczynek
            expect($scope.currentExercise.details.name).toBe('rest');
            expect($scope.currentExercise.duration).toBe(WorkoutService.sampleWorkout.restBetweenExercise);
            $interval.flush(WorkoutService.sampleWorkout.restBetweenExercise * 1000);
            // durgie ćwiczenie
            expect($scope.currentExercise).toBe(WorkoutService.sampleWorkout.exercises[1]);
            $interval.flush(WorkoutService.sampleWorkout.exercises[1].duration * 1000);
            // ćwiczenie - odpoczynek
            expect($scope.currentExercise.details.name).toBe('rest');
            expect($scope.currentExercise.duration).toBe(WorkoutService.sampleWorkout.restBetweenExercise);
        }));

        it("powinien resetować currentExerciseDuration po zmianie ćwiczenia", inject(function (WorkoutService, $interval) {
            expect($scope.currentExerciseDuration).toBe(0);
            $interval.flush(1000);
            expect($scope.currentExerciseDuration).toBe(1);
            $interval.flush(1000);
            expect($scope.currentExerciseDuration).toBe(2);
            $interval.flush((WorkoutService.sampleWorkout.exercises[0].duration - 2) * 1000);
            expect($scope.currentExerciseDuration).toBe(0);
        }));

        it("powinien zmienać obrazek w karuzeli na następny po zakończeniu ćwiczenia", inject(function (WorkoutService, $interval) {
            $interval.flush(WorkoutService.sampleWorkout.exercises[0].duration * 1000);
            expect($scope.carousel.next).toHaveBeenCalled();
            $interval.flush(WorkoutService.sampleWorkout.restBetweenExercise * 1000);
            expect($scope.carousel.next.calls.count()).toEqual(2);
        }));

        it("powinien zakończyć trening po wykonaniu wszystkich ćwiczeń", inject(function (WorkoutService, $interval, workoutHistoryTracker, $location) {
            $interval.flush(WorkoutService.sampleWorkout.exercises[0].duration * 1000);
            $interval.flush(WorkoutService.sampleWorkout.restBetweenExercise * 1000);
            $interval.flush(WorkoutService.sampleWorkout.exercises[1].duration * 1000);
            $interval.flush(WorkoutService.sampleWorkout.restBetweenExercise * 1000);
            $interval.flush(WorkoutService.sampleWorkout.exercises[2].duration * 1000);

            expect(workoutHistoryTracker.endTracking).toHaveBeenCalled();
            expect($location.path()).toEqual("/finish");
            expect($scope.workoutTimeRemaining).toBe(0);
            expect($scope.currentExercise).toBe(WorkoutService.sampleWorkout.exercises[2]);

        }));

        it("powinien wstrzymać trening po wywołaniu pauseWorkout", function () {
            expect($scope.workoutPaused).toBeFalsy();
            $scope.pauseWorkout();
            expect($scope.workoutPaused).toBe(true);
        });

        it("nie powinien aktualizować wartości workoutTimeRemaining wraz z upływem czasu, jeśli trening został wstrzymany", inject(function (WorkoutService, $interval) {
            expect($scope.workoutPaused).toBeFalsy();

            $interval.flush(1000);
            expect($scope.workoutTimeRemaining).toBe(WorkoutService.totalWorkoutDuration - 1);

            $scope.pauseWorkout();
            expect($scope.workoutPaused).toBe(true);

            $interval.flush(5000);
            expect($scope.workoutTimeRemaining).toBe(WorkoutService.totalWorkoutDuration - 1);
        }));

        it("nie powinien aktualizować wartości currentExerciseDuration wraz z upływem czasu, jeśli trening został wstrzymany", inject(function (WorkoutService, $interval) {
            expect($scope.workoutPaused).toBeFalsy();

            $interval.flush(1000);
            expect($scope.currentExerciseDuration).toBe(1);

            $scope.pauseWorkout();
            expect($scope.workoutPaused).toBe(true);

            $interval.flush(5000);
            expect($scope.currentExerciseDuration).toBe(1);
        }));

        it("nie powinien zgłaszać błędów jeśli trening został wstrzymany wiele razy", function () {
            expect($scope.workoutPaused).toBeFalsy();
            $scope.pauseWorkout();
            expect($scope.workoutPaused).toBe(true);
            $scope.pauseWorkout();
            expect($scope.workoutPaused).toBe(true);
            $scope.pauseWorkout();
            expect($scope.workoutPaused).toBe(true);
        });

        it("powinien wznowić trening po wywołaniu resumeWorkout", inject(function ($interval) {
            expect($scope.workoutPaused).toBeFalsy();
            $scope.pauseWorkout();
            expect($scope.workoutPaused).toBe(true);
            $scope.resumeWorkout();
            expect($scope.workoutPaused).toBe(false);

            $interval.flush(1000);
            expect($scope.currentExerciseDuration).toBe(1);
        }));

        it("nie powinien zgłaszać błędu w przypadku wielu wywołań resumeWorkout", inject(function ($interval) {
            expect($scope.workoutPaused).toBeFalsy();
            $scope.pauseWorkout();
            expect($scope.workoutPaused).toBe(true);
            $scope.resumeWorkout();
            expect($scope.workoutPaused).toBe(false);

            $interval.flush(1000);
            expect($scope.currentExerciseDuration).toBe(1);

            $scope.resumeWorkout();
            expect($scope.workoutPaused).toBe(false);

            $interval.flush(1000);
            expect($scope.currentExerciseDuration).toBe(2);

            $interval.flush(1000);
            expect($scope.currentExerciseDuration).toBe(3);
        }));

        it("powinien przełączać stan treningu po wywołaniu pauseResumeToggle", function () {
            expect($scope.workoutPaused).toBeFalsy();
            $scope.pauseResumeToggle();
            expect($scope.workoutPaused).toBe(true);
            $scope.pauseResumeToggle();
            expect($scope.workoutPaused).toBeFalsy();
        });

        it("powinien przełączać stan treningu po wpisaniu litery 'p' lub 'P'", function () {
            expect($scope.workoutPaused).toBeFalsy();
            $scope.onKeyPressed({ which: 80 });
            expect($scope.workoutPaused).toBe(true);
            $scope.onKeyPressed({ which: 112 });
            expect($scope.workoutPaused).toBeFalsy();
        });
    });

    describe("WorkoutAudioController", function () {
        function AudioController() {
            this.pause = function () { }
            this.play = function () { }
            this.currentTime = 0;
            this.duration = 0;
        };

        var ctrl, $scope;

        beforeEach(inject(function ($rootScope, $controller, $interval, $location, $timeout, workoutHistoryTracker, WorkoutService, appEvents, Exercise, WorkoutService) {
            $scope = $rootScope.$new();

            // Atrapa kontrolera audio
            $scope.ticksAudio = new AudioController();
            $scope.nextUpAudio = new AudioController();
            $scope.nextUpExerciseAudio = new AudioController();
            $scope.halfWayAudio = new AudioController();
            $scope.aboutToCompleteAudio = new AudioController();


            ctrl = $controller('WorkoutAudioController', {
                $scope: $scope,
                $interval: $interval,
                $location: $location,
                $timeout: $timeout,
            });

            $scope.$digest();
        }));

        it("powinien wczytać WorkoutAudioController", function () {
            expect(ctrl).toBeDefined();
        });

        it("wczytać pliki audio po wczytaniu treningu", inject(function (WorkoutService) {
            $scope.workoutPlan = WorkoutService.sampleWorkout;
            $scope.$digest();
            expect($scope.exercisesAudio.length).toBe(3);
        }));

        it("powinien odtworzyć odpowiedni plik audio po upłynięciu połowy czasu wykonywania ćwiczenia", inject(function (WorkoutService) {
            spyOn($scope.halfWayAudio, "play");
            $scope.currentExercise = WorkoutService.sampleWorkout.exercises[0];
            $scope.currentExerciseDuration = 2;
            $scope.$digest();

            expect($scope.halfWayAudio.play).not.toHaveBeenCalled();

            $scope.currentExerciseDuration = WorkoutService.sampleWorkout.exercises[0].duration / 2;
            $scope.$digest();

            expect($scope.halfWayAudio.play).toHaveBeenCalled();
        }));

        it("powinien odtworzyć odpowiedni plik audio, gdy czas wykonywania ćwiczenia zacznie się zbliżać do końca", inject(function (WorkoutService) {
            spyOn($scope.aboutToCompleteAudio, "play");

            $scope.currentExercise = WorkoutService.sampleWorkout.exercises[0];

            $scope.currentExerciseDuration = 2;
            $scope.$digest();
            expect($scope.aboutToCompleteAudio.play).not.toHaveBeenCalled();

            $scope.currentExerciseDuration = WorkoutService.sampleWorkout.exercises[0].duration / 2;
            $scope.$digest();
            expect($scope.aboutToCompleteAudio.play).not.toHaveBeenCalled();

            $scope.currentExerciseDuration = WorkoutService.sampleWorkout.exercises[0].duration - 3;
            $scope.$digest();
            expect($scope.aboutToCompleteAudio.play).toHaveBeenCalled();

        }));

        it("powinien odtworzyć plik audio następnego ćwiczenia po zakończeniu ćwiczenia-odpoczynku", inject(function (WorkoutService, $timeout) {
            spyOn($scope.nextUpAudio, "play");
            spyOn($scope.nextUpExerciseAudio, "play");

            $scope.currentExercise = { details: { name: 'rest' } };
            $scope.$digest();

            expect($scope.nextUpAudio.play).not.toHaveBeenCalled();
            expect($scope.nextUpExerciseAudio.play).not.toHaveBeenCalled();

            $timeout.flush(2000);
            expect($scope.nextUpAudio.play).toHaveBeenCalled();
            expect($scope.nextUpExerciseAudio.play).not.toHaveBeenCalled();

            $timeout.flush(1000);
            expect($scope.nextUpExerciseAudio.play).toHaveBeenCalled();
        }));

        it("powinien wstrzymać odtwarzania wszystkich klipów po wstrzymaniu treningu", function () {
            spyOn($scope.ticksAudio, "pause");
            spyOn($scope.nextUpAudio, "pause");
            spyOn($scope.nextUpExerciseAudio, "pause");
            spyOn($scope.halfWayAudio, "pause");
            spyOn($scope.aboutToCompleteAudio, "pause");

            $scope.workoutPaused = true;
            $scope.$digest();

            expect($scope.ticksAudio.pause).toHaveBeenCalled();
            expect($scope.nextUpAudio.pause).toHaveBeenCalled();
            expect($scope.nextUpExerciseAudio.pause).toHaveBeenCalled();
            expect($scope.halfWayAudio.pause).toHaveBeenCalled();
            expect($scope.aboutToCompleteAudio.pause).toHaveBeenCalled();

        });

        it("powinien wznowić odtwarzanie dźwięku upływającego czasu po wznowieniu treningu", function () {
            spyOn($scope.ticksAudio, "play");

            $scope.workoutPaused = false;
            $scope.$digest();

            expect($scope.ticksAudio.play).toHaveBeenCalled();
        });
    });
});