import {Component, ViewChild, Input, Output, ViewEncapsulation, EventEmitter, HostBinding, HostListener} from '@angular/core';
import template from './editor.html!text';
import {TagInputManager} from '../../tags/tag-input-manager';

@Component({
  selector: 'ngc-editor',
  host: {
    class: 'editor'
  },
  template,
  encapsulation: ViewEncapsulation.None
})
export class Editor {
  // Zapamiętanie referencji do widoku potomnego w lokalnej zmiennej.
  @ViewChild('editableContentElement') editableContentElement;
  // Edytowana i wyświetlana treść.
  @Input() content;
  // Utworzenie dowiązania do atrybutu class elementu hostującego, który będzie zależał od właściwości editMode.
  @Input() @HostBinding('class.editor--edit-mode') editMode;
  @Input() showControls;
  // Czy edytor powinien obsługiwać etykiety?
  @Input() enableTags;
  @Output() editModeChange = new EventEmitter();
  @Output() editSaved = new EventEmitter();
  @Output() editCanceled = new EventEmitter();
  @Output() editableInput = new EventEmitter();

  constructor() {
    // Używamy TagInputManager, aby wspomóc proces obsługi etykiet.
    this.tagInputManager = new TagInputManager();
  }

  // Metoda wywoływana, gdy edytowany element otrzyma zdarzenie keydown.
  onKeyDown(event) {
    // Delegujemy zdarzenie keydown do obiektu klasy TagInputManager.
    this.tagInputManager.onKeyDown(event);
  }

  // Metoda wywoływana, gdy edytowany element otrzyma zdarzenie keypress.
  onKeyPress(event) {
    // Delegujemy zdarzenie keypress do obiektu klasy TagInputManager.
    this.tagInputManager.onKeyPress(event);
  }

  // Metoda wywoływana, gdy komponent potomny tag-select emituje zdarzenie wybrania etykiety.
  onTagSelected(tag) {
    // Zastępujemy wpisany fragment znacznika etykiety pełną reprezentacją wybraną  listy przez użytkownika.
    this.setEditableContent(this.getEditableContent().replace(this.tagInputManager.textTag, tag.textTag));
    this.tagInputManager.reset();
  }

  // Musimy mieć pewność, aby odwzorować zmianę w edytowanym elemencie, jeśli treść zostanie zaktualizowana z zewnątrz.
  ngOnChanges() {
    if (this.editableContentElement && this.content) {
      this.setEditableContent(this.content);
    }
  }

  ngAfterViewInit() {
    this.setEditableContent(this.content);
  }

  // Zwraca zawartość edytowanej treści.
  getEditableContent() {
    return this.editableContentElement.nativeElement.textContent;
  }

  // Ustawia zawartość edytowanej treści.
  setEditableContent(content) {
    this.editableContentElement.nativeElement.textContent = content;
  }

  // Zmiana trybu edycji.
  switchEditMode(editMode) {
    this.editMode = editMode;
    this.editModeChange.next(editMode);
  }

  // Adnotacja utworzy kod nasłuchujący zdarzenia kliknięcia elementu hostującego, który spowoduje wywołanie kodu umieszczonego w funkcji.
  @HostListener('click')
  focusEditableContent() {
    if (this.editMode) {
      this.editableContentElement.nativeElement.focus();
    }
  }

  // Metoda wywoływana, jeśli edytowany element ulegnie zmianie.
  onInput() {
    // Wyemituj zdarzenie editableInput z edytowaną treścią.
    this.editableInput.next(this.getEditableContent());
  }

  // W przypadku zapisu pobieramy treść edytowanego elementu, przenosimy ją do pola content i emitujemy zdarzenie.
  save() {
    this.editSaved.next(this.getEditableContent());
    this.setEditableContent(this.content);
    // Ustawienie editMode na wartość false wyłącza tryb edycji.
    this.switchEditMode(false);
    // Dodatkowo czyścimy menedżera wpisywania etykiety, jeśli użytkownik wybrał zapis aktualnej wersji.
    this.tagInputManager.reset();
  }

  // Anulowanie edycji nie kopiuje edytowanej zawartości i przełącza element w tryb podglądu.
  cancel() {
    this.setEditableContent(this.content);
    this.editCanceled.next(this.content);
    this.editableInput.next(this.getEditableContent());
    this.switchEditMode(false);
    // Dodatkowo czyścimy menedżera wpisywania etykiety, jeśli użytkownik wybrał anulowanie aktualnej wersji tekstu.
    this.tagInputManager.reset();
  }

  // Metoda ta powoduje włączenie edytowalnego elementu i przejście w tryb edycji.
  edit() {
    this.switchEditMode(true);
  }
}
