import {Injectable, Inject} from '@angular/core';
import {DataProvider} from '../../../data-access/data-provider';
import {UserService} from '../../user/user-service/user-service';
import {ReplaySubject} from 'rxjs/Rx';

@Injectable()
export class ActivityService {
  constructor(@Inject(DataProvider) dataProvider, @Inject(UserService) userService) {
    // Udostępniamy obiekt ReplaySubject, który będzie emitował zdarzenia, gdy tylko zmieni się lista aktywności.
    this.change = new ReplaySubject(1);
    this.dataProvider = dataProvider;
    this.userService = userService;
    this.activities = [];

    // Dokonujemy subskrypcji zmian w magazynie danych, aby otrzymywać aktualizacje dotyczące aktywności.
    this.activitiesSubscription = this.dataProvider.getLiveChanges()
      .map((change) => change.doc)
      .filter((document) => document.type === 'activity')
      .subscribe((changedActivity) => {
        this.activities = this.activities.slice();
        // Ponieważ aktywności można tylko dodawać, zakładamy, że zmiana jest nową aktywnością.
        this.activities.push(changedActivity);
        // Sortujemy aktywności na podstawie czasu, aby mieć pewność, że problemy z synchronizacją nie spowodują zmiany kolejności.
        this.activities.sort((a, b) => a.time > b.time ? -1 : a.time < b.time ? 1 : 0);
        this.change.next(this.activities);
      });
  }

  // Metoda odpowiada za umieszczenie w dzienniku nowej aktywności.
  logActivity(subject, category, title, message) {
    // Użycie DataProvider do utworzenia nowego dokumentu w magazynie danych.
    this.dataProvider.createOrUpdateDocument({
      type: 'activity',
      user: this.userService.currentUser,
      time: new Date().getTime(),
      subject,
      category,
      title,
      message
    });
  }
}
