import {Injectable, Inject} from '@angular/core';
import {ReplaySubject} from 'rxjs/Rx';
import {ProjectService} from '../project/project-service/project-service';
import {Tag} from './tag';
import {generateTag} from './generate-tag';

// Funkcja pomocnicza zastępująca wszystkie wystapienia tekstu innym tekstem.
function replaceAll(target, search, replacement) {
  return target.split(search).join(replacement);
}

// Funkcja poszukuje w tekście etykiet i zwraca tablicę ze znalezionymi etykietami.
function findTags(str) {
  const result = [];
  const regex = /#[\w\/-]+/g;
  let match;
  while (match = regex.exec(str)) {
    result.push(match[0]);
  }
  return result;
}

@Injectable()
export class TagsService {
  constructor(@Inject(ProjectService) projectService) {
    // Jeśli dostępne w systemie etykiety ulegną zmianie, wyemitujemy to zdarzenie.
    this.change = new ReplaySubject(1);
    // Aby wygenerować etykiety dla projektu, użyjemy usługi.
    this.projectService = projectService;
    this.projectService.change.subscribe((projects) => {
      // Po zmianach w projektach, zapamiętujemy nową listę projektów i ponownie inicjalizujemy listę etykiet.
      this.projects = projects;
      this.initializeTags();
    });
  }

  // Metoda używana wewnętrznie do inicjalizacji wszystkich dostępnych etykiet.
  initializeTags() {
    // Tworzymy etykiety dla wszystkich projektów za pomocą funkcji generateTag().
    this.tags = this.projects.map(generateTag);
    // Ponieważ uaktualniliśmy listę etykiet, emitujemy zdarzenie zmiany.
    this.change.next(this.tags);
  }

  // Metoda ta służy do wygenerowania reprezentacji HTML etykiety.
  renderTag(tag) {
    if (tag instanceof Tag) {
      return `<a class="tags__tag tags__tag--${tag.type}" href="${tag.link}">${tag.title}</a>`;
    } else {
      return tag;
    }
  }

  // Metoda wyszukuje etykietę po jej reprezentacji tekstowej lub zwraca argument wejściowy, jeśli jej nie znajdzie.
  parseTag(textTag) {
    return this.tags.find((tag) => tag.textTag === textTag) || textTag;
  }

  // Metoda przyjmuje tekst i zamienia w nim wszystkie wystąpienia etykiet na ich reprezentacje HTML.
  parse(value) {
    // First we find all possible tags within the text
    const tags = findTags(value);
    // For each found text tag, we're parsing and rendering them while replacing the text tag with the HTML representation if applicable
    tags.forEach((tag) => value = replaceAll(value, tag, this.renderTag(this.parseTag(tag))));
    return value
  }
}
