/*
 * Copyright (C) 2012 Juhani Lehtimaki
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.androiduipatterns.smashingandroidui.examples.tabs;

import android.app.ActionBar;
import android.app.FragmentTransaction;
import android.os.Bundle;
import android.support.v4.app.Fragment;
import android.support.v4.app.FragmentActivity;
import android.support.v4.app.FragmentManager;
import android.support.v4.app.FragmentPagerAdapter;
import android.support.v4.view.ViewPager;
import android.view.Gravity;
import android.view.LayoutInflater;
import android.view.Menu;
import android.view.View;
import android.view.ViewGroup;
import android.widget.TextView;

import com.androiduipatterns.smashingandroidui.examples.R;

public class TabsExampleActivity extends FragmentActivity implements ActionBar.TabListener {

    /**
     * Obiekt typu {@link android.support.v4.view.PagerAdapter} udostpniajcy fragmenty dla poszczeglnych
     * sekcji. Wykorzystujemy klas potomn {@link android.support.v4.app.FragmentPagerAdapter}, ktra zachowa
     * wszystkie zaadowane fragmenty w pamici. Jeli to rozwizanie bdzie zbyt mocno obciao pami,
     * warto rozway uycie klasy {@link android.support.v4.app.FragmentStatePagerAdapter}.
     */
    SectionsPagerAdapter mSectionsPagerAdapter;

    /**
     * Obiekt klasy {@link ViewPager}, ktry bdzie zawiera treci poszczeglnych sekcji.
     */
    ViewPager mViewPager;

    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_tabs_example);
        // Tworzy adapter, ktry zwrci fragment dla kadej z trzech gwnych sekcji
        // aplikacji.
        mSectionsPagerAdapter = new SectionsPagerAdapter(getSupportFragmentManager());

        // Ustawia pasek akcji.
        final ActionBar actionBar = getActionBar();
        actionBar.setNavigationMode(ActionBar.NAVIGATION_MODE_TABS);

        // Ustawia komponent ViewPager z adapterem sekcji.
        mViewPager = (ViewPager) findViewById(R.id.pager);
        mViewPager.setAdapter(mSectionsPagerAdapter);

        // Podczas przerzucania sekcji zaznacza odpowiedni zakadk.
        // Gdybymy dysponowali referencj do zakadki, moglibymy uy
        // do tego celu metody ActionBar.Tab#select().
        mViewPager.setOnPageChangeListener(new ViewPager.SimpleOnPageChangeListener() {
            @Override
            public void onPageSelected(int position) {
                actionBar.setSelectedNavigationItem(position);
            }
        });

        // Dla kadej sekcji w ramach aplikacji dodaje zakadk do paska akcji.
        for (int i = 0; i < mSectionsPagerAdapter.getCount(); i++) {
            // Tworzy zakadk z tekstem waciwym tytuowi strony zdefiniowanemu przez adapter.
            // Dodatkowo wskazujemy ten obiekt Activity (implementujcy interfejs TabListener) jako
            // obiekt nasuchujcy zdarzenia zaznaczenia tej zakadki.
            actionBar.addTab(
                    actionBar.newTab()
                            .setText(mSectionsPagerAdapter.getPageTitle(i))
                            .setTabListener(this));
        }
    }

    @Override
    public boolean onCreateOptionsMenu(Menu menu) {
        getMenuInflater().inflate(R.menu.activity_tabs_example, menu);
        return true;
    }

    

    @Override
    public void onTabUnselected(ActionBar.Tab tab, FragmentTransaction fragmentTransaction) {
    }

    @Override
    public void onTabSelected(ActionBar.Tab tab, FragmentTransaction fragmentTransaction) {
        // W momencie zaznaczenia tej zakadki metoda przecza na odpowiedni stron w ramach ViewPagera.
        mViewPager.setCurrentItem(tab.getPosition());
    }

    @Override
    public void onTabReselected(ActionBar.Tab tab, FragmentTransaction fragmentTransaction) {
    }

    /**
     * Klasa potomna wzgldem {@link FragmentPagerAdapter}, ktra zwraca fragment waciwy jednej
     * z podstawowych sekcji aplikacji.
     */
    public class SectionsPagerAdapter extends FragmentPagerAdapter {

        public SectionsPagerAdapter(FragmentManager fm) {
            super(fm);
        }

        @Override
        public Fragment getItem(int i) {
            Fragment fragment = new DummySectionFragment();
            Bundle args = new Bundle();
            args.putInt(DummySectionFragment.ARG_SECTION_NUMBER, i + 1);
            fragment.setArguments(args);
            return fragment;
        }

        @Override
        public int getCount() {
            return 3;
        }

        @Override
        public CharSequence getPageTitle(int position) {
            switch (position) {
                case 0: return getString(R.string.title_section1).toUpperCase();
                case 1: return getString(R.string.title_section2).toUpperCase();
                case 2: return getString(R.string.title_section3).toUpperCase();
            }
            return null;
        }
    }

    /**
     * Przykadowy fragment reprezentujcy sekcj aplikacji (ten fragment ogranicza si do wywietlenia przypadkowego tekstu).
     */
    public static class DummySectionFragment extends Fragment {
        public DummySectionFragment() {
        }

        public static final String ARG_SECTION_NUMBER = "section_number";

        @Override
        public View onCreateView(LayoutInflater inflater, ViewGroup container,
                Bundle savedInstanceState) {
            TextView textView = new TextView(getActivity());
            textView.setGravity(Gravity.CENTER);
            Bundle args = getArguments();
            textView.setText(Integer.toString(args.getInt(ARG_SECTION_NUMBER)));
            return textView;
        }
    }
}
