/*
 * Copyright (C) 2009 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gameproject.multiplayerpaddle;

import java.util.Set;

import android.annotation.TargetApi;
import android.app.Activity;
import android.bluetooth.BluetoothAdapter;
import android.bluetooth.BluetoothDevice;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.os.Bundle;
import android.util.Log;
import android.view.View;
import android.view.Window;
import android.view.View.OnClickListener;
import android.widget.AdapterView;
import android.widget.ArrayAdapter;
import android.widget.Button;
import android.widget.ListView;
import android.widget.TextView;
import android.widget.AdapterView.OnItemClickListener;

/**
 * Ta Aktywno pojawia si w postaci okna dialogowego z wymienionymi wszystkimi sparowanymi urzdzeniami
 * a take urzdzeniami wykrytymi w obszarze otoczenia. Po wybraniu urzdzenia,When a device is chosen
 * jego adres MAC jest odsyany do nadrzdnej Aktywnoci 
 */
@TargetApi(11)
public class DeviceListActivity extends Activity {
    // Debugowanie
    private static final String TAG = "DeviceListActivity";
    private static final boolean D = true;

    public static String EXTRA_DEVICE_ADDRESS = "device_address";

    // Pola klasy
    private BluetoothAdapter mBtAdapter;
    private ArrayAdapter<String> mPairedDevicesArrayAdapter;
    private ArrayAdapter<String> mNewDevicesArrayAdapter;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        // Nowe okno
        requestWindowFeature(Window.FEATURE_INDETERMINATE_PROGRESS);
        setContentView(R.layout.device_list);

        // W przypadku anulowania ustaw wynik CANCELED
        setResult(Activity.RESULT_CANCELED);

        // Utwrz przycisk do przeszukiwania otoczenia
        Button scanButton = (Button) findViewById(R.id.button_scan);
        scanButton.setOnClickListener(new OnClickListener() {
            public void onClick(View v) {
                doDiscovery();
                v.setVisibility(View.GONE);
            }
        });

        // Inicjalizacja tablic. Jednej dla aktualnie sparowanych urzdze i drugiej dla nowo odkrytych
        mPairedDevicesArrayAdapter = new ArrayAdapter<String>(this, R.layout.device_name);
        mNewDevicesArrayAdapter = new ArrayAdapter<String>(this, R.layout.device_name);

        // Konfiguracja widoku listy dla sparowanych urzdze
        ListView pairedListView = (ListView) findViewById(R.id.paired_devices);
        pairedListView.setAdapter(mPairedDevicesArrayAdapter);
        pairedListView.setOnItemClickListener(mDeviceClickListener);

        // Konfiguracja widoku listy dla nowo wykrytych urzdze
        ListView newDevicesListView = (ListView) findViewById(R.id.new_devices);
        newDevicesListView.setAdapter(mNewDevicesArrayAdapter);
        newDevicesListView.setOnItemClickListener(mDeviceClickListener);

        // Zarejestruj nadawanie po wykryciu urzdzenia
        IntentFilter filter = new IntentFilter(BluetoothDevice.ACTION_FOUND);
        this.registerReceiver(mReceiver, filter);

        // Zarejestruj nadawnie po zakoczeniu wykrywania
        filter = new IntentFilter(BluetoothAdapter.ACTION_DISCOVERY_FINISHED);
        this.registerReceiver(mReceiver, filter);

        // Pobierz lokalny adapter Bluetooth
        mBtAdapter = BluetoothAdapter.getDefaultAdapter();

        // Pobierz zestaw aktualnie sparowanych urzdze
        Set<BluetoothDevice> pairedDevices = mBtAdapter.getBondedDevices();

        // W przypadku istnienia urzdze sparowanych dodaj je do obiektu ArrayAdapter
        if (pairedDevices.size() > 0) {
            findViewById(R.id.title_paired_devices).setVisibility(View.VISIBLE);
            for (BluetoothDevice device : pairedDevices) {
                mPairedDevicesArrayAdapter.add(device.getName() + "\n" + device.getAddress());
            }
        } else {
            String noDevices = getResources().getText(R.string.none_paired).toString();
            mPairedDevicesArrayAdapter.add(noDevices);
        }
    }

    @Override
    protected void onDestroy() {
        super.onDestroy();

        // Upewnij si e nie jest wykonywane ponowne wykrywanie
        if (mBtAdapter != null) {
            mBtAdapter.cancelDiscovery();
        }

        // Usu rejestracj metod nasuchujcych
        this.unregisterReceiver(mReceiver);
    }

    /**
     * Uruchom wykrywaie urzdze za pomoc obiektu BluetoothAdapter
     */
    private void doDiscovery() {
        if (D) Log.d(TAG, "doDiscovery()");

        //  Sygnalizuj wyszukiwanie w nagwku
        setProgressBarIndeterminateVisibility(true);
        setTitle(R.string.scanning);

        // Dla nowych urzdze wywietl list
        findViewById(R.id.title_new_devices).setVisibility(View.VISIBLE);

        // Zakocz ewentualne rwnoleg wykrywanie
        if (mBtAdapter.isDiscovering()) {
            mBtAdapter.cancelDiscovery();
        }

        // Zarzdaj wyszukiwania od obiektu klasy BluetoothAdapter
        mBtAdapter.startDiscovery();
    }

    // Metoda nasuchujca kliknicia na licie urzdze 
    private OnItemClickListener mDeviceClickListener = new OnItemClickListener() {
        public void onItemClick(AdapterView<?> av, View v, int arg2, long arg3) {
            // Zakocz wyszukiwanie poniewa jest ono kosztowne a zaraz zostanie nawizanie poczenie
            mBtAdapter.cancelDiscovery();

            // Pobierz adres MAC
            String info = ((TextView) v).getText().toString();
            String address = info.substring(info.length() - 17);

            Intent intent = new Intent();
            intent.putExtra(EXTRA_DEVICE_ADDRESS, address);

            // Wylij wynik i zakocz aktywno
            setResult(Activity.RESULT_OK, intent);
            finish();
        }
    };

    //  BroadcastReceiver nasuchujcy wykrytych urzdze i zmieniajcy nagwek po wykryciu urzdzenia

    private final BroadcastReceiver mReceiver = new BroadcastReceiver() {
        @Override
        public void onReceive(Context context, Intent intent) {
            String action = intent.getAction();

            // Kiedy wyszukiwanie odkryje urzdzenie
            if (BluetoothDevice.ACTION_FOUND.equals(action)) {
                // Pobierz obiekt BluetoothDevice 
                BluetoothDevice device = intent.getParcelableExtra(BluetoothDevice.EXTRA_DEVICE);
                // Jeeli jest ju sparowane pomi
                if (device.getBondState() != BluetoothDevice.BOND_BONDED) {
                    mNewDevicesArrayAdapter.add(device.getName() + "\n" + device.getAddress());
                }
            // Po skoczeniu wyszukiwania, zmie nagwek Aktywnoci
            } else if (BluetoothAdapter.ACTION_DISCOVERY_FINISHED.equals(action)) {
                setProgressBarIndeterminateVisibility(false);
                setTitle(R.string.select_device);
                if (mNewDevicesArrayAdapter.getCount() == 0) {
                    String noDevices = getResources().getText(R.string.none_found).toString();
                    mNewDevicesArrayAdapter.add(noDevices);
                }
            }
        }
    };

}
