package com.manning.aip.dealdroid;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import android.app.ListActivity;
import android.app.ProgressDialog;
import android.content.Context;
import android.content.Intent;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.os.AsyncTask;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import android.view.LayoutInflater;
import android.view.Menu;
import android.view.MenuItem;
import android.view.View;
import android.view.ViewGroup;
import android.widget.AdapterView;
import android.widget.ArrayAdapter;
import android.widget.ImageView;
import android.widget.ListView;
import android.widget.Spinner;
import android.widget.TextView;
import android.widget.Toast;
import android.widget.AdapterView.OnItemSelectedListener;

import com.manning.aip.dealdroid.export.DealExporter;
import com.manning.aip.dealdroid.model.Item;
import com.manning.aip.dealdroid.model.Section;

public class DealList extends ListActivity {

   private static final int MENU_REPARSE = 0;
   private static final int MENU_EXPORT = 1;

   private DealDroidApp app;
   private List<Item> items;
   private DealsAdapter dealsAdapter;
   private ArrayAdapter<Section> spinnerAdapter;
   private int currentSelectedSection;
   private ProgressDialog progressDialog;
   private ParseFeedTask parseFeedTask;

   @Override
   public void onCreate(Bundle savedInstanceState) {
      super.onCreate(savedInstanceState);
      setContentView(R.layout.deallist);

      progressDialog = new ProgressDialog(this);
      progressDialog.setMax(2);
      progressDialog.setCancelable(false);
      progressDialog.setMessage(getString(R.string.deal_list_retrieving_data));

      // Obiekt typu Application przechowuje stan caej aplikacji.
      app = (DealDroidApp) getApplication();

      // Tworzenie adaptera z pocztkowo pust kolekcj elementw.
      items = new ArrayList<Item>();
      dealsAdapter = new DealsAdapter(items);

      // Adapter widoku ListView (niniejsza klasa dziedziczy po ListActivity).
      setListAdapter(dealsAdapter);

      // Pobieranie listy kategorii (w razie potrzeby naley
      // te przetworzy dane z kanau RSS).
      if (app.getSectionList().isEmpty()) {
         if (app.connectionPresent()) {
            parseFeedTask = new ParseFeedTask();
            parseFeedTask.execute();
         } else {
            Toast.makeText(this,
                     getString(R.string.deal_list_network_unavailable),
                     Toast.LENGTH_LONG).show();
         }
      } else {
         resetListItems(app.getSectionList().get(0).getItems());
      }

      // Kontrolka Spinner do wybierania kategorii.
      Spinner sectionSpinner = (Spinner) findViewById(R.id.section_spinner);
      spinnerAdapter =
               new ArrayAdapter<Section>(DealList.this,
                        android.R.layout.simple_spinner_item, app
                                 .getSectionList());
      spinnerAdapter
               .setDropDownViewResource(android.R.layout.simple_spinner_dropdown_item);
      sectionSpinner.setAdapter(spinnerAdapter);
      sectionSpinner.setOnItemSelectedListener(new OnItemSelectedListener() {
         public void onItemSelected(AdapterView<?> parentView,
                  View selectedItemView, int position, long id) {
            if (currentSelectedSection != position) {
               currentSelectedSection = position;
               resetListItems(app.getSectionList().get(position).getItems());
            }
         }

         public void onNothingSelected(AdapterView<?> parentView) {
            // Brak operacji.
         }
      });
   }

   public List<Item> getItems() {
      return items;
   }

   private void resetListItems(List<Item> newItems) {
      items.clear();
      items.addAll(newItems);
      dealsAdapter.notifyDataSetChanged();
   }

   @Override
   protected void onListItemClick(ListView listView, View view, int position,
            long id) {
      view.setBackgroundColor(android.R.color.background_light);
      app.setCurrentItem(app.getSectionList().get(currentSelectedSection)
               .getItems().get(position));
      Intent dealDetails = new Intent(DealList.this, DealDetails.class);
      startActivity(dealDetails);
   }

   @Override
   public void onPause() {
      if (progressDialog.isShowing()) {
         progressDialog.dismiss();
      }
      super.onPause();
   }

   @Override
   public boolean onCreateOptionsMenu(Menu menu) {
      menu.add(0, DealList.MENU_REPARSE, 0, R.string.deal_list_reparse_menu);
      menu.add(0, DealList.MENU_EXPORT, 0, R.string.deal_list_export_menu);
      return true;
   }

   @Override
   public boolean onOptionsItemSelected(MenuItem item) {
      switch (item.getItemId()) {
         case MENU_REPARSE:
            if (app.connectionPresent()) {
               new ParseFeedTask().execute();
            } else {
               Toast.makeText(this,
                        getString(R.string.deal_list_network_unavailable),
                        Toast.LENGTH_LONG).show();
            }
            break;
         case MENU_EXPORT:
            try {
               new DealExporter(this, items).export();
               Toast.makeText(this, "Gotowe!", Toast.LENGTH_SHORT).show();
            } catch (IOException e) {
               Toast.makeText(this, "Bd w czasie eksportowania", Toast.LENGTH_SHORT)
                        .show();
            }
            break;
      }
      return false;
   }

   // Uywanie niestandardowego adaptera do kontrolowania ukadu i widokw
   private class DealsAdapter extends ArrayAdapter<Item> {

      public DealsAdapter(List<Item> items) {
         super(DealList.this, R.layout.list_item, items);
      }

      @Override
      public View getView(int position, View convertView, ViewGroup parent) {

         if (convertView == null) {
            LayoutInflater inflater =
                     (LayoutInflater) getSystemService(Context.LAYOUT_INFLATER_SERVICE);
            convertView = inflater.inflate(R.layout.list_item, parent, false);
         }

         // Uywanie klasy ViewHolder, aby zapobiec wielu wywoaniom metody findViewById 
         // (jeli kolekcja jest dua)
         TextView text = (TextView) convertView.findViewById(R.id.deal_title);
         ImageView image = (ImageView) convertView.findViewById(R.id.deal_img);
         image.setImageBitmap(BitmapFactory.decodeResource(getResources(),
                  R.drawable.ddicon));

         Item item = getItem(position);

         if (item != null) {
            text.setText(item.getTitle());
            Bitmap bitmap = app.getImageCache().get(item.getItemId());
            if (bitmap != null) {
               image.setImageBitmap(bitmap);
            } else {
               // Dodawanie identyfikatora do rysunku jako znacznika uywanego w zadaniu
               image.setTag(item.getItemId());
               // Odrbny wtek (dla zadania) do pobierania kadego rysunku.
               // Rozwizanie jest podatne na bdy. Naley zatrzyma wszystkie wtki
               // w metodzie onPause.
               new RetrieveImageTask(image).execute(item.getSmallPicUrl());
            }
         }

         return convertView;
      }
   }

   public ParseFeedTask getParseFeedTask() {
      return parseFeedTask;
   }

   // Uyj klasy AsyncTask<Params, Progress, Result> do atwego wykonywania zada
   // poza wtkiem interfejsu uytkownika
   public class ParseFeedTask extends AsyncTask<Void, Integer, List<Section>> {

      @Override
      protected void onPreExecute() {
         if (progressDialog.isShowing()) {
            progressDialog.dismiss();
         }
      }

      @Override
      protected List<Section> doInBackground(Void... args) {
         publishProgress(1);
         List<Section> sections = app.getParser().parse();
         publishProgress(2);
         return sections;
      }

      @Override
      protected void onProgressUpdate(Integer... progress) {
         int currentProgress = progress[0];
         if ((currentProgress == 1) && !progressDialog.isShowing()) {
            progressDialog.show();
         } else if ((currentProgress == 2) && progressDialog.isShowing()) {
            progressDialog.dismiss();
         }
         progressDialog.setProgress(progress[0]);
      }

      @Override
      protected void onPostExecute(List<Section> taskSectionList) {
         if (!taskSectionList.isEmpty()) {
            app.getSectionList().clear();
            app.getSectionList().addAll(taskSectionList);
            spinnerAdapter.notifyDataSetChanged();

            resetListItems(app.getSectionList().get(0).getItems());
         } else {
            Toast.makeText(DealList.this,
                     getString(R.string.deal_list_missing_data),
                     Toast.LENGTH_LONG).show();
         }
      }

      public List<Section> waitAndUpdate() throws Exception {
         final List<Section> sections = this.get();
         Handler handler = new Handler(Looper.getMainLooper());
         handler.post(new Runnable() {
            public void run() {
               if (!getStatus().equals(Status.FINISHED)) {
                  onPostExecute(sections);
               }
            }
         });
         return sections;
      }
   }

   private class RetrieveImageTask extends AsyncTask<String, Void, Bitmap> {
      private ImageView imageView;

      public RetrieveImageTask(ImageView imageView) {
         this.imageView = imageView;
      }

      @Override
      protected Bitmap doInBackground(String... args) {
         Bitmap bitmap = app.retrieveBitmap(args[0]);
         return bitmap;
      }

      @Override
      protected void onPostExecute(Bitmap bitmap) {
         if (bitmap != null) {
            imageView.setImageBitmap(bitmap);
            app.getImageCache().put((Long) imageView.getTag(), bitmap);
            imageView.setTag(null);
         }
      }
   }
}