package jabagator.android;

// import jabagator.view.MotionMode;
import android.content.Context;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.Path;
import android.graphics.RectF;
import android.util.AttributeSet;
import android.util.Log;
import android.view.MotionEvent;
import android.view.View;

/**
 * Gwny widok z ptnem do rysowania.
 * Kod zaadaptowano na podstawie aplikacji Erica Burke'a, "Smooth Signatures". Opisano go
 * na stronie http://corner.squareup.com/2010/07/smooth-signatures.html. Wedug komentarza
 * CrazyBoba kod dostpny jest na licencji ASL-2 (wedug uytkownika
 * maciel310 jest to licencja Apache 2.0-licensed.").
 */
public class DrawingView extends View {
	
	public enum MotionMode {
	    SELECT_AND_MOVE, DRAW_POLY, DRAW_RECTANGLE, DRAW_OVAL
	}
    
    private static final float STROKE_WIDTH = 2f;

    /** Trzeba to uwzgldni, aby modyfikowany obszar obejmowa grubo kreski **/
    private static final float HALF_STROKE_WIDTH = STROKE_WIDTH / 2;

    private Paint paint = new Paint();
    private Path path = new Path();

    /**
     * Optymalizacja wywietlania przez zmian jak najmniejszego obszaru
     */
    private float lastTouchX;
    private float lastTouchY;
    private final RectF dirtyRect = new RectF();       
    
    final OnTouchListener selectionAndMoveListener = new OnTouchListener() {
        @Override
        public boolean onTouch(View v, MotionEvent event) {
            return false;
        }
    };
    final OnTouchListener drawRectangleListener = new OnTouchListener() {
        @Override
        public boolean onTouch(View v, MotionEvent event) {
            return false;
        }
    };
    final OnTouchListener drawOvalListener = new OnTouchListener() {
        @Override
        public boolean onTouch(View v, MotionEvent event) {
            return false;
        }
    };
    
    final OnTouchListener drawPolyLineListener = new OnTouchListener() {
        
        @Override
        public boolean onTouch(View v, MotionEvent event) {
            // Log.d("jabagator", "onTouch: " + event);
            float eventX = event.getX();
            float eventY = event.getY();

            switch (event.getAction()) {
              case MotionEvent.ACTION_DOWN:
                path.moveTo(eventX, eventY);
                lastTouchX = eventX;
                lastTouchY = eventY;
                // Na razie nie istnieje punkt kocowy, dlatego nie warto
                // marnowa cykli na dezaktualizowanie obszaru
                return true;

              case MotionEvent.ACTION_MOVE:
              case MotionEvent.ACTION_UP:
                // Rozpoczynanie ledzenia zmodyfikowanego obszaru
                resetDirtyRect(eventX, eventY);

                // Gdy sprzt rejestruje zdarzenia szybciej, ni s one przetwarzane,
                // zdarzenia obejmuj list pominitych punktw.
                int historySize = event.getHistorySize();
                for (int i = 0; i < historySize; i++) {
                  float historicalX = event.getHistoricalX(i);
                  float historicalY = event.getHistoricalY(i);
                  expandDirtyRect(historicalX, historicalY);
                  path.lineTo(historicalX, historicalY);
                }

                // Po odtworzeniu pominitych punktw naley je poczy
                path.lineTo(eventX, eventY);
                break;

              default:
                Log.d("jabagator", "Nieznane zdarzenie " + event.toString());
                return false;
            }

            // Uwzgldnienie poowy szerokoci kreski, aby unikn przycicia
            invalidate(
                (int) (dirtyRect.left - HALF_STROKE_WIDTH),
                (int) (dirtyRect.top - HALF_STROKE_WIDTH),
                (int) (dirtyRect.right + HALF_STROKE_WIDTH),
                (int) (dirtyRect.bottom + HALF_STROKE_WIDTH));
            
            lastTouchX = eventX;
            lastTouchY = eventY;

            return true;
        }
        
          /**
           * Wywoywana w trakcie odtwarzania punktw. Gwarantuje, e zmodyfikowany
           * obszar obejmuje wszystkie punkty.
           */
          private void expandDirtyRect(float historicalX, float historicalY) {
            if (historicalX < dirtyRect.left) {
              dirtyRect.left = historicalX;
            } else if (historicalX > dirtyRect.right) {
              dirtyRect.right = historicalX;
            }
            if (historicalY < dirtyRect.top) {
              dirtyRect.top = historicalY;
            } else if (historicalY > dirtyRect.bottom) {
              dirtyRect.bottom = historicalY;
            }
          }

          /**
           * Czyci zmodyfikowany obszar po zarejestrowaniu ruchu.
           */
          private void resetDirtyRect(float eventX, float eventY) {

            // Wartoci lastTouchX i lastTouchY zostay ustawione po wykryciu 
            // zdarzenia ACTION_DOWN.
            dirtyRect.left = Math.min(lastTouchX, eventX);
            dirtyRect.right = Math.max(lastTouchX, eventX);
            dirtyRect.top = Math.min(lastTouchY, eventY);
            dirtyRect.bottom = Math.max(lastTouchY, eventY);
          }
    };

    public DrawingView(Context context, AttributeSet attrs) {
        super(context, attrs);

        paint.setAntiAlias(true);
        paint.setColor(Color.WHITE);
        paint.setStyle(Paint.Style.STROKE);
        paint.setStrokeJoin(Paint.Join.ROUND);
        paint.setStrokeWidth(STROKE_WIDTH);
        
        setMode(MotionMode.DRAW_POLY);
    }
    
    public void clear() {
        path.reset();

        // Ponownie wywietla cay widok
        invalidate();
    }

    @Override
    protected void onDraw(Canvas canvas) {
        canvas.drawPath(path, paint);
    }

    /**
     * Ustawia widok DrawingView na jeden z kilku trybw, np.
     * "wyboru" (do przenoszenia lub zmiany wielkoci obiektw), 
     * "rysowania wieloktw", "rysowania prostokta" itd.
     * Mona j wywoa w konstruktorze, a take z paska narzdzi itd.
     * MotionMode to typ wyliczeniowy z JabaGatora.
     */
    
    private void setMode(MotionMode motionMode) {
        switch(motionMode) {
        case SELECT_AND_MOVE:
            setOnTouchListener(selectionAndMoveListener);
            break;
        case DRAW_POLY:
            setOnTouchListener(drawPolyLineListener);
            break;
        case DRAW_RECTANGLE:
            setOnTouchListener(drawRectangleListener);
            break;
        case DRAW_OVAL:
            setOnTouchListener(drawOvalListener);
            break;
        default:
            throw new IllegalStateException("Nieznany tryb " + motionMode);
        }
    } 
}
