package com.eat.chapter14;

import android.content.AsyncTaskLoader;
import android.content.Context;
import android.os.FileObserver;
import android.util.Log;

import java.io.File;
import java.util.Arrays;
import java.util.List;

/**
 * 1. Obserwacja danych
 * 2. Ładowanie danych w wątku tła podczas zmiany zawartości.
 * 3. Buforowanie danych
 * 4. Dostarczanie danych jedynie po rozpoczęciu.
 * 5. Uwalnianie zasobów podczas resetu.
 * 6. Anulowanie ładowania.
 * 7. Obsługa metody deliverResult
 */
public class FileLoader extends AsyncTaskLoader<List<String>> {
    // Buforowanie listy nazw plików.
    private List<String> mFileNames;


    // 1. Obserwacja danych
    private class SdCardObserver extends FileObserver {

        public SdCardObserver(String path) {
            super(path, FileObserver.CREATE|FileObserver.DELETE);
        }

        @Override
        public void onEvent(int event, String path) {
            // Raportowanie o zachodzącej zmianie zawartości.
            // To wywołanie wymusi nowe asynchroniczne ładowanie danych,
            // jeśli ładowarka jest uruchomiona. W przeciwnym razie dla przyszłych operacji ładowania
			// zostanie zachowana informacja o tym, że dane zostały zmienione.
            onContentChanged();
        }
    }

    private SdCardObserver mSdCardObserver;

    public FileLoader(Context context) {
        super(context);
        String path = context.getFilesDir().getPath();
        mSdCardObserver = new SdCardObserver(path);
    }



    /**
     * Decydowanie czy ładowanie danych powinno być inicjowane czy nie.
     */
    @Override
    protected void onStartLoading() {
        super.onStartLoading();
        mSdCardObserver.startWatching();

        if (mFileNames != null) {
            // Zwracanie pamięci podręcznej
            deliverResult(mFileNames);
        }

        // Wymuszanie ładowania danych, jeśli nie było żadnych poprzednich danych
        // lub jeśli zawartość została oznaczona jako zmieniona wcześniej, ale niedostarczona.
        if (takeContentChanged() || mFileNames == null) {
            forceLoad();
        }
    }

    @Override
    public List<String> loadInBackground() {
        File directory = getContext().getFilesDir();
        return Arrays.asList(directory.list());
    }

    @Override
    public void deliverResult(List<String> data) {
        if (isReset()) {
            return;
        }

        // Buforowanie danych
        mFileNames = data;

        // Dostarczenie rezultatu tylko wtedy, gdy ładowarka jest uruchomiona. 
        if (isStarted()) {
            super.deliverResult(data);
        }
    }

    @Override
    protected void onStopLoading() {
        super.onStopLoading();
        cancelLoad();
    }

    @Override
    protected void onReset() {
        super.onReset();
        mSdCardObserver.stopWatching();
        clearResources();
    }

    private void clearResources() {
       mFileNames = null;
    }
}
