package viewmodels
{
  import com.metaphile.IMetaData;
  import com.metaphile.id3.ID3Data;
  import com.metaphile.id3.ID3Reader;
  
  import flash.display.Bitmap;
  import flash.display.BitmapData;
  import flash.display.Loader;
  import flash.events.Event;
  import flash.events.EventDispatcher;
  import flash.events.TimerEvent;
  import flash.filesystem.File;
  import flash.filesystem.FileMode;
  import flash.filesystem.FileStream;
  import flash.media.Sound;
  import flash.media.SoundChannel;
  import flash.media.SoundTransform;
  import flash.net.URLRequest;
  import flash.utils.Timer;
  
  import models.MusicEntry;
  
  import mx.collections.ArrayCollection;
  import mx.events.PropertyChangeEvent;

  [Event(name="songEnded", type="flash.events.Event")]
  
  [Bindable]
  public class SongViewModel extends EventDispatcher {
    public static const SONG_ENDED:String = "songEnded";
    
    public var albumCover:BitmapData;
    public var albumTitle:String = "";
    public var songTitle:String = "";
    public var artistName:String = "";    
    public var isPlaying:Boolean = false;
    
    private var _volume:Number = 0.5;
    private var _pan:Number = 0.0;
    private var _percentComplete:int = 0;
    
    private var pausePosition:Number = 0;
    
    /** Kolekcja obiektw MusicEntry. */ 
    private var songList:ArrayCollection;
    private var currentIndex:Number = 0;
    
    private var song:Sound;
    private var channel:SoundChannel = new SoundChannel();
    private var timer:Timer;
    
    public function SongViewModel(songList:ArrayCollection, index:Number) {
      this.songList = songList;
      this.currentIndex = index;
      
      timer = new Timer(500, 0);
      timer.addEventListener(TimerEvent.TIMER, onTimer);

      loadCurrentSong();
    }
    
    public function get percentComplete():int {
      return _percentComplete;
    }

    /**
     * Ustawienie tej wartoci powoduje, e pozycja odtwarzania zostaje zaktualizowana.
     */ 
    public function set percentComplete(value:int):void {
      _percentComplete = clipToPercentageBounds(value)
      updateSongPosition();
    }
    
    public function get volume():Number {
      return _volume;
    }
    
    public function set volume(val:Number):void {
      _volume = val;
      updateChannelVolume();
    }
                               
    public function get pan():Number {
      return _pan;
    }
    
    public function set pan(val:Number):void {
      _pan = val;
      updateChannelPan();
    }
 
    /**
     * Skok na pocztek kolejnego utworu na licie. W razie potrzeby
     * powrt do pocztek listy.
     */
    public function nextSong():void {
      incrementCurrentSongIndex();
      loadCurrentSong();
      
      if (isPlaying) {
        pauseSong();
        playSong();
      } else {
        percentComplete = 0;
      }
    }
    
    /**
     * Przesuwa pozycj odtwarznia do pocztku biecego utworu,
     * chyba, e ju jestemy w odlegoci 3 sekund od pocztku, w takim przypadku
     * przeskakuje do pocztku biecego utworu.
     * W razie potrzeby wraca do koca biecego utworu.
     */ 
    public function previousSong():void {
      if (channel && channel.position < 3000) {
        decrementCurrentSongIndex();
        loadCurrentSong();
        
        if (isPlaying) {
          pauseSong();
          playSong();
        }
      } else {
        percentComplete = 0;
      }
    }
    
    /**
     * Przecza stany play/pause utworu.
     */ 
    public function onPlayPause():void {
      if (isPlaying) {
        pausePosition = pauseSong();
      } else {
        playSong(pausePosition);
      }
    }
    
    /**
     * Odtwarza biecy utwr z uwzgldnieniem wartoci przesunicia.
     */ 
    private function playSong(offset:Number = 0):void {
      if (offset == 0) {
        updatePercentComplete(0);
      }
      
      channel = song.play(offset);
      channel.addEventListener(Event.SOUND_COMPLETE, onSoundComplete);
      
      updateChannelVolume();
      updateChannelPan();
      
      timer.start();
      isPlaying = true;
    }
    
    /**
     * Pauzuje biecy utwr.
     * @return Pozycja w utworze w chwili zatrzymania.
     */ 
    private function pauseSong():Number {
      var pausePosition:Number = channel.position;
      
      channel.stop();
      timer.stop();
      
      channel.removeEventListener(Event.SOUND_COMPLETE, onSoundComplete);
      isPlaying = false;
      
      return pausePosition;
    }
    
    /**
     * Wczytuje dane utworw dla pozycji w songList wskazanej
     * za pomoc wartoci currentSongIndex.
     */ 
    private function loadCurrentSong():void {
      try {
        var songFile:MusicEntry = songList[currentIndex];
        
        song = new Sound(new URLRequest(songFile.url));
        
        var id3Reader:ID3Reader = new ID3Reader();
        id3Reader.onMetaData = onMetaData;
        id3Reader.autoLimit = -1;
        id3Reader.autoClose = true;
        
        id3Reader.read(songFile.stream);
      } catch (err:Error) {
        trace("Bd odczytu utworu lub metadanych: "+err.message);
      }
    }
    
    /**
     * Wywoywana po zaadowaniu metadanych utworu z biblioteki Metaphile.
     */ 
    private function onMetaData(metaData:IMetaData):void {
      var songFile:MusicEntry = songList[currentIndex];
      var id3:ID3Data = ID3Data(metaData);
      
      artistName = id3.performer ? id3.performer.text : "Nieznany";
      albumTitle = id3.albumTitle ? id3.albumTitle.text : "Nieznany";
      songTitle = id3.songTitle ? id3.songTitle.text : songFile.name;
      
      if (id3.image) {
        var loader:Loader = new Loader();
        loader.contentLoaderInfo.addEventListener(Event.COMPLETE, 
                                                  onLoadComplete)
        loader.loadBytes(id3.image);
      } else {
        albumCover = null;
      }
    }
    
    /**
     * Wywoywana po zaadowaniu obrazu albumu z metadanych.
     */ 
    private function onLoadComplete(e:Event):void  {
      albumCover = Bitmap(e.target.content).bitmapData
    }
    
    /**
     * Wywoywana po odtworzeniu utworu.
     */ 
    private function onSoundComplete(event:Event):void {
      updatePercentComplete(100);
      dispatchEvent(new Event(SONG_ENDED));
      nextSong();
    }

    /**
     * Inkrementuje biecy indeks utworu. Przechodzi do poczktu songList,
     * jeli inkrementacja spowoduje wyjcie za jej koniec.
     */ 
    private function incrementCurrentSongIndex():void {
      var startIndex:Number = currentIndex;
      
      while (true) {
        if (++currentIndex >= songList.length)
          currentIndex = 0;
        
        if (currentIndex == startIndex || songList[currentIndex].isSong)
          return
      }
    }
    
    /**
     * Dekrementuje biecy indeks utworu. Przechodzi do poczktu songList,
     * jeli dekrementacja spowoduje wyjcie za jej pocztek.
     */ 
    private function decrementCurrentSongIndex():void {
      var startIndex:Number = currentIndex;
      
      while (true) {
        if (--currentIndex < 0)
          currentIndex = songList.length-1;
        
        if (currentIndex == startIndex || songList[currentIndex].isSong)
          return
      }
    }
    
    /**
     * Aktualizuje warto percentComplete przy kadym kroku obiektu timer.
     */
    private function onTimer(event:TimerEvent):void {
      var oldValue:int = _percentComplete;
      
      var percent:Number = channel.position / song.length * 100;
      updatePercentComplete(Math.round(percent));
    }
    
    /**
     * Aktualizuje warto _percentComplete bez wpywu na odtwarzanie
     * biecego utworu (updateSongPosition NIE jest wywoywana). Ta funkcja
     * bdzie wywoywa zdarzenie zmiany waciwoci, aby poinformowa o aktualizacji klientw
     * powizanych z waciwoci percentComplete.
     */ 
    private function updatePercentComplete(value:int):void {
      var oldValue:int = _percentComplete;
      _percentComplete = clipToPercentageBounds(value);
      
      var pce:Event = PropertyChangeEvent.createUpdateEvent(this, 
            "percentComplete", oldValue, _percentComplete);
      dispatchEvent(pce);
    }
    
    /**
     * Przycina warto, aby upewni si, e pozostanie ona w przedziale od 0 do 100 (wcznie).
     */
    private function clipToPercentageBounds(value:int):int {
      return Math.max(0, Math.min(100, value));
    }
    
    /**
     * Ustawia pozycj utworu na podstawie wartoci percentComplete.
     */ 
    private function updateSongPosition():void {
      var newPos:Number = _percentComplete / 100.0 * song.length;
      if (isPlaying) {
        pauseSong()
        playSong(newPos);
      } else {
        pausePosition = newPos;
      }
    }
    
    /**
     * Aktualizuje obiekt SoundTransform odpowiednio do biecej gonoci.
     */ 
    private function updateChannelVolume():void {
      if (channel) {
        var xform:SoundTransform = channel.soundTransform
        xform.volume = _volume;
        channel.soundTransform = xform
      }
    }
    
    /**
     * Aktualizuje obiekt SoundTransform odpowiednio do biecego balansu.
     */ 
    private function updateChannelPan():void {
      if (channel) {
        var xform:SoundTransform = channel.soundTransform
        xform.pan = _pan;
        channel.soundTransform = xform
      }
    }
  }
}