// MainActivity.java
// Klasa wyświetlająca szesnastodniową prognozę pogody dla wybranego miasta.
package com.deitel.weatherviewer;

import android.content.Context;
import android.os.AsyncTask;
import android.os.Bundle;
import android.support.design.widget.FloatingActionButton;
import android.support.design.widget.Snackbar;
import android.support.v7.app.AppCompatActivity;
import android.support.v7.widget.Toolbar;
import android.view.View;
import android.view.inputmethod.InputMethodManager;
import android.widget.EditText;
import android.widget.ListView;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.List;

public class MainActivity extends AppCompatActivity {
   // lista obiektów Weather zawierających prognozę pogody
   private List<Weather> weatherList = new ArrayList<>();

   // adapter ArrayAdapter wiążący obiekty Weather z listą ListView
   private WeatherArrayAdapter weatherArrayAdapter;
   private ListView weatherListView; // wyświetla informacje o pogodzie

   // konfiguruj obiekty Toolbar, ListView i FloatingActionButton
   @Override
   protected void onCreate(Bundle savedInstanceState) {
      super.onCreate(savedInstanceState);
      // automatycznie wygenerowany kod przygotowujący rozkład do wyświetlenia i konfigurujący pasek Toolbar
      setContentView(R.layout.activity_main);
      Toolbar toolbar = (Toolbar) findViewById(R.id.toolbar);
      setSupportActionBar(toolbar);

      // utwórz adapter ArrayAdapter łączący obiekt weatherList z widokiem weatherListView
      weatherListView = (ListView) findViewById(R.id.weatherListView);
      weatherArrayAdapter = new WeatherArrayAdapter(this, weatherList);
      weatherListView.setAdapter(weatherArrayAdapter);

      // skonfiguruj FloatingActionButton służący do ukrywania klawiatury i wysłania żądania do usługi sieciowej
      FloatingActionButton fab =
         (FloatingActionButton) findViewById(R.id.fab);
      fab.setOnClickListener(new View.OnClickListener() {
         @Override
         public void onClick(View view) {
            // odczytaj tekst z obiektu locationEditText i utwórz obiekt URL odwołujący się do usługi sieciowej
            EditText locationEditText =
               (EditText) findViewById(R.id.locationEditText);
            URL url = createURL(locationEditText.getText().toString());

            // ukryj klawiaturę i uruchom zadanie GetWeatherTask pobierające w oddzielnym wątku
            // dane prognozy pogody z serwisu OpenWeatherMap.org
            if (url != null) {
               dismissKeyboard(locationEditText);
               GetWeatherTask getLocalWeatherTask = new GetWeatherTask();
               getLocalWeatherTask.execute(url);
            }
            else {
               Snackbar.make(findViewById(R.id.coordinatorLayout),
                  R.string.invalid_url, Snackbar.LENGTH_LONG).show();
            }
         }
      });
   }

   // ukryj klawiaturę, gdy użytkownik dotknie przycisku FloatingActionButtion
   private void dismissKeyboard(View view) {
      InputMethodManager imm = (InputMethodManager) getSystemService(
         Context.INPUT_METHOD_SERVICE);
      imm.hideSoftInputFromWindow(view.getWindowToken(), 0);
   }

   // skorzystaj z nazwy miasta i utwórz żądanie w postaci adresu URL kierowanego do serwisu openweathermap.org
   private URL createURL(String city) {
      String apiKey = getString(R.string.api_key);
      String baseUrl = getString(R.string.web_service_url);

      try {
         // utwórz adres URL dla wprowadzonego miasta i jednostek metrycznych (stopnie Celsjusza)
         String urlString = baseUrl + URLEncoder.encode(city, "UTF-8") +
            "&units=metric&lang=pl&cnt=16&APPID=" + apiKey;
         return new URL(urlString);
      }
      catch (Exception e) {
         e.printStackTrace();
      }

      return null; // błąd tworzenia adresu URL
   }

   // generuje wywołanie usługi sieciowej REST w celu uzyskania danych prognozy pogody
   // i zapisania ich w lokalnym pliku HTML
   private class GetWeatherTask
      extends AsyncTask<URL, Void, JSONObject> {

      @Override
      protected JSONObject doInBackground(URL... params) {
         HttpURLConnection connection = null;

         try {
            connection = (HttpURLConnection) params[0].openConnection();
            int response = connection.getResponseCode();

            if (response == HttpURLConnection.HTTP_OK) {
               StringBuilder builder = new StringBuilder();

               try (BufferedReader reader = new BufferedReader(
                  new InputStreamReader(connection.getInputStream()))) {

                  String line;

                  while ((line = reader.readLine()) != null) {
                     builder.append(line);
                  }
               }
               catch (IOException e) {
                  Snackbar.make(findViewById(R.id.coordinatorLayout),
                     R.string.read_error, Snackbar.LENGTH_LONG).show();
                  e.printStackTrace();
               }

               return new JSONObject(builder.toString());
            }
            else {
               Snackbar.make(findViewById(R.id.coordinatorLayout),
                  R.string.connect_error, Snackbar.LENGTH_LONG).show();
            }
         }
         catch (Exception e) {
            Snackbar.make(findViewById(R.id.coordinatorLayout),
               R.string.connect_error, Snackbar.LENGTH_LONG).show();
            e.printStackTrace();
         }
         finally {
            connection.disconnect(); // zamknij połączenie HttpURLConnection
         }

         return null;
      }

      // przetwórz odpowiedź w formacie JSON i aktualizuj obiekt ListView
      @Override
      protected void onPostExecute(JSONObject weather) {
         convertJSONtoArrayList(weather); // ponownie wypełnij obiekt weatherList
         weatherArrayAdapter.notifyDataSetChanged(); // odśwież wiązania elementów listy ListView
         weatherListView.smoothScrollToPosition(0); // przewiń do góry
      }
   }

   // utwórz obiekty Weather na podstawie obiektu JSONObject zawierającego prognozę pogody
   private void convertJSONtoArrayList(JSONObject forecast) {
      weatherList.clear(); // clear old weather data

      try {
         // odczytaj „listę” prognozy pogody — JSONArray
         JSONArray list = forecast.getJSONArray("list");

         // zamień każdy element listy na obiekt Weather
         for (int i = 0; i < list.length(); ++i) {
            JSONObject day = list.getJSONObject(i); // odczytaj dane dotyczące jednego dnia

            // odczytaj z obiektu JSONObject dane dotyczące temperatury danego dnia ("temp")
            JSONObject temperatures = day.getJSONObject("temp");

            // odczytaj z obiektu JSONObject opis i ikonę "weather"
            JSONObject weather =
               day.getJSONArray("weather").getJSONObject(0);

            // dodaj nowy obiekt Weather do listy weatherList
            weatherList.add(new Weather(
               day.getLong("dt"), // znacznik czasu i daty
               temperatures.getDouble("min"), // temperatura minimalna
               temperatures.getDouble("max"), // temperatura maksymalna
               day.getDouble("humidity"), // procentowa wilgotność powietrza
               weather.getString("description"), // warunki pogodowe
               weather.getString("icon"))); // nazwa ikony
         }
      }
      catch (JSONException e) {
         e.printStackTrace();
      }
   }
}

/**************************************************************************
 * (C) Copyright 1992-2016 by Deitel & Associates, Inc. and               *
 * Pearson Education, Inc. All Rights Reserved.                           *
 *                                                                        *
 * DISCLAIMER: The authors and publisher of this book have used their     *
 * best efforts in preparing the book. These efforts include the          *
 * development, research, and testing of the theories and programs        *
 * to determine their effectiveness. The authors and publisher make       *
 * no warranty of any kind, expressed or implied, with regard to these    *
 * programs or to the documentation contained in these books. The authors *
 * and publisher shall not be liable in any event for incidental or       *
 * consequential damages in connection with, or arising out of, the       *
 * furnishing, performance, or use of these programs.                     *
 **************************************************************************/
