// MainActivityFragment.java
// Zawiera kod odpowiadający za logikę aplikacji Flag Quiz
package com.deitel.flagquiz;

import java.io.IOException;
import java.io.InputStream;
import java.security.SecureRandom;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Set;

import android.animation.Animator;
import android.animation.AnimatorListenerAdapter;
import android.app.AlertDialog;
import android.app.Dialog;
import android.content.DialogInterface;
import android.content.SharedPreferences;
import android.content.res.AssetManager;
import android.graphics.drawable.Drawable;
import android.os.Bundle;
import android.os.Handler;
import android.support.v4.app.DialogFragment;
import android.support.v4.app.Fragment;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.View;
import android.view.View.OnClickListener;
import android.view.ViewAnimationUtils;
import android.view.ViewGroup;
import android.view.animation.Animation;
import android.view.animation.AnimationUtils;
import android.widget.Button;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.TextView;

public class MainActivityFragment extends Fragment {
   // Łańcuch używany podczas zapisywania komunikatów błędów w dzienniku.
   private static final String TAG = "FlagQuiz Activity";

   private static final int FLAGS_IN_QUIZ = 10;

   private List<String> fileNameList; // nazwy plików flag
   private List<String> quizCountriesList; // kraje bieżącego quizu
   private Set<String> regionsSet; // obszary bieżącego quizu
   private String correctAnswer; // poprawna nazwa kraju przypisana do bieżącej flagi
   private int totalGuesses; // liczba prób odpowiedzi
   private int correctAnswers; // liczba poprawnych odpowiedzi
   private int guessRows; // liczba wierszy przycisków odpowiedzi wyświetlanych na ekranie
   private SecureRandom random; // obiekt używany podczas losowania
   private Handler handler; // zmienna używana podczas opóźniana ładowania kolejnej flagi
   private Animation shakeAnimation; // animacja błędnej odpowiedzi

   private LinearLayout quizLinearLayout; // rozkład zawierający quiz
   private TextView questionNumberTextView; // numer bieżącego pytania
   private ImageView flagImageView; // wyświetla flagę
   private LinearLayout[] guessLinearLayouts; // wiersze przycisków odpowiedzi
   private TextView answerTextView; // wyświetla poprawną odpowiedź

   // konfiguruje MainActivityFragment w chwili tworzenia jego widoku
   @Override
   public View onCreateView(LayoutInflater inflater, ViewGroup container,
      Bundle savedInstanceState) {
      super.onCreateView(inflater, container, savedInstanceState);
      View view =
         inflater.inflate(R.layout.fragment_main, container, false);

      fileNameList = new ArrayList<>();
      quizCountriesList = new ArrayList<>();
      random = new SecureRandom();
      handler = new Handler();

      // ładuje animacje trzęsienia flagąm, która jest odtwarzana po udzieleniu błędnej odpowiedzi
      shakeAnimation = AnimationUtils.loadAnimation(getActivity(),
         R.anim.incorrect_shake);
      shakeAnimation.setRepeatCount(3); // animacja jest powtarzana 3 razy

      // uzyskaj odwołania do komponentów graficznego interfejsu użytkownika
      quizLinearLayout =
         (LinearLayout) view.findViewById(R.id.quizLinearLayout);
      questionNumberTextView =
         (TextView) view.findViewById(R.id.questionNumberTextView);
      flagImageView = (ImageView) view.findViewById(R.id.flagImageView);
      guessLinearLayouts = new LinearLayout[4];
      guessLinearLayouts[0] =
         (LinearLayout) view.findViewById(R.id.row1LinearLayout);
      guessLinearLayouts[1] =
         (LinearLayout) view.findViewById(R.id.row2LinearLayout);
      guessLinearLayouts[2] =
         (LinearLayout) view.findViewById(R.id.row3LinearLayout);
      guessLinearLayouts[3] =
         (LinearLayout) view.findViewById(R.id.row4LinearLayout);
      answerTextView = (TextView) view.findViewById(R.id.answerTextView);

      // konfiguruje obiekty nasłuchujące przycisków odpowiedzi
      for (LinearLayout row : guessLinearLayouts) {
         for (int column = 0; column < row.getChildCount(); column++) {
            Button button = (Button) row.getChildAt(column);
            button.setOnClickListener(guessButtonListener);
         }
      }

      // określa tekst wyświetlany w polach questionNumberTextView
      questionNumberTextView.setText(
         getString(R.string.question, 1, FLAGS_IN_QUIZ));
      return view; // zwróć widok fragmentu do wyświetlenia
   }

   // aktualizuje zmienną guessRows na podstawie wartości SharedPreferences
   public void updateGuessRows(SharedPreferences sharedPreferences) {
      // ustal liczbę przycisków odpowiedzi, które mają zostać wyświetlone
      String choices =
         sharedPreferences.getString(MainActivity.CHOICES, null);
      guessRows = Integer.parseInt(choices) / 2;

      // ukryj wszystkie obiekty LinearLayout przycisków odpowiedzi
      for (LinearLayout layout : guessLinearLayouts)
         layout.setVisibility(View.GONE);

      //  wyświetla właściwe obiekty LinearLayout przycisków odpowiedzi
      for (int row = 0; row < guessRows; row++)
         guessLinearLayouts[row].setVisibility(View.VISIBLE);
   }

   // aktualizuje obszary, które ma obejmować quiz na podstawie wartości SharedPreferences
   public void updateRegions(SharedPreferences sharedPreferences) {
      regionsSet =
         sharedPreferences.getStringSet(MainActivity.REGIONS, null);
   }

   // przygotowuje i uruchamia kolejny quiz
   public void resetQuiz() {
      // AssetManager jest używany do uzyskiwania nazw plików obrazów flag z wybranych obszarów
      AssetManager assets = getActivity().getAssets();
      fileNameList.clear(); // empty list of image file names

      try {
         // pętla przechodzi przez każdy obszar
         for (String region : regionsSet) {
            // uzyskaj listę wszystkich plików z obrazami flag znajdujących się w danym obszarze
            String[] paths = assets.list(region);

            for (String path : paths)
               fileNameList.add(path.replace(".png", ""));
         }
      }
      catch (IOException exception) {
         Log.e(TAG, "Błąd ładowania plików obrazów", exception);
      }

      correctAnswers = 0; // resetuj liczbę poprawnych odpowiedzi
      totalGuesses = 0; // resetuj liczbę wszystkich odpowiedzi udzielonych przez użytkownika
      quizCountriesList.clear(); // wyczyść poprzednią listę krajów

      int flagCounter = 1;
      int numberOfFlags = fileNameList.size();

      // dodaj losowe nazwy plików FLAGS_IN_QUIZ do quizCountriesList
      while (flagCounter <= FLAGS_IN_QUIZ) {
         int randomIndex = random.nextInt(numberOfFlags);

         // dodaj losowe nazwy plików FLAGS_IN_QUIZ do quizCountriesList
         String filename = fileNameList.get(randomIndex);

         // jeżeli obszar jest aktywny, ale nie został jeszcze wybrany
         if (!quizCountriesList.contains(filename)) {
            quizCountriesList.add(filename); // dodaj pliki do listy
            ++flagCounter;
         }
      }

      loadNextFlag(); // uruchom quiz, ładując pierwszą flagę
   }

   // załaduj kolejną flagę po udzieleniu przez użytkownika poprawnej odpowiedzi
   private void loadNextFlag() {
      // ustal nazwę pliku kolejnej flagi i usuń ją z listy
      String nextImage = quizCountriesList.remove(0);
      correctAnswer = nextImage; // aktualizuj poprawną odpowiedź
      answerTextView.setText(""); // wyczyść answerTextView

      // wyświetl numer bieżącego pytania
      questionNumberTextView.setText(getString(
         R.string.question, (correctAnswers + 1), FLAGS_IN_QUIZ));

      // odczytaj informację o obszarze z nazwy kolejnego pliku obrazu
      String region = nextImage.substring(0, nextImage.indexOf('-'));

      // skorzystaj z AssetManager w celu załadowania kolejnego obrazu z folderu assets
      AssetManager assets = getActivity().getAssets();

      // uzyskaj InputStream zasobu kolejnej flagi
      // i spróbuj z niego skorzystać
      try (InputStream stream =
              assets.open(region + "/" + nextImage + ".png")) {
         // ładuje zasób jako obiekt typu Drawable i wyświetla go w polu flagImageView
         Drawable flag = Drawable.createFromStream(stream, nextImage);
         flagImageView.setImageDrawable(flag);

         animate(false); // animuj flagę wprowadzaną na ekran
      }
      catch (IOException exception) {
         Log.e(TAG, "Błąd ładowania " + nextImage, exception);
      }

      Collections.shuffle(fileNameList); // pomieszaj nazwy plików

      // prawidłową odpowiedź umieść na końcu listy fileNameList
      int correct = fileNameList.indexOf(correctAnswer);
      fileNameList.add(fileNameList.remove(correct));

      // dodaj 2, 4, 6 lub 8 przycisków odpowiedzi w zależności od wartości zmiennej guessRows
      for (int row = 0; row < guessRows; row++) {
         // umieść przyciski w currentTableRow
         for (int column = 0;
              column < guessLinearLayouts[row].getChildCount();
              column++) {
            // uzyskaj odwołanie do przycisku w celu jego skonfigurowania
            Button newGuessButton =
               (Button) guessLinearLayouts[row].getChildAt(column);
            newGuessButton.setEnabled(true);

            // ustal nazwę kraju i przekształć ją na tekst wyświetlany w obiekcie newGuessButton
            String filename = fileNameList.get((row * 2) + column);
            newGuessButton.setText(getCountryName(filename));
         }
      }

      // zastąp losowo wybrany przycisk poprawną odpowiedzią
      int row = random.nextInt(guessRows); // pick random row
      int column = random.nextInt(2); // pick random column
      LinearLayout randomRow = guessLinearLayouts[row]; // get the row
      String countryName = getCountryName(correctAnswer);
      ((Button) randomRow.getChildAt(column)).setText(countryName);
   }

   // parsuje nazwę pliku flagi i zwraca nazwę państwa
   private String getCountryName(String name) {
      return name.substring(name.indexOf('-') + 1).replace('_', ' ');
   }

   // animacja wyświetlająca cały rozkład quizLinearLayout na ekranie lub usuwająca go z ekranu
   private void animate(boolean animateOut) {
      // zapobiegaj wyświetleniu animacji podczas umieszczania pierwszej flagi na ekranie
      if (correctAnswers == 0)
         return;

      // oblicz współrzędne x i y środka
      int centerX = (quizLinearLayout.getLeft() +
         quizLinearLayout.getRight()) / 2; // oblicz współrzędną x
      int centerY = (quizLinearLayout.getTop() +
         quizLinearLayout.getBottom()) / 2; // oblicz współrzędne y

      // oblicz promień animacji
      int radius = Math.max(quizLinearLayout.getWidth(),
         quizLinearLayout.getHeight());

      Animator animator;

      // jeżeli rozkład quizLinearLayout ma być umieszczony na ekranie, a nie z niego zdejmowany
      if (animateOut) {
         // utwórz animację okręgu odsłaniającą nowe elementy na ekranie
         animator = ViewAnimationUtils.createCircularReveal(
            quizLinearLayout, centerX, centerY, radius, 0);
         animator.addListener(
            new AnimatorListenerAdapter() {
               // wywołaj, gdy animacja się skończy
               @Override
               public void onAnimationEnd(Animator animation) {
                  loadNextFlag();
               }
            }
         );
      }
      else { // jeżeli rozkład quizLinearLayout ma zostać wyświetlony na ekranie
         animator = ViewAnimationUtils.createCircularReveal(
            quizLinearLayout, centerX, centerY, 0, radius);
      }

      animator.setDuration(500); // określ czas animacji — 500 ms
      animator.start(); // uruchom animację
   }

   // kod wykonywany po dotknięciu przycisku odpowiedzi
   private OnClickListener guessButtonListener = new OnClickListener() {
      @Override
      public void onClick(View v) {
         Button guessButton = ((Button) v);
         String guess = guessButton.getText().toString();
         String answer = getCountryName(correctAnswer);
         ++totalGuesses; // inkrementuj liczbę odpowiedzi udzielonych przez użytkownika

         if (guess.equals(answer)) { // jeżeli odpowiedź jest poprawna
            ++correctAnswers; // inkrementuj liczbę poprawnych odpowiedzi udzielonych przez użytkownika

            // poprawną odpowiedź wyświetl zieloną czcionką
            answerTextView.setText(answer + "!");
            answerTextView.setTextColor(
               getResources().getColor(R.color.correct_answer,
                  getContext().getTheme()));

            disableButtons(); // dezaktywuj wszystkie przyciski odpowiedzi

            // jeżeli użytkownik poprawnie zidentyfikował flagi FLAGS_IN_QUIZ
            if (correctAnswers == FLAGS_IN_QUIZ) {
               // DialogFragment wyświetla status quizu i uruchamia nowy quiz
               DialogFragment quizResults =
                  new DialogFragment() {
                     // utwórz okno AlertDialog i je zwróć
                     @Override
                     public Dialog onCreateDialog(Bundle bundle) {
                        AlertDialog.Builder builder =
                           new AlertDialog.Builder(getActivity());
                        builder.setMessage(
                           getString(R.string.results,
                              totalGuesses,
                              (1000 / (double) totalGuesses)));

                        // przycisk „Resetuj quiz”
                        builder.setPositiveButton(R.string.reset_quiz,
                           new DialogInterface.OnClickListener() {
                              public void onClick(DialogInterface dialog,
                                 int id) {
                                 resetQuiz();
                              }
                           }
                        );

                        return builder.create(); // zwróć AlertDialog
                     }
                  };

               // skorzystaj z FragmentManager w celu wyświetlenia okna DialogFragment
               quizResults.setCancelable(false);
               quizResults.show(getFragmentManager(), "quiz results");
            }
            else { // odpowiedź jest poprawna, ale quiz się jeszcze nie skończył
               // odczekaj 2 sekundy i załaduj kolejną flagę
               handler.postDelayed(
                  new Runnable() {
                     @Override
                     public void run() {
                        animate(true); // animacja usuwa flagę z ekranu
                     }
                  }, 2000); // odczekaj 2 000 milisekund (2 sekundy)
            }
         }
         else { // odpowiedź jest poprawna
            flagImageView.startAnimation(shakeAnimation); // odtwórz animację trzęsącej się flagi

            // wyświetla czerwony napis „Błąd!
            answerTextView.setText(R.string.incorrect_answer);
            answerTextView.setTextColor(getResources().getColor(
               R.color.incorrect_answer, getContext().getTheme()));
            guessButton.setEnabled(false); // dezaktywuj przycisk błędnej odpowiedzi
         }
      }
   };

   // metoda narzędziowa dezaktywująca wszystkie przyciski odpowiedzi
   private void disableButtons() {
      for (int row = 0; row < guessRows; row++) {
         LinearLayout guessRow = guessLinearLayouts[row];
         for (int i = 0; i < guessRow.getChildCount(); i++)
            guessRow.getChildAt(i).setEnabled(false);
      }
   }
}


/*************************************************************************
 * (C) Copyright 1992-2016 by Deitel & Associates, Inc. and               *
 * Pearson Education, Inc. All Rights Reserved.                           *
 *                                                                        *
 * DISCLAIMER: The authors and publisher of this book have used their     *
 * best efforts in preparing the book. These efforts include the          *
 * development, research, and testing of the theories and programs        *
 * to determine their effectiveness. The authors and publisher make       *
 * no warranty of any kind, expressed or implied, with regard to these    *
 * programs or to the documentation contained in these books. The authors *
 * and publisher shall not be liable in any event for incidental or       *
 * consequential damages in connection with, or arising out of, the       *
 * furnishing, performance, or use of these programs.                     *
 *************************************************************************/
