// MainActivityFragment.java
// Fragment, w którym wyświetlana jest klasa DoodleView
package com.deitel.doodlz;

import android.Manifest;
import android.app.AlertDialog;
import android.content.Context;
import android.content.DialogInterface;
import android.content.pm.PackageManager;
import android.hardware.Sensor;
import android.hardware.SensorEvent;
import android.hardware.SensorEventListener;
import android.hardware.SensorManager;
import android.os.Bundle;
import android.support.v4.app.Fragment;
import android.view.LayoutInflater;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.view.View;
import android.view.ViewGroup;

public class MainActivityFragment extends Fragment {
   private DoodleView doodleView; // obsługuje zdarzenia dotyku i rysuje
   private float acceleration;
   private float currentAcceleration;
   private float lastAcceleration;
   private boolean dialogOnScreen = false;

   // wartość używana w celu określenia tego, czy użytkownik wstrząsnął urządzeniem w celu skasowania rysunku
   private static final int ACCELERATION_THRESHOLD = 100000;

   // zmienna używana do identyfikacji żądania użycia zewnętrznego magazynu
   // przez funkcję zapisującą obraz
   private static final int SAVE_IMAGE_PERMISSION_REQUEST_CODE = 1;

   // metoda wywoływana w razie konieczności utworzenia widoku obiektu Fragment
   @Override
   public View onCreateView(LayoutInflater inflater, ViewGroup container,
      Bundle savedInstanceState) {
      super.onCreateView(inflater, container, savedInstanceState);
      View view =
         inflater.inflate(R.layout.fragment_main, container, false);

      setHasOptionsMenu(true); // ten fragment posiada elementy, które mają zostać wyświetlone w menu

      // uzyskaj odwołanie do klasy DoodleView
      doodleView = (DoodleView) view.findViewById(R.id.doodleView);

      // inicjuj wartość przyśpieszenia
      acceleration = 0.00f;
      currentAcceleration = SensorManager.GRAVITY_EARTH;
      lastAcceleration = SensorManager.GRAVITY_EARTH;
      return view;
   }

   // rozpocznij nasłuchiwanie zdarzeń czujnika
   @Override
   public void onResume() {
      super.onResume();
      enableAccelerometerListening(); // nasłuchuj zdarzenia wstrząsu
   }

   // pozwól na nasłuchiwanie zdarzeń przyspieszeniomierza
   private void enableAccelerometerListening() {
      // uzyskaj dostęp do usługi SensorManage
      SensorManager sensorManager =
         (SensorManager) getActivity().getSystemService(
            Context.SENSOR_SERVICE);

      // zarejestruj obiekt nasłuchujący zdarzeń przyspieszeniomierza
      sensorManager.registerListener(sensorEventListener,
         sensorManager.getDefaultSensor(Sensor.TYPE_ACCELEROMETER),
         SensorManager.SENSOR_DELAY_NORMAL);
   }

   // zatrzymaj proces nasłuchiwania zdarzeń przyspieszeniomierza
   @Override
   public void onPause() {
      super.onPause();
      disableAccelerometerListening(); // zatrzymaj proces nasłuchiwania wstrząsów
   }

   // wyłącz nasłuchiwanie zdarzeń przyspieszeniomierza
   private void disableAccelerometerListening() {
      // uzyskaj dostęp do usługi SensorManager
      SensorManager sensorManager =
         (SensorManager) getActivity().getSystemService(
            Context.SENSOR_SERVICE);

      // zatrzymaj proces nasłuchiwania zdarzeń przyspieszeniomierza
      sensorManager.unregisterListener(sensorEventListener,
         sensorManager.getDefaultSensor(Sensor.TYPE_ACCELEROMETER));
   }

   // interfejs obsługujący zdarzenia przyspieszeniomierza
   private final SensorEventListener sensorEventListener =
      new SensorEventListener() {
         // skorzystaj z przyspieszeniomierza w celu określenia tego, czy użytkownik wstrząsnął urządzeniem
         @Override
         public void onSensorChanged(SensorEvent event) {
            // upewnij się, że na ekranie nie są wyświetlane inne okna dialogowe
            if (!dialogOnScreen) {
               // przekaż wartości x, y, i z dla usługi SensorEvent
               float x = event.values[0];
               float y = event.values[1];
               float z = event.values[2];

               // zapisz poprzednią wartość przyspieszenia
               lastAcceleration = currentAcceleration;

               // oblicz bieżącą wartość przyspieszenia
               currentAcceleration = x * x + y * y + z * z;

               // oblicz zmianę przyspieszenia
               acceleration = currentAcceleration *
                  (currentAcceleration - lastAcceleration);

               // jeżeli wartość przyspieszenia przekracza wartość progową…
               if (acceleration > ACCELERATION_THRESHOLD)
                  confirmErase();
            }
         }

         // żądana metoda interfejsu SensorEventListener
         @Override
         public void onAccuracyChanged(Sensor sensor, int accuracy) {}
      };

   // potwierdź chęć usunięcia obrazu
   private void confirmErase() {
      EraseImageDialogFragment fragment = new EraseImageDialogFragment();
      fragment.show(getFragmentManager(), "erase dialog");
   }

   // wyświetla elementy menu fragmentu
   @Override
   public void onCreateOptionsMenu(Menu menu, MenuInflater inflater) {
      super.onCreateOptionsMenu(menu, inflater);
      inflater.inflate(R.menu.doodle_fragment_menu, menu);
   }

   // obsłuż element wybrany przez użytkownika
   @Override
   public boolean onOptionsItemSelected(MenuItem item) {
      // przełącz na podstawie identyfikatora id MenuItem
      switch (item.getItemId()) {
         case R.id.color:
            ColorDialogFragment colorDialog = new ColorDialogFragment();
            colorDialog.show(getFragmentManager(), "color dialog");
            return true; // kasuj zdarzenie menu
         case R.id.line_width:
            LineWidthDialogFragment widthDialog =
               new LineWidthDialogFragment();
            widthDialog.show(getFragmentManager(), "line width dialog");
            return true; // konsumuj zdarzenie menu
         case R.id.delete_drawing:
            confirmErase(); // potwierdź przed skasowaniem obrazu
            return true; // konsumuj zdarzenie menu
         case R.id.save:
            saveImage(); // sprawdź uprawnienia i zapisz obraz
            return true; // konsumuj zdarzenie menu
         case R.id.print:
            doodleView.printImage(); // drukuj obraz
            return true; // konsumuj zdarzenie menu
      }

      return super.onOptionsItemSelected(item);
   }

   // w razie konieczności prosi o wydanie pozwolenia niezbędnego do zapisu obrazu
   // jeżeli pozwolenie zostało już udzielone, to zapisuje obraz
   private void saveImage() {
      // sprawdza, czy aplikacja posiada pozwolenie niezbędne
      // do zapisania obrazu
      if (getContext().checkSelfPermission(
         Manifest.permission.WRITE_EXTERNAL_STORAGE) !=
         PackageManager.PERMISSION_GRANTED) {

         // pokazuje wyjaśnienie potrzeby wydania pozwolenia
         if (shouldShowRequestPermissionRationale(
            Manifest.permission.WRITE_EXTERNAL_STORAGE)) {
            AlertDialog.Builder builder =
               new AlertDialog.Builder(getActivity());

            // określa treść komunikatu wyświetlanego w oknie
            builder.setMessage(R.string.permission_explanation);

            // wyświetl przycisk OK w oknie
            builder.setPositiveButton(android.R.string.ok,
               new DialogInterface.OnClickListener() {
                  @Override
                  public void onClick(DialogInterface dialog, int which) {
                     // zażądaj pozwolenia
                     requestPermissions(new String[]{
                        Manifest.permission.WRITE_EXTERNAL_STORAGE},
                        SAVE_IMAGE_PERMISSION_REQUEST_CODE);
                  }
               }
            );

            // wyświetl okno dialogowe
            builder.create().show();
         }
         else {
            // zażądaj pozwolenia
            requestPermissions(
               new String[]{Manifest.permission.WRITE_EXTERNAL_STORAGE},
               SAVE_IMAGE_PERMISSION_REQUEST_CODE);
         }
      }
      else { // jeżeli aplikacji pozwolono wcześniej na zapis danych w magazynie zewnętrznym
         doodleView.saveImage(); // zapisz obraz
      }
   }

   // metoda wywoływana przez system, gdy użytkownik udzieli lub odmówi
   // pozwolenia na zapis obrazu
   @Override
   public void onRequestPermissionsResult(int requestCode,
      String[] permissions, int[] grantResults) {
      // polecenie switch wykonuje operację w zależności od
      // żądanego pozwolenia
      switch (requestCode) {
         case SAVE_IMAGE_PERMISSION_REQUEST_CODE:
            if (grantResults[0] == PackageManager.PERMISSION_GRANTED)
               doodleView.saveImage(); // zapisz obraz
            return;
      }
   }

   // wraca do DoodleView
   public DoodleView getDoodleView() {
      return doodleView;
   }

   // wskazuje, czy wyświetlane jest okno dialogowe
   public void setDialogOnScreen(boolean visible) {
      dialogOnScreen = visible;
   }
}

/**************************************************************************
 * (C) Copyright 1992-2016 by Deitel & Associates, Inc. and               *
 * Pearson Education, Inc. All Rights Reserved.                           *
 *                                                                        *
 * DISCLAIMER: The authors and publisher of this book have used their     *
 * best efforts in preparing the book. These efforts include the          *
 * development, research, and testing of the theories and programs        *
 * to determine their effectiveness. The authors and publisher make       *
 * no warranty of any kind, expressed or implied, with regard to these    *
 * programs or to the documentation contained in these books. The authors *
 * and publisher shall not be liable in any event for incidental or       *
 * consequential damages in connection with, or arising out of, the       *
 * furnishing, performance, or use of these programs.                     *
 **************************************************************************/
