// CannonView.java
// Wyświetla elementy gry Cannon Game i decyduje o ich zachowaniu.
package com.deitel.cannongame;

import android.app.Activity;
import android.app.AlertDialog;
import android.app.Dialog;
import android.app.DialogFragment;
import android.content.Context;
import android.content.DialogInterface;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.Point;
import android.media.AudioAttributes;
import android.media.SoundPool;
import android.os.Build;
import android.os.Bundle;
import android.util.AttributeSet;
import android.util.Log;
import android.util.SparseIntArray;
import android.view.MotionEvent;
import android.view.SurfaceHolder;
import android.view.SurfaceView;
import android.view.View;

import java.util.ArrayList;
import java.util.Random;

public class CannonView extends SurfaceView
   implements SurfaceHolder.Callback {

   private static final String TAG = "CannonView"; // do logowania błędów

   // stałe rozgrywki
   public static final int MISS_PENALTY = 2; // liczba sekund odejmowana za trafienie w przeszkodę
   public static final int HIT_REWARD = 3; // liczba sekund odejmowana  za trafienie w cel

   // stałe działa
   public static final double CANNON_BASE_RADIUS_PERCENT = 3.0 / 40;
   public static final double CANNON_BARREL_WIDTH_PERCENT = 3.0 / 40;
   public static final double CANNON_BARREL_LENGTH_PERCENT = 1.0 / 10;

   // stałe kuli
   public static final double CANNONBALL_RADIUS_PERCENT = 3.0 / 80;
   public static final double CANNONBALL_SPEED_PERCENT = 3.0 / 2;

   // stałe celów
   public static final double TARGET_WIDTH_PERCENT = 1.0 / 40;
   public static final double TARGET_LENGTH_PERCENT = 3.0 / 20;
   public static final double TARGET_FIRST_X_PERCENT = 3.0 / 5;
   public static final double TARGET_SPACING_PERCENT = 1.0 / 60;
   public static final double TARGET_PIECES = 9;
   public static final double TARGET_MIN_SPEED_PERCENT = 3.0 / 4;
   public static final double TARGET_MAX_SPEED_PERCENT = 6.0 / 4;

   // stałe przeszkody
   public static final double BLOCKER_WIDTH_PERCENT = 1.0 / 40;
   public static final double BLOCKER_LENGTH_PERCENT = 1.0 / 4;
   public static final double BLOCKER_X_PERCENT = 1.0 / 2;
   public static final double BLOCKER_SPEED_PERCENT = 1.0;

   // tekst o rozmiarze 1/18 szerokości ekranu
   public static final double TEXT_SIZE_PERCENT = 1.0 / 18;

   private CannonThread cannonThread; // steruje pętlą gry
   private Activity activity; // do wyświetlenia komunikatu kończącego grę w wątku graficznego interfejsu użytkownika
   private boolean dialogIsDisplayed = false;

   // obiekty gry
   private Cannon cannon;
   private Blocker blocker;
   private ArrayList<Target> targets;

   // zmienne określające wymiary
   private int screenWidth;
   private int screenHeight;

   // zmienne pętli gry i zmienne przeznaczone do śledzenia statystyk gry
   private boolean gameOver; // Czy gra się skończyła?
   private double timeLeft; // pozostały czas gry wyrażony w sekundach
   private int shotsFired; // liczba wykonanych wystrzałów
   private double totalElapsedTime; // wyrażony w sekundach czas, który upłynął od rozpoczęcia gry

   // stałe i zmienne zarządzające dźwiękami
   public static final int TARGET_SOUND_ID = 0;
   public static final int CANNON_SOUND_ID = 1;
   public static final int BLOCKER_SOUND_ID = 2;
   private SoundPool soundPool; // odtwarza efekty dźwiękowe
   private SparseIntArray soundMap; // mapuje identyfikatory ID do obiektu SoundPool

   // zmienne Paint używane do wyświetlania każdego elementu na ekranie
   private Paint textPaint; // zmienna Paint używana do wyświetlenia tekstu
   private Paint backgroundPaint; // zmienna Paint używana do czyszczenia obszaru wyświetlania

   // konstruktor
   public CannonView(Context context, AttributeSet attrs) {
      super(context, attrs); // wywołaj konstruktora klasy nadrzędnej
      activity = (Activity) context; // zachowaj odwołanie do MainActivity

      // zarejestruj obiekt nasłuchujący SurfaceHolder.Callback
      getHolder().addCallback(this);

      // skonfiguruj atrybuty dźwięku pod kątem dźwięków odtwarzanych przez grę
      AudioAttributes.Builder attrBuilder = new AudioAttributes.Builder();
      attrBuilder.setUsage(AudioAttributes.USAGE_GAME);

      // zainicjuj obiekt SoundPool w celu odtwarzania trzech dźwięków aplikacji
      SoundPool.Builder builder = new SoundPool.Builder();
      builder.setMaxStreams(1);
      builder.setAudioAttributes(attrBuilder.build());
      soundPool = builder.build();

      // stwórz obiekt Map zawierający dźwięki i załaduj dźwięki do tego obiektu
      soundMap = new SparseIntArray(3); // tworzy nową tablicę SparseIntArray
      soundMap.put(TARGET_SOUND_ID,
         soundPool.load(context, R.raw.target_hit, 1));
      soundMap.put(CANNON_SOUND_ID,
         soundPool.load(context, R.raw.cannon_fire, 1));
      soundMap.put(BLOCKER_SOUND_ID,
         soundPool.load(context, R.raw.blocker_hit, 1));

      textPaint = new Paint();
      backgroundPaint = new Paint();
      backgroundPaint.setColor(Color.WHITE);
   }

   // metoda wywoływana przy zmianie rozmiaru obiektu SurfaceView,
   // do zmiany takiej dochodzi, gdy jest on dodawany po raz pierwszy do hierarchii obiektów View
   @Override
   protected void onSizeChanged(int w, int h, int oldw, int oldh) {
      super.onSizeChanged(w, h, oldw, oldh);

      screenWidth = w; // zapisz szerokość obiektu CannonView
      screenHeight = h; // zapisz wysokość obiektu CannonView

      // konfiguruj właściwości tekstu
      textPaint.setTextSize((int) (TEXT_SIZE_PERCENT * screenHeight));
      textPaint.setAntiAlias(true); // wygładza tekst
   }

   // ustal szerokość ekranu gry
   public int getScreenWidth() {
      return screenWidth;
   }

   // ustal wysokość ekranu gry
   public int getScreenHeight() {
      return screenHeight;
   }

   // odtwarza dźwięk o identyfikatorze soundId w obiekcie soundMap
   public void playSound(int soundId) {
      soundPool.play(soundMap.get(soundId), 1, 1, 1, 0, 1f);
   }

   // przywraca początkowe wartości wszystkim obiektom widocznym na ekranie i uruchamia nową grę
   public void newGame() {
      // skonstruuj nowe działo
      cannon = new Cannon(this,
         (int) (CANNON_BASE_RADIUS_PERCENT * screenHeight),
         (int) (CANNON_BARREL_LENGTH_PERCENT * screenWidth),
         (int) (CANNON_BARREL_WIDTH_PERCENT * screenHeight));

      Random random = new Random(); // do określania losowych prędkości
      targets = new ArrayList<>(); // utwórz nową listę celów

      // inicjuj współrzędną targetX dla pierwszego celu, licząc od lewej
      int targetX = (int) (TARGET_FIRST_X_PERCENT * screenWidth);

      // oblicz współrzędną y celów
      int targetY = (int) ((0.5 - TARGET_LENGTH_PERCENT / 2) *
         screenHeight);

      // dodaj cele TARGET_PIECES do listy celów
      for (int n = 0; n < TARGET_PIECES; n++) {

         // określ losową prędkość celu
         // znajdującą się w podanym zakresie
         double velocity = screenHeight * (random.nextDouble() *
            (TARGET_MAX_SPEED_PERCENT - TARGET_MIN_SPEED_PERCENT) +
            TARGET_MIN_SPEED_PERCENT);

         // zmieniaj kolory celów — przełączaj pomiędzy kolorem jasnym i ciemnym
         int color =  (n % 2 == 0) ?
            getResources().getColor(R.color.dark,
               getContext().getTheme()) :
            getResources().getColor(R.color.light,
               getContext().getTheme());

         velocity *= -1; // odwróć początkową wartość prędkości kolejnego celu

         // utwórz nowy cel i dodaj go do listy
         targets.add(new Target(this, color, HIT_REWARD, targetX, targetY,
            (int) (TARGET_WIDTH_PERCENT * screenWidth),
            (int) (TARGET_LENGTH_PERCENT * screenHeight),
            (int) velocity));

         // zmień wartość współrzędnej x, przechodząc
         // do kolejnego celu znajdującego się po prawej stronie
         targetX += (TARGET_WIDTH_PERCENT + TARGET_SPACING_PERCENT) *
            screenWidth;
      }

      // utwórz nową przeszkodę
      blocker = new Blocker(this, Color.BLACK, MISS_PENALTY,
         (int) (BLOCKER_X_PERCENT * screenWidth),
         (int) ((0.5 - BLOCKER_LENGTH_PERCENT / 2) * screenHeight),
         (int) (BLOCKER_WIDTH_PERCENT * screenWidth),
         (int) (BLOCKER_LENGTH_PERCENT * screenHeight),
         (float) (BLOCKER_SPEED_PERCENT * screenHeight));

      timeLeft = 10; // rozpocznij odliczanie od 10 sekund

      shotsFired = 0; // ustaw początkową liczbę oddanych strzałów
      totalElapsedTime = 0.0; // wyzeruj wartość określającą czas, jaki upłynął od rozpoczęcia gry

      if (gameOver) { // uruchom nową grę po zakończeniu poprzedniej
         gameOver = false; // gra jeszcze się nie skończyła
         cannonThread = new CannonThread(getHolder()); // utwórz wątek
         cannonThread.start(); // uruchom wątek pętli gry
      }

      hideSystemBars();
   }

   // metoda wywoływana wielokrotnie przez wątek CannonThread w celu aktualizacji położenia elementów gry
   private void updatePositions(double elapsedTimeMS) {
      double interval = elapsedTimeMS / 1000.0; // zamień na sekundy

      // jeżeli kula jest wyświetlana na ekranie, to aktualizuj jej położenie
      if (cannon.getCannonball() != null)
         cannon.getCannonball().update(interval);

      blocker.update(interval); // aktualizuj położenie przeszkody

      for (GameElement target : targets)
         target.update(interval); // aktualizuj położenie celów

      timeLeft -= interval; // odejmuj od czasu, który pozostał

      // jeżeli stoper osiągnął zero
      if (timeLeft <= 0) {
         timeLeft = 0.0;
         gameOver = true; // gra jest zakończona
         cannonThread.setRunning(false); // zakończ wątek
         showGameOverDialog(R.string.lose); // pokaż komunikat informujący o przegranej
      }

      // jeżeli wszystkie cele zostały trafione
      if (targets.isEmpty()) {
         cannonThread.setRunning(false); // zakończ wątek
         showGameOverDialog(R.string.win); // pokaż komunikat informujący o wygranej
         gameOver = true;
      }
   }

   // metoda ustawiająca lufę działa pod właściwym kątem i dokonująca wystrzału
   // jeżeli kula wystrzelona wcześniej nie znajduje się już na ekranie
   public void alignAndFireCannonball(MotionEvent event) {
      // ustal lokalizację dotknięcia widoku
      Point touchPoint = new Point((int) event.getX(),
         (int) event.getY());

      // oblicz odległość dotknięcia od środka ekranu
      // w płaszczyźnie osi y
      double centerMinusY = (screenHeight / 2 - touchPoint.y);

      double angle = 0; // przypisz kątowi wartość 0

      // oblicz kąt pomiędzy lufą a osią x
      angle = Math.atan2(touchPoint.x, centerMinusY);

      // skieruj działo w kierunku punktu dotknięcia ekranu
      cannon.align(angle);

      // wystrzel kulę, jeżeli wystrzelona wcześniej kula nie jest już wyświetlana na ekranie
      if (cannon.getCannonball() == null ||
         !cannon.getCannonball().isOnScreen()) {
         cannon.fireCannonball();
         ++shotsFired;
      }
   }

   // wyświetl okno AlertDialog, gdy gra zostanie zakończona
   private void showGameOverDialog(final int messageId) {
      // obiekt DialogFragment wyświetlający wynik gry i mogący uruchomić kolejną grę
      final DialogFragment gameResult =
         new DialogFragment() {
            // utwórz obiekt AlertDialog i go zwróć
            @Override
            public Dialog onCreateDialog(Bundle bundle) {
               // utwórz okno wyświetlające łańcuch messageId
               AlertDialog.Builder builder =
                  new AlertDialog.Builder(getActivity());
               builder.setTitle(getResources().getString(messageId));

               // wyświetl liczbę oddanych strzałów i całkowity czas gry
               builder.setMessage(getResources().getString(
                  R.string.results_format, shotsFired, totalElapsedTime));
               builder.setPositiveButton(R.string.reset_game,
                  new DialogInterface.OnClickListener() {
                     // metoda wywoływana po wciśnięciu przez użytkownika przycisku Uruchom ponownie
                     @Override
                     public void onClick(DialogInterface dialog,
                        int which) {
                        dialogIsDisplayed = false;
                        newGame(); // przygotuj i uruchom nową grę
                     }
                  }
               );

               return builder.create(); // zwróć obiekt AlertDialog
            }
         };

      // wyświetl obiekt DialogFragment za pomocą menedżera FragmentManager w wątku interfejsu użytkownika
      activity.runOnUiThread(
         new Runnable() {
            public void run() {
               showSystemBars();
               dialogIsDisplayed = true;
               gameResult.setCancelable(false); // okno modalne
               gameResult.show(activity.getFragmentManager(), "results");
            }
         }
      );
   }

   // rysuje grę na danym obiekcie Canvas
   public void drawGameElements(Canvas canvas) {
      // wyczyść tło
      canvas.drawRect(0, 0, canvas.getWidth(), canvas.getHeight(),
         backgroundPaint);

      // wyświetl pozostałe elementy
      canvas.drawText(getResources().getString(
         R.string.time_remaining_format, timeLeft), 50, 100, textPaint);

      cannon.draw(canvas); // rysuj działo

      // narysuj obiekty typu GameElement
      if (cannon.getCannonball() != null &&
         cannon.getCannonball().isOnScreen())
         cannon.getCannonball().draw(canvas);

      blocker.draw(canvas); // narysuj przeszkodę

      // narysuj wszystkie obiekty Target
      for (GameElement target : targets)
         target.draw(canvas);
   }

   // sprawdza, czy nie doszło do zderzenia kuli z przeszkodą lub celami
   // i obsługuje zdarzenie wywołane kolizją
   public void testForCollisions() {
      // usuń wszystkie cele,
      // w które trafiła kula
      if (cannon.getCannonball() != null &&
         cannon.getCannonball().isOnScreen()) {
         for (int n = 0; n < targets.size(); n++) {
            if (cannon.getCannonball().collidesWith(targets.get(n))) {
               targets.get(n).playSound(); // odtwórz dźwięk trafienia w cel

               // do pozostałego czasu gry dodaj czas będący nagrodą za trafienie w cel
               timeLeft += targets.get(n).getHitReward();

               cannon.removeCannonball(); // usuń kulę
               targets.remove(n); // usuń trafiony cel
               --n; // zapewnia sprawdzenia kolizji kuli z nowym celem numer n
               break;
            }
         }
      }
      else { // usuń kulę, jeżeli nie powinno jej być na ekranie
         cannon.removeCannonball();
      }

      // sprawdź, czy kula zderzyła się przeszkodą
      if (cannon.getCannonball() != null &&
         cannon.getCannonball().collidesWith(blocker)) {
         blocker.playSound(); // odtwórz dźwięk trafienia w przeszkodę

         // odwróć kierunek ruchu kuli
         cannon.getCannonball().reverseVelocityX();

         // od ilości pozostałego czasu odejmij czas będący karą za trafienie w przeszkodę
         timeLeft -= blocker.getMissPenalty();
      }
   }

   // metoda zatrzymująca grę wywoływana przez metodę onPause  obiektu CannonGameFragment
   public void stopGame() {
      if (cannonThread != null)
         cannonThread.setRunning(false); // wyślij polecenie zamknięcia wątku
   }

   // metoda zwalniająca zasoby wywoływana przez metodę onDestroy obiektu  CannonGame
   public void releaseResources() {
      soundPool.release(); // zwolnij wszystkie zasoby używane przez obiekt SoundPool
      soundPool = null;
   }

   // metoda wywoływana w wyniku zmiany rozmiaru powierzchni
   @Override
   public void surfaceChanged(SurfaceHolder holder, int format,
      int width, int height) { }

   // metoda wywoływana, gdy powierzchnia jest tworzona po raz pierwszy
   @Override
   public void surfaceCreated(SurfaceHolder holder) {
      if (!dialogIsDisplayed) {
         newGame(); // przygotuj nową grę i uruchom ją
         cannonThread = new CannonThread(holder); // utwórz wątek
         cannonThread.setRunning(true); // uruchom grę
         cannonThread.start(); // uruchom wątek pętli gry
      }
   }

   // metoda wywoływana, gdy powierzchnia jest usuwana
   @Override
   public void surfaceDestroyed(SurfaceHolder holder) {
      // upewnij się, że wątek zostanie poprawnie zakończony
      boolean retry = true;
      cannonThread.setRunning(false); // zakończ wątek cannonThread

      while (retry) {
         try {
            cannonThread.join(); // poczekaj na zakończenie wątku cannonThread
            retry = false;
         }
         catch (InterruptedException e) {
            Log.e(TAG, "Thread interrupted", e);
         }
      }
   }

   // metoda wywoływana po dotknięciu ekranu przez użytkownika w tej aktywności
   @Override
   public boolean onTouchEvent(MotionEvent e) {
      // uzyskaj wartość int opisującą rodzaj czynności, która doprowadziła do tego zdarzenia
      int action = e.getAction();

      // użytkownik dotknął ekranu lub przeciągnął po nim palcem
      if (action == MotionEvent.ACTION_DOWN ||
         action == MotionEvent.ACTION_MOVE) {
         // wystrzel kulę w kierunku punktu dotyku
         alignAndFireCannonball(e);
      }

      return true;
   }

   // Klasa zagnieżdżona wątku sterująca pętlą gry.
   private class CannonThread extends Thread {
      private SurfaceHolder surfaceHolder; // do manipulowania obiektem canvas
      private boolean threadIsRunning = true; // domyślnie uruchamiany

      // inicjalizuje obiekt SurfaceHolder
      public CannonThread(SurfaceHolder holder) {
         surfaceHolder = holder;
         setName("CannonThread");
      }

      // zmienia stan wykonywania
      public void setRunning(boolean running) {
         threadIsRunning = running;
      }

      // steruje pętlą gry
      @Override
      public void run() {
         Canvas canvas = null; // używany do rysowania
         long previousFrameTime = System.currentTimeMillis();

         while (threadIsRunning) {
            try {
               // uzyskaj obiekt Canvas przeznaczony do wyświetlania elementów tylko przez ten wątek
               canvas = surfaceHolder.lockCanvas(null);

               // zablokuj obiekt surfaceHolder w celu rysowania elementów
               synchronized(surfaceHolder) {
                  long currentTime = System.currentTimeMillis();
                  double elapsedTimeMS = currentTime - previousFrameTime;
                  totalElapsedTime += elapsedTimeMS / 1000.0;
                  updatePositions(elapsedTimeMS); // aktualizuj status gry
                  testForCollisions(); // sprawdź, czy nie dochodzi do kolizji z elementem GameElement
                  drawGameElements(canvas); // rysuj, korzystając z obiektu canvas
                  previousFrameTime = currentTime; // aktualizuj czas
               }
            }
            finally {
               // wyświetl zawartość obiektu canvas w widoku CannonView
               // i pozwól innym wątkom na korzystanie z tego obiektu
               if (canvas != null)
                  surfaceHolder.unlockCanvasAndPost(canvas);
            }
         }
      }
   }
   // ukryj paski systemowe i pasek aplikacji
   private void hideSystemBars() {
      if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT)
         setSystemUiVisibility(
            View.SYSTEM_UI_FLAG_LAYOUT_STABLE |
            View.SYSTEM_UI_FLAG_LAYOUT_HIDE_NAVIGATION |
            View.SYSTEM_UI_FLAG_LAYOUT_FULLSCREEN |
            View.SYSTEM_UI_FLAG_HIDE_NAVIGATION |
            View.SYSTEM_UI_FLAG_FULLSCREEN |
            View.SYSTEM_UI_FLAG_IMMERSIVE);
   }

   // pokaż paski systemowe i pasek aplikacji
   private void showSystemBars() {
      if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT)
         setSystemUiVisibility(
            View.SYSTEM_UI_FLAG_LAYOUT_STABLE |
            View.SYSTEM_UI_FLAG_LAYOUT_HIDE_NAVIGATION |
            View.SYSTEM_UI_FLAG_LAYOUT_FULLSCREEN);
   }
}

/*********************************************************************************
 * (C) Copyright 1992-2016 by Deitel & Associates, Inc. and * Pearson Education, *
 * Inc. All Rights Reserved. * * DISCLAIMER: The authors and publisher of this   *
 * book have used their * best efforts in preparing the book. These efforts      *
 * include the * development, research, and testing of the theories and programs *
 * * to determine their effectiveness. The authors and publisher make * no       *
 * warranty of any kind, expressed or implied, with regard to these * programs   *
 * or to the documentation contained in these books. The authors * and publisher *
 * shall not be liable in any event for incidental or * consequential damages in *
 * connection with, or arising out of, the * furnishing, performance, or use of  *
 * these programs.                                                               *
 *********************************************************************************/
