// ContactsFragment.java
// Klasa nadrzędna obiektu Fragment wyświetlającego alfabetyczną listę nazw kontaktów.
package com.deitel.addressbook;

import android.content.Context;
import android.database.Cursor;
import android.net.Uri;
import android.os.Bundle;
import android.support.design.widget.FloatingActionButton;
import android.support.v4.app.Fragment;
import android.support.v4.app.LoaderManager;
import android.support.v4.content.CursorLoader;
import android.support.v4.content.Loader;
import android.support.v7.widget.LinearLayoutManager;
import android.support.v7.widget.RecyclerView;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;

import com.deitel.addressbook.data.DatabaseDescription.Contact;

public class ContactsFragment extends Fragment
   implements LoaderManager.LoaderCallbacks<Cursor> {

   // metoda wywołania zwrotnego implementowana przez aktywność MainActivity
   public interface ContactsFragmentListener {
      // wywoływana w wyniku wybrania kontaktu
      void onContactSelected(Uri contactUri);

      // wywoływana w wyniku wciśnięcia przycisku
      void onAddContact();
   }

   private static final int CONTACTS_LOADER = 0; // identyfikuje obiekt Loader

   // obiekt powiadamiający aktywność MainActivity o wybraniu kontaktu
   private ContactsFragmentListener listener;

   private ContactsAdapter contactsAdapter; // adapter obiektu recyclerView

   // konfiguruje graficzny interfejs użytkownika tego fragmentu
   @Override
   public View onCreateView(
      LayoutInflater inflater, ViewGroup container,
      Bundle savedInstanceState) {
      super.onCreateView(inflater, container, savedInstanceState);
      setHasOptionsMenu(true); // we fragmencie znajdują się elementy menu, które należy wyświetlić

      // przygotuj do wyświetlenia graficzny interfejs użytkownika i uzyskaj odwołanie do widoku RecyclerView
      View view = inflater.inflate(
         R.layout.fragment_contacts, container, false);
      RecyclerView recyclerView =
         (RecyclerView) view.findViewById(R.id.recyclerView);

      // widok recyclerView powinien wyświetlać elementy w formie pionowej listy
      recyclerView.setLayoutManager(
         new LinearLayoutManager(getActivity().getBaseContext()));

      // utwórz adapter widoku recyclerView i obiekt nasłuchujący kliknięć elementów
      contactsAdapter = new ContactsAdapter(
         new ContactsAdapter.ContactClickListener() {
            @Override
            public void onClick(Uri contactUri) {
               listener.onContactSelected(contactUri);
            }
         }
      );
      recyclerView.setAdapter(contactsAdapter); // ustaw adapter

      // dołącz spersonalizowany obiekt ItemDecorator w celu wyświetlenia linii pomiędzy elementami listy
      recyclerView.addItemDecoration(new ItemDivider(getContext()));

      // wydajność aplikacji zostanie zwiększona, jeżeli rozmiar rozkładu widoku RecyclerView nie ulegnie zmianie
      recyclerView.setHasFixedSize(true);

      // uzyskaj dostęp do przycisku FloatingActionButton i skonfiguruj obiekt nasłuchujący dotknięcia go
      FloatingActionButton addButton =
         (FloatingActionButton) view.findViewById(R.id.addButton);
      addButton.setOnClickListener(
         new View.OnClickListener() {
            // wyświetla fragment AddEditFragment pod dotknięciu przycisku FloatingActionButton
            @Override
            public void onClick(View view) {
               listener.onAddContact();
            }
         }
      );

      return view;
   }

   // ustaw obiekt ContactsFragmentListener po dołączeniu fragmentu
   @Override
   public void onAttach(Context context) {
      super.onAttach(context);
      listener = (ContactsFragmentListener) context;
   }

   // usuń obiekt ContactsFragmentListener po odłączeniu fragmentu
   @Override
   public void onDetach() {
      super.onDetach();
      listener = null;
   }

   // inicjuj obiekt Loader po utworzeniu aktywności tego fragmentu
   @Override
   public void onActivityCreated(Bundle savedInstanceState) {
      super.onActivityCreated(savedInstanceState);
      getLoaderManager().initLoader(CONTACTS_LOADER, null, this);
   }

   //  metoda wywoływana przez aktywność MainActivity po zaktualizowaniu bazy danych przez inne fragment
   public void updateContactList() {
      contactsAdapter.notifyDataSetChanged();
   }

   // metoda wywoływana przez menedżera LoaderManager w celu utworzenia obiektu Loader
   @Override
   public Loader<Cursor> onCreateLoader(int id, Bundle args) {
      // utwórz właściwy obiekt CursorLoader na podstawie argumentu będącego identyfikatorem;
      // fragment ten zawiera tylko jeden obiekt Loader, a więc instrukcja switch jest zbędna
      switch (id) {
         case CONTACTS_LOADER:
            return new CursorLoader(getActivity(),
               Contact.CONTENT_URI, // adres Uri tabeli kontaktów
               null, // rzutowanie wartości null zwraca wszystkie kolumny
               null, // wybranie wartości null zwraca wszystkie rzędy
               null, // brak argumentów selekcji
               Contact.COLUMN_NAME + " COLLATE NOCASE ASC"); // kolejność sortowania
         default:
            return null;
      }
   }

   // metoda wywoływana przez menedżera LoaderManager po zakończeniu ładowania
   @Override
   public void onLoadFinished(Loader<Cursor> loader, Cursor data) {
      contactsAdapter.swapCursor(data);
   }

   // metoda wywoływana przez menedżera LoaderManager, gdy obiekt Loader jest wyzerowywany
   @Override
   public void onLoaderReset(Loader<Cursor> loader) {
      contactsAdapter.swapCursor(null);
   }
}


/**************************************************************************
 * (C) Copyright 1992-2016 by Deitel & Associates, Inc. and               *
 * Pearson Education, Inc. All Rights Reserved.                           *
 *                                                                        *
 * DISCLAIMER: The authors and publisher of this book have used their     *
 * best efforts in preparing the book. These efforts include the          *
 * development, research, and testing of the theories and programs        *
 * to determine their effectiveness. The authors and publisher make       *
 * no warranty of any kind, expressed or implied, with regard to these    *
 * programs or to the documentation contained in these books. The authors *
 * and publisher shall not be liable in any event for incidental or       *
 * consequential damages in connection with, or arising out of, the       *
 * furnishing, performance, or use of these programs.                     *
 **************************************************************************/
