#!/usr/bin/perl

# _union_vertex_set
#
#       $G->_union_vertex_set($u, $v)
#
#       (TYLKO DO UZYTKU WEWNETRZNEGO)
#       Dodaje wierzcholki $u i $v grafu $G do tego samego zbioru wierzcholkow.
#
sub _union_vertex_set {
    my ($G, $u, $v) = @_;

    my $su = $G->vertex_set( $u );
    my $sv = $G->vertex_set( $v );
    my $ru = $G->{ VertexSetRank }->{ $su };
    my $rv = $G->{ VertexSetRank }->{ $sv };

    if ( $ru < $rv ) {  # Sumowanie wedlug rang (rownowazenie wag).
        $G->{ VertexSetParent }->{ $su } = $sv;
    } else {
        $G->{ VertexSetParent }->{ $sv } = $su;
        $G->{ VertexSetRank   }->{ $sv }++ if $ru == $rv;
    }
}

# vertex_set
#
#       $s = $G->vertex_set($v)
#
#       Zwraca zbior wierzcholkow $v grafu $G. "Zbior wierzcholkow" 
#       jest reprezentowany przez jego wierzcholek-rodzica.
#
sub vertex_set {
    my ($G, $v) = @_;

    if ( exists  $G->{ VertexSetParent }->{ $v } ) {
        # Kompresja sciezki.
        $G->{ VertexSetParent }->{ $v } =
          $G->vertex_set( $G->{ VertexSetParent }->{ $v } )
            if $v ne $G->{ VertexSetParent }->{ $v };
    } else {
        $G->{ VertexSetParent }->{ $v } = $v;

        $G->{ VertexSetRank   }->{ $v } = 0;
    }

    return $G->{ VertexSetParent }->{ $v };
}

# MST_Kruskal
#
#       $MST = $G->MST_Kruskal;
#
#       Zwraca minimalne drzewo rozpinajace Kruskala (jako graf) 
#       dla grafu $G uwzglednaiajace 'wagi' (weight) krawedzi.
#       (Uzywa metody vertex_set(), a procedura
#       add_edge() wymaga procedury _union_vertex_set().)
#
sub MST_Kruskal {
    my $G   = shift;
    my $MST = (ref $G)->new;
    my @E   = $G->edges;
    my (@W, $u, $v, $w);

    while (($u, $v) = splice(@E, 0, 2)) {
        $w = $G->get_attribute('weight', $u, $v);
        next unless defined $w; # waga undef == nieskonczenie ciezka
        push @W, [ $u, $v, $w ];
    }

    $MST->directed( $G->directed );

    # Sortowanie wedlug wag.
    foreach my $e ( sort { $a->[ 2 ] <=> $b->[ 2 ] } @W ) {
        ($u, $v, $w) = @$e;
        $MST->add_weighted_edge( $u, $w, $v )
            unless $MST->vertex_set( $u ) eq $MST->vertex_set( $v );
    }

    return $MST;
}

# MST_Prim
#
#       $MST = $G->MST_Prim($s)
#
#       Zwraca minimalne drzewo rozpinajace Prima (w postaci grafu)
#       dla grafu $G w oparciu o 'wagi' (weight) jego krawedzi.
#       Opcjonalny wierzcholek startowy to $s; jesli nie zostanie podany inny, 
#       szczesliwie wybrany zostal dobry (z duza liczba odchodzacych krawedzi).
#

sub MST_Prim {
    my ( $G, $s ) = @_;
    my $MST       = (ref $G)->new;

    $u = $G->largest_out_degree( $G->vertices ) unless defined $u;

    use Heap::Fibonacci;
    my $heap = Heap::Fibonacci->new;
    my ( %in_heap, %weight, %parent );

    $G->_heap_init( $heap, $s, \%in_heap, \%weight, \%parent );

    # Przegladamy krawedzie z biezacej pozycji przeszukiwania wszerz 
    # wedlug wag, w porzadku rosnacym.
    while ( defined $heap->minimum ) {
        my $u = $heap->extract_minimum;
        delete $in_heap{ $u->vertex };

        # Teraz rozszerzamy zakres przeszukiwania wszerz.

        foreach my $v ( $G->successors( $u->vertex ) ) {
            if ( defined( $v = $in_heap{ $v } ) ) {
                my $nw = $G->get_attribute( 'weight',
                                            $u->vertex, $v->vertex );
                my $ow = $v->weight;

                if ( not defined $ow or $nw < $ow ) {
                    $v->weight( $nw );
                    $v->parent( $u->vertex );
                    $heap->decrease_key( $v );
                }
            }
        }
    }

    foreach my $v ( $G->vertices ) {
        $MST->add_weighted_edge( $v, $weight{ $v }, $parent{ $v } )
            if defined $parent{ $v };
    }

    return $MST;
}

use Graph;

my $graph = Graph->new;

# add_weighted_path() is defined using add_path()
# and set_attribute('weight', ...).
$graph->add_weighted_path( qw( a 4 b 1 c 2 f 3 i 2 h 1 g 2 d 1 a ) );
$graph->add_weighted_path( qw( a 3 e 6 i ) );
$graph->add_weighted_path( qw( d 1 e 2 f ) );
$graph->add_weighted_path( qw( b 2 e 5 h ) );
$graph->add_weighted_path( qw( e 1 g ) );
$graph->add_weighted_path( qw( b 1 f ) );

my $mst_kruskal = $graph->MST_Kruskal;
my $mst_prim    = $graph->MST_Prim;

print "Minimalne drzewo rozpinajace Kruskala: $mst_kruskal\n";
print "Minimalne drzewo rozpinajace Prima: $mst_prim\n";
