package  com.wrox.algorithms.hashing;

import com.wrox.algorithms.iteration.Iterator;
import com.wrox.algorithms.lists.LinkedList;
import com.wrox.algorithms.lists.List;

public class BucketingHashtable implements Hashtable {
    /** graniczny stopie wypenienia tablicy, po przekroczeniu ktrego 
     *  konieczna jest reorganizacja 
     */
    private final float _loadFactor;

    /** porcje danych */
    private List[] _buckets;

    /**
     * Konstruktor.
     *
     * Parametry: 
     *    - pocztkowa liczba porcji
     *    - graniczny stopie wypenienia tablicy
     */
    public BucketingHashtable(int initialCapacity, float loadFactor) {
        assert initialCapacity > 0 : "pocztkowa liczba porcji musi by dodatnia";
        assert loadFactor > 0 : "graniczny stopie wypenienia musi by dodatni";

        _loadFactor = loadFactor;
        _buckets = new List[initialCapacity];
    }

    public void add(Object value) {
        List bucket = bucketFor(value);

        if (!bucket.contains(value)) {
            bucket.add(value);
            maintainLoad();
        }
    }

    public boolean contains(Object value) {
        List bucket = _buckets[bucketIndexFor(value)];
        return bucket != null && bucket.contains(value);
    }

    public int size() {
        int size = 0;
        for (int i = 0; i < buckets.length; ++i) {
            if (_buckets[i] != null) {
                size += _buckets[i].size;
            }
        }
        return size;
    }
   
    /**
     * Odszukuje porcj zawierajc wskazan warto
     * 
     * Parametr: szukana warto
     * Wynik: porcja zawierajca t warto
     */
    private List bucketFor(Object value) {
        int bucketIndex = bucketIndexFor(value);

        List bucket = _buckets[bucketIndex];
        if (bucket == null) {
            bucket = new LinkedList();
            _buckets[bucketIndex] = bucket;
        }

        return bucket;
    }

    /**
     * Wylicza indeks porcji, do ktrej naley zapisa podan warto
     *
     * Parametr: warto, dla ktrej obliczany jest indeks
     * Wynik: indeks porcji
     */
    private int bucketIndexFor(Object value) {
        assert value != null : "podano pust warto";
        return Math.abs(value.hashCode() % _buckets.length);
    }

    /**
     * Zapewnienie waciwych rozmiarw tablicy 
     */
    private void maintainLoad() {
        if (loadFactorExceeded()) {
            resize();
        }
    }

    /**
     * Sprawdzenie, czy osignito lub przekroczono graniczny stopie wypenienia
     *
     */
    private boolean loadFactorExceeded() {
        return size() > _buckets.length * _loadFactor;
    }

    /**
     * Zwikszenie rozmiarw tablicy po przekroczeniu granicznego stopnia wypenienia
     */
    private void resize() {
        BucketingHashtable copy = 
                new BucketingHashtable(_buckets.length * 2, _loadFactor);

        for (int i = 0; i < _buckets.length; ++i) {
            if (_buckets[i] != null) {
                copy.addAll(_buckets[i].iterator());
            }
        }

        _buckets = copy._buckets;
    }

    /**
     * Dodanie do tablicy wszystkich wartoci danej porcji
     *
     * Parametry: dodawane wartoci
     */
    private void addAll(Iterator values) {
        assert values != null : "nie okrelono wartoci";

        for (values.first(); !values.isDone(); values.next()) {
            add(values.current());
        }
    }
}
