package com.wrox.algorithms.bstrees;

public class Node implements Cloneable {
    /** Warto reprezentowana przez wze */
    private Object _value;

    /** Wze-ojciec lub warto pusta */
    private Node _parent;

    /** Lewy syn ("mniejszy" wze) lub warto pusta */
    private Node _smaller;

    /** Prawy syn ("wikszy" wze) lub warto pusta */
    private Node _larger;

    /**
     * Konstruktor - tworzy wze-li 
     * Parametr: warto reprezentowana przez wze
     */
    public Node(Object value) {
        this(value, null, null);
    }

    /**
     * Konstruktor. Tworzy wze poredni
     * Parametry:
     *   - warto reprezentowana przez wze
     *   - lewy syn
     *   - prawy syn
     */
    public Node(Object value, Node smaller, Node larger) {
        setValue(value);
        setSmaller(smaller);
        setLarger(larger);

        if (smaller != null) {
            smaller.setParent(this);
        }

        if (larger != null) {
            larger.setParent(this);
        }
    }

    /**
     * Pobranie wartoci reprezentowanej przez wze
     */
    public Object getValue() {
        return _value;
    }

    /**
     * Zmiana wartoci reprezentowanej przez wze
     * Parametr - nowa warto
     */
    public void setValue(Object value) {
        assert value != null : "podano pust warto";
        _value = value;
    }

    /**
     * Wze-ojciec lub warto pusta
     */
    public Node getParent() {
        return _parent;
    }

    /**
     * Przypisanie wza-ojca
     * Parametr: wze-ojciec lub warto pusta
     */
    public void setParent(Node parent) {
        _parent = parent;
    }

    /**
     * Lewy syn lub warto pusta
     */
    public Node getSmaller() {
        return _smaller;
    }

    /**
     * Przypisanie lewego syna
     * Parametr: lewy syn lub warto pusta
     */
    public void setSmaller(Node smaller) {
        assert smaller != getLarger() : "lewy syn nie moe by tosamy z prawym";
        _smaller = smaller;
    }

    /**
     * Prawy syn lub warto pusta
     */
    public Node getLarger() {
        return _larger;
    }


   /**
     * Przypisanie prawego syna
     * Parametr: prawy syn lub warto pusta
     */
    public void setLarger(Node larger) {
        assert larger != getSmaller() : "prawy syn nie moe by tosamy z lewym";
        _larger = larger;
    }

    /**
     * Sprawdzenie, czy wze jest lewym synem swego ojca
     *
     */
    public boolean isSmaller() {
        return getParent() != null && this == getParent().getSmaller();
    }


   /**
     * Sprawdzenie, czy wze jest prawym synem swego ojca
     *
     */

    public boolean isLarger() {
        return getParent() != null && this == getParent().getLarger();
    }

    /**
     * Poszukiwanie wza-minimum w drzewie o okrelonym korzeniu
     */
    public Node minimum() {
        Node node = this;

        while (node.getSmaller() != null) {
            node = node.getSmaller();
        }

        return node;
    }

   /**
     * Poszukiwanie wza-maksimum w drzewie o okrelonym korzeniu
     */

    public Node maximum() {
        Node node = this;

        while (node.getLarger() != null) {
            node = node.getLarger();
        }

        return node;
    }

    /**
     * Poszukiwanie nastpnika
     */
    public Node successor() {
        if (getLarger() != null) {
            return getLarger().minimum();
        }

        Node node = this;

        while (node.isLarger()) {
            node = node.getParent();
        }

        return node.getParent();
    }

    /**
     * Poszukiwanie poprzednika
     */


    public Node predecessor() {
        if (getSmaller() != null) {
            return getSmaller().maximum();
        }

        Node node = this;

        while (node.isSmaller()) {
            node = node.getParent();
        }

        return node.getParent();
    }

    /**
     * Liczba wzw drzewa o danym korzeniu
     */
    public int size() {
        return size(this);
    }

    /**
     * Wysoko drzewa o danym korzeniu
     */
    public int height() {
        return height(this) - 1;
    }


    public boolean equals(Object object) {
        if (this == object) {
            return true;
        }

        if (object == null || object.getClass() != getClass()) {
            return false;
        }

        Node other = (Node) object;

        return getValue().equals(other.getValue())
                && equalsSmaller(other.getSmaller())
                && equalsLarger(other.getLarger());
    }

    /**
     * Rekurencyjne obliczanie liczby wzw w drzewie o danym korzeniu
     * Parametr: korze drzewa
     */
    private int size(Node node) {
        if (node == null) {
            return 0;
        }

        return 1 + size(node.getSmaller()) + size(node.getLarger());
    }



    private int hashCode(Node node) {
        if (node == null) {
            return 0;
        }

        return getValue().hashCode() ^ hashCode(node.getSmaller()) ^ hashCode(node.getLarger());
    }







    private boolean equalsSmaller(Node other) {
        return getSmaller() == null 
                    && other == null || getSmaller() != null 
                    && getSmaller().equals(other);
    }

    private boolean equalsLarger(Node other) {
        return getLarger() == null 
                    && other == null || getLarger() != null 
                    && getLarger().equals(other);
    }
}
