package com.wrox.algorithms.bstrees;

import com.wrox.algorithms.sorting.Comparator;

/**
 * Binarne drzewo wyszukiwawcze bez mechanizmw rwnowaenia
 *
 */
public class BinarySearchTree {
    /** Komparator wyznaczajcy porzdek wartoci reprezentowanych przez wzy */
    private final Comparator _comparator;

    /** Wskazanie na korze lub warto pusta dla pustego drzewa */
    private Node _root;

    /**
     * Konstruktor
     * Parametr: Komparator wyznaczajcy porzdek wartoci
     */
    public BinarySearchTree(Comparator comparator) {
        assert comparator != null : "nie okrelono komparatora";
        _comparator = comparator;
    }

    /**
     * Poszukiwanie wartoci w drzewie
     * Parametr: szukana warto
     * Wynik: wze reprezentujcy warto lub warto pusta
     */
    public Node search(Object value) {
        assert value != null : "nie podano wartoci";

        Node node = _root;

        while (node != null) {
            int cmp = _comparator.compare(value, node.getValue());
            if (cmp == 0) {
               break;
            }

            node = cmp < 0 ? node.getSmaller() : node.getLarger();
        }

        return node;
    }

    /**
     * Wstawianie wza do drzewa
     *
     * Parametr: warto do wstawienia
     * Wynik: wze reprezentujcy warto
     */
    public Node insert(Object value) {
        Node parent = null;
        Node node = _root;
        int cmp = 0;

        while (node != null) {
            parent = node;
            cmp = _comparator.compare(value, node.getValue());
            node = cmp <= 0 ? node.getSmaller() : node.getLarger();
        }

        Node inserted = new Node(value);
        inserted.setParent(parent);

        if (parent == null) {
            _root = inserted;
        } else if (cmp < 0) {
            parent.setSmaller(inserted);
        } else {
            parent.setLarger(inserted);
        }

        return inserted;
    }

    /**
     * Usuwanie wartoci z drzewa
     * Parametr: warto do usunicia
     * Wynik: usunity wze lub warto pusta
     */
    public Node delete(Object value) {
        Node node = search(value);
        if (node == null) {
            return null;
        }

        Node deleted = 
            node.getSmaller() != null && node.getLarger() != null 
                              ? node.successor() : node;
        assert deleted != null : "podano pust warto";

        Node replacement = 
            deleted.getSmaller() != null ? deleted.getSmaller() : deleted.getLarger();
        if (replacement != null) {
            replacement.setParent(deleted.getParent());
        }

        if (deleted == _root) {
            _root = replacement;
        } else if (deleted.isSmaller()) {
            deleted.getParent().setSmaller(replacement);
        } else {
            deleted.getParent().setLarger(replacement);
        }

        if (deleted != node) {
            Object deletedValue = node.getValue();
            node.setValue(deleted.getValue());
            deleted.setValue(deletedValue);
        }

        return deleted;
    }

    /**
     * Zwraca korze drzewa
     * Wynik: korze drzewa lub warto pusta
     */
    public Node getRoot() {
        return _root;
    }
}
