/*****************************************************************************
*                                                                            *
*  -------------------------------- lz77.c --------------------------------  *
*                                                                            *
*****************************************************************************/

#include <netinet/in.h>
#include <stdlib.h>
#include <string.h>

#include "bit.h"
#include "compress.h"

/*****************************************************************************
*                                                                            *
*  ------------------------------ compare_win -----------------------------  *
*                                                                            *
*****************************************************************************/

static int compare_win(const unsigned char *window, const unsigned char
   *buffer, int *offset, unsigned char *next) {

int                match,
                   longest,
                   i,
                   j,
                   k;

/*****************************************************************************
*                                                                            *
*  Inicjalizacja pooenia, cho jest ono poprawne tylko przy dopasowaniu.   *
*                                                                            *
*****************************************************************************/

*offset = 0;

/*****************************************************************************
*                                                                            *
*  Jeli nie znaleziono dopasowanie, przygotowanie do zwrcenia 0            *
*  i nastpnego symbolu z bufora z podgldem.                                *
*                                                                            *
*****************************************************************************/

longest = 0;
*next = buffer[0];

/*****************************************************************************
*                                                                            *
*  Odszukanie najlepszego dopasowania w buforze i oknie przesuwnym.          *
*                                                                            *
*****************************************************************************/

for (k = 0; k < LZ77_WINDOW_SIZE; k++) {

   i = k;
   j = 0;
   match = 0;

   /**************************************************************************
   *                                                                         *
   *  Sprawdzenie, ile symboli pasuje do pozycji k w oknie przesuwnym.       *
   *                                                                         *
   **************************************************************************/

   while (i < LZ77_WINDOW_SIZE && j < LZ77_BUFFER_SIZE - 1) {

      if (window[i] != buffer[j])
         break;

      match++;
      i++;
      j++;

   }

   /**************************************************************************
   *                                                                         *
   *  Pamitamy pozycj, dugo i nastpny symbol najlepszego dopasowania.  *
   *                                                                         *
   **************************************************************************/

   if (match > longest) {

      *offset = k;
      longest = match;
      *next = buffer[j];

   }

}

return longest;

}

/*****************************************************************************
*                                                                            *
*  ----------------------------- lz77_compress ----------------------------  *
*                                                                            *
*****************************************************************************/

int lz77_compress(const unsigned char *original, unsigned char **compressed,
   int size) {

unsigned long      token;

unsigned char      window[LZ77_WINDOW_SIZE],
                   buffer[LZ77_BUFFER_SIZE],
                   *comp,
                   *temp,
                   next;

int                offset,
                   length,
                   remaining,
                   tbits,
                   hsize,
                   ipos,
                   opos,
                   tpos,
                   i;

/*****************************************************************************
*                                                                            *
*  Sprawiamy, aby wskanik na skompresowane dane nie by na razie dostpny.  *
*                                                                            *
*****************************************************************************/

*compressed = NULL;

/*****************************************************************************
*                                                                            *
*  Zapisujemy informacje nagwkowe.                                         *
*                                                                            *
*****************************************************************************/

hsize = sizeof(int);

if ((comp = (unsigned char *)malloc(hsize)) == NULL)
   return -1;

memcpy(comp, &size, sizeof(int));

/*****************************************************************************
*                                                                            *
*  Inicjalizujemy okno przesuwne i bufor z podgldem.                        *
*                                                                            *
*****************************************************************************/

memset(window, 0, LZ77_WINDOW_SIZE);
memset(buffer, 0, LZ77_BUFFER_SIZE);

/*****************************************************************************
*                                                                            *
*  adujemy bufor z podgldem.                                               *
*                                                                            *
*****************************************************************************/

ipos = 0;

for (i = 0; i < LZ77_BUFFER_SIZE && ipos < size; i++) {

   buffer[i] = original[ipos];
   ipos++;

}

/*****************************************************************************
*                                                                            *
*  Kompresujemy dane.                                                        *
*                                                                            *
*****************************************************************************/

opos = hsize * 8;
remaining = size;

while (remaining > 0) {

   if ((length = compare_win(window, buffer, &offset, &next)) != 0) {

      /***********************************************************************
      *                                                                      *
      *  Kodujemy haso zdania.                                              *
      *                                                                      *
      ***********************************************************************/

      token = 0x00000001 << (LZ77_PHRASE_BITS - 1);

      /***********************************************************************
      *                                                                      *
      *  Ustawiamy pozycj, dla ktrej znaleziono dopasowanie w oknie        *
      *  przesuwnym.                                                         *
      *                                                                      *
      ***********************************************************************/

      token = token | (offset << (LZ77_PHRASE_BITS - LZ77_TYPE_BITS -
         LZ77_WINOFF_BITS));

      /***********************************************************************
      *                                                                      *
      *  Ustawiamy dugo dopasowania.                                      *
      *                                                                      *
      ***********************************************************************/

      token = token | (length << (LZ77_PHRASE_BITS - LZ77_TYPE_BITS -
         LZ77_WINOFF_BITS - LZ77_BUFLEN_BITS));

      /***********************************************************************
      *                                                                      *
      *  Ustawiamy nastpny symbol w buforze z podglem za dopasowaniem.     *
      *                                                                      *
      ***********************************************************************/

      token = token | next;

      /***********************************************************************
      *                                                                      *
      *  Ustawiamy liczb bitw w hale.                                     *
      *                                                                      *
      ***********************************************************************/

      tbits = LZ77_PHRASE_BITS;
      
      }

   else {

      /***********************************************************************
      *                                                                      *
      *  Kodujemy haso symbolu.                                             *
      *                                                                      *
      ***********************************************************************/

      token = 0x00000000;

      /***********************************************************************
      *                                                                      *
      *  Ustawiamy niedopasowany symbol.                                     *
      *                                                                      *
      ***********************************************************************/

      token = token | next;

      /***********************************************************************
      *                                                                      *
      *  Ustawiamy liczb bitw w hale.                                     *
      *                                                                      *
      ***********************************************************************/

      tbits = LZ77_SYMBOL_BITS;

   }

   /**************************************************************************
   *                                                                         *
   *  Zapewniamy, e haso zostanie zapisane od najstarszych bitw.          *
   *                                                                         *
   **************************************************************************/

   token = htonl(token);

   /**************************************************************************
   *                                                                         *
   *  Zapisujemy haso w buforze ze skompresowanymi danymi.                  *
   *                                                                         *
   **************************************************************************/

   for (i = 0; i < tbits; i++) {

      if (opos % 8 == 0) {

         /********************************************************************
         *                                                                   *
         *  Alokujemy kolejny bajt na skompresowane dane.                    *
         *                                                                   *
         ********************************************************************/

         if ((temp = (unsigned char *)realloc(comp,(opos / 8) + 1)) == NULL) {

            free(comp);
            return -1;

         }

         comp = temp;

      }

      tpos = (sizeof(unsigned long) * 8) - tbits + i;
      bit_set(comp, opos, bit_get((unsigned char *)&token, tpos));
      opos++;

   }

   /**************************************************************************
   *                                                                         *
   *  Korygujemy dugo zdania, aby uwzgldni niedopasowany symbol.        *
   *                                                                         *
   **************************************************************************/

   length++;
      
   /**************************************************************************
   *                                                                         *
   *  Kopiujemy dane z bufora z podgldem do okna przesuwnego.               *
   *                                                                         *
   **************************************************************************/

   memmove(&window[0], &window[length], LZ77_WINDOW_SIZE - length);
   memmove(&window[LZ77_WINDOW_SIZE - length], &buffer[0], length);

   /**************************************************************************
   *                                                                         *
   *  Odczytujemy kolejne dane do bufora z podgldem.                        *
   *                                                                         *
   **************************************************************************/

   memmove(&buffer[0], &buffer[length], LZ77_BUFFER_SIZE - length);

   for (i = LZ77_BUFFER_SIZE - length; i<LZ77_BUFFER_SIZE && ipos<size; i++) {

      buffer[i] = original[ipos];
      ipos++;

   }

   /**************************************************************************
   *                                                                         *
   *  Korygujemy liczb pozostaych symboli o dugo zdania.                *
   *                                                                         *
   **************************************************************************/

   remaining = remaining - length;

}

/*****************************************************************************
*                                                                            *
*  Wskazujemy bufor ze skompresowanymi danymi.                               *
*                                                                            *
*****************************************************************************/

*compressed = comp;

/*****************************************************************************
*                                                                            *
*  Zwracamy liczb bajtw w skompresowanych danych.                          *
*                                                                            *
*****************************************************************************/

return ((opos - 1) / 8) + 1;

}

/*****************************************************************************
*                                                                            *
*  ---------------------------- lz77_uncompress ---------------------------  *
*                                                                            *
*****************************************************************************/

int lz77_uncompress(const unsigned char *compressed, unsigned char
   **original) {

unsigned char      window[LZ77_WINDOW_SIZE],
                   buffer[LZ77_BUFFER_SIZE],
                   *orig,
                   *temp,
                   next;

int                offset,
                   length,
                   remaining,
                   hsize,
                   size,
                   ipos,
                   opos,
                   tpos,
                   state,
                   i;

/*****************************************************************************
*                                                                            *
*  Wskanik na dane pierwotne ma by na razie niedostpny.                   *
*                                                                            *
*****************************************************************************/

*original = orig = NULL;

/*****************************************************************************
*                                                                            *
*  Pobieramy informacje nagwkowe.                                          *
*                                                                            *
*****************************************************************************/

hsize = sizeof(int);
memcpy(&size, compressed, sizeof(int));

/*****************************************************************************
*                                                                            *
*  Inicjalizujemy okno przesuwne i bufor z podgldem.                        *
*                                                                            *
*****************************************************************************/

memset(window, 0, LZ77_WINDOW_SIZE);
memset(buffer, 0, LZ77_BUFFER_SIZE);

/*****************************************************************************
*                                                                            *
*  Dekompresujemy dane.                                                      *
*                                                                            *
*****************************************************************************/

ipos = hsize * 8;
opos = 0;
remaining = size;

while (remaining > 0) {

   /**************************************************************************
   *                                                                         *
   *  Pobieramy nastpny bit ze skompresowanych danych.                      *
   *                                                                         *
   **************************************************************************/

   state = bit_get(compressed, ipos);
   ipos++;

   if (state == 1) {

      /***********************************************************************
      *                                                                      *
      *  Obsuga przetwarzania hasa zdania.                                 *
      *                                                                      *
      ***********************************************************************/

      memset(&offset, 0, sizeof(int));

      for (i = 0; i < LZ77_WINOFF_BITS; i++) {

         tpos = (sizeof(int) * 8) - LZ77_WINOFF_BITS + i;
         bit_set((unsigned char *)&offset, tpos, bit_get(compressed, ipos));
         ipos++;

      }

      memset(&length, 0, sizeof(int));

      for (i = 0; i < LZ77_BUFLEN_BITS; i++) {

         tpos = (sizeof(int) * 8) - LZ77_BUFLEN_BITS + i;
         bit_set((unsigned char *)&length, tpos, bit_get(compressed, ipos));
         ipos++;

      }

      next = 0x00;

      for (i = 0; i < LZ77_NEXT_BITS; i++) {

         tpos = (sizeof(unsigned char) * 8) - LZ77_NEXT_BITS + i;
         bit_set((unsigned char *)&next, tpos, bit_get(compressed, ipos));
         ipos++;

      }

      /***********************************************************************
      *                                                                      *
      *  Zapewniamy, e pozycja i dlugo maj odpowiedni kolejno bitw.  *
      *                                                                      *
      ***********************************************************************/

      offset = ntohl(offset);
      length = ntohl(length);

      /***********************************************************************
      *                                                                      *
      *  Zapis zdania z okna do bufora z danymi oryginalnymi.                *
      *                                                                      *
      ***********************************************************************/

      i = 0;

      if (opos > 0) {

         if ((temp = (unsigned char *)realloc(orig, opos+length+1)) == NULL) {

            free(orig);
            return -1;

         }

         orig = temp;

         }

      else {

         if ((orig = (unsigned char *)malloc(length + 1)) == NULL)
            return -1;

      }

      while (i < length && remaining > 0) {

         orig[opos] = window[offset + i];
         opos++;

         /********************************************************************
         *                                                                   *
         *  Zapis kadego symbolu w buforze z podgldem, pki nie bdziemy   *
         *  gotowi do aktualizacji okna przesuwnego.                         *
         *                                                                   *
         ********************************************************************/

         buffer[i] = window[offset + i];
         i++;

         /********************************************************************
         *                                                                   *
         *  Koreygujemy liczb pozostaych symboli, aby uwzgldni wszystkie *
         *  uyte dotd symbole.                                             *
         *                                                                   *
         ********************************************************************/

         remaining--;

      }

      /***********************************************************************
      *                                                                      *
      *  Zapis niedopasowanego symbolu do bufora z oryginalnymi danymi.      *
      *                                                                      *
      ***********************************************************************/

      if (remaining > 0) {

         orig[opos] = next;
         opos++;

         /********************************************************************
         *                                                                   *
         *  Zapisujemy ten symbol w buforze z podgldem.                     *
         *                                                                   *
         ********************************************************************/

         buffer[i] = next;

         /********************************************************************
         *                                                                   *
         *  Korygujemy liczb pozostaych symboli o niedopasowany symbol.    *
         *                                                                   *
         ********************************************************************/

         remaining--;

      }

      /***********************************************************************
      *                                                                      *
      *  Korygujemy dugo zdania, aby uwzgldni niedopasowany symbol.     *
      *                                                                      *
      ***********************************************************************/

      length++;

      }

   else {

      /***********************************************************************
      *                                                                      *
      *  Obsuga przetwarzania hasa symbolu.                                *
      *                                                                      *
      ***********************************************************************/

      next = 0x00;

      for (i = 0; i < LZ77_NEXT_BITS; i++) {

         tpos = (sizeof(unsigned char) * 8) - LZ77_NEXT_BITS + i;
         bit_set((unsigned char *)&next, tpos, bit_get(compressed, ipos));
         ipos++;

      }

      /***********************************************************************
      *                                                                      *
      *  Zapis symbolu do bufora z oryginalnymi danymi.                      *
      *                                                                      *
      ***********************************************************************/

      if (opos > 0) {

         if ((temp = (unsigned char *)realloc(orig, opos + 1)) == NULL) {

            free(orig);
            return -1;

         }

         orig = temp;

         }

      else {

         if ((orig = (unsigned char *)malloc(1)) == NULL)
            return -1;

      }

      orig[opos] = next;
      opos++;

      /***********************************************************************
      *                                                                      *
      *  Zapis symbolu w w buforze z podgldem, pki nie bdziemy gotowi     *
      *  do aktualizacji okna przesuwnego.                                   *
      *                                                                      *
      ***********************************************************************/

      if (remaining > 0)
         buffer[0] = next;

      /***********************************************************************
      *                                                                      *
      *  Korygujemy sum symboli pozostaych, aby uwzgldnie symbol          *
      *  niedopasowany.                                                      *
      *                                                                      *
      ***********************************************************************/

      remaining--;

      /***********************************************************************
      *                                                                      *
      *  Ustawiamy dugo zdania, aby uwzgldni niedopasowany symbol.      *
      *                                                                      *
      ***********************************************************************/

      length = 1;

   }

   /**************************************************************************
   *                                                                         *
   *  Kopiujemy bufor z podgldem do okna przesuwnego.                       *
   *                                                                         *
   **************************************************************************/

   memmove(&window[0], &window[length], LZ77_WINDOW_SIZE - length);
   memmove(&window[LZ77_WINDOW_SIZE - length], &buffer[0], length);

}

/*****************************************************************************
*                                                                            *
*  Wskazujemy bufor z danymi oryginalnymi.                                   *
*                                                                            *
*****************************************************************************/

*original = orig;

/*****************************************************************************
*                                                                            *
*  Zwracamy liczb bajtw w danych oryginalnych.                             *
*                                                                            *
*****************************************************************************/

return opos;

}
