/*************************************************************************
 *  Kompilacja:  javac AcyclicLP.java
 *  Wykonanie:    java AcyclicP V E
 *  Zalenoci: EdgeWeightedDigraph.java DirectedEdge.java Topological.java
 *  Pliki z danymi:   http://algs4.cs.princeton.edu/44sp/tinyEWDAG.txt
 *  
 *  Wyznacza najdusze cieki w waonym digrafie acyklicznym.
 *
 *  Uwaga: warto doda sprawdzanie przed uruchomieniem, czy graf jest grafem DAG
 *
 *  % java AcyclicLP tinyEWDAG.txt 5
 *  5 do 0 (2.44)  5->1  0,32   1->3  0,29   3->6  0,52   6->4  0,93   4->0  0,38   
 *  5 do 1 (0.32)  5->1  0,32   
 *  5 do 2 (2.77)  5->1  0,32   1->3  0,29   3->6  0,52   6->4  0,93   4->7  0,37   7->2  0,34   
 *  5 do 3 (0.61)  5->1  0,32   1->3  0,29   
 *  5 do 4 (2.06)  5->1  0,32   1->3  0,29   3->6  0,52   6->4  0,93   
 *  5 do 5 (0.00)  
 *  5 do 6 (1.13)  5->1  0,32   1->3  0,29   3->6  0,52   
 *  5 do 7 (2.43)  5->1  0,32   1->3  0,29   3->6  0,52   6->4  0,93   4->7  0,37   
 *
 *************************************************************************/

public class AcyclicLP {
    private double[] distTo;          // distTo[v] = dugo najduszej cieki s->v
    private DirectedEdge[] edgeTo;    // edgeTo[v] = ostatnia krawd najduszej cieki s->v

    public AcyclicLP(EdgeWeightedDigraph G, int s) {
        distTo = new double[G.V()];
        edgeTo = new DirectedEdge[G.V()];
        for (int v = 0; v < G.V(); v++) distTo[v] = Double.NEGATIVE_INFINITY;
        distTo[s] = 0.0;

        // Relaksacja wierzchokw w porzdku topologicznym.
        Topological topological = new Topological(G);
        for (int v : topological.order()) {
            for (DirectedEdge e : G.adj(v))
                relax(e);
        }
    }

    // Relaksacja krawdzi e, naley jednak zaktualizowa wartoci po znalezieniu *duszej* cieki.
    private void relax(DirectedEdge e) {
        int v = e.from(), w = e.to();
        if (distTo[w] < distTo[v] + e.weight()) {
            distTo[w] = distTo[v] + e.weight();
            edgeTo[w] = e;
        }       
    }

    // Zwraca dugo najduszej cieki z s do v (lub -nieskoczono, jeli taka nie istnieje).
    public double distTo(int v) {
        return distTo[v];
    }

    //  Czy istnieje cieka z s do v?
    public boolean hasPathTo(int v) {
        return distTo[v] > Double.NEGATIVE_INFINITY;
    }

    // Zwraca elementy najduszej cieki z s do v (lub null, jeli taka cieka nie istnieje).
    public Iterable<DirectedEdge> pathTo(int v) {
        if (!hasPathTo(v)) return null;
        Stack<DirectedEdge> path = new Stack<DirectedEdge>();
        for (DirectedEdge e = edgeTo[v]; e != null; e = edgeTo[e.from()]) {
            path.push(e);
        }
        return path;
    }



public static void main(String[] args) {
        In in = new In(args[0]);
        int s = Integer.parseInt(args[1]);
        EdgeWeightedDigraph G = new EdgeWeightedDigraph(in);

        AcyclicLP lp = new AcyclicLP(G, s);

        for (int v = 0; v < G.V(); v++) {
            if (lp.hasPathTo(v)) {
                StdOut.printf("%d do %d (%.2f)  ", s, v, lp.distTo(v));
                for (DirectedEdge e : lp.pathTo(v)) {
                    StdOut.print(e + "   ");
                }
                StdOut.println();
            }
            else {
                StdOut.printf("%d do %d         - brak cieki\n", s, v);
            }
        }
    }
}
