unit SimListC;
//*******************************************************
// Klasa implementujca list opart o tablic dynamiczn
//*******************************************************
// Copyright (C) 1998 John Wiley & Sons, Inc.
// All rights reserved. See additional copyright
// information in Readme.txt.
//
// Revised (C) 1999 Andrzej Grayski, HELION Publishing
//*******************************************************

interface

uses
    Dialogs, SysUtils, Classes;

type
    String10     = String[10];

    TSimpleList = class(TObject)
        private
            List        : array of String10;
            NumItems     : Longint;      // Liczba zajtych elementw
            NumAllocated : Longint;      // Liczba przydzielonych elementw
            ShrinkWhen   : Longint;      // Granica usuwania bez kompresji
            procedure ResizeList;

        public
            destructor Destroy; override;
            procedure Add(value : String10);
            function Count : Longint;
            function Item(index : Longint) : String10;
            procedure RemoveLast;
            constructor Create;
    end;

implementation

constructor TSimpleList.Create;
begin
  Inherited Create;
  List := NIL;
  NumItems     := 0;
  NumAllocated := 0;
end;



destructor TSimpleList.Destroy;
begin
  List := NIL;
  inherited Destroy;
end;





// Dodaj element do tablicy.
procedure TSimpleList.Add(value : String10);
begin
    // Zapewnij miejsce dla nowego elementu.
    if (NumItems >= NumAllocated)
    then
      ResizeList;
    NumItems := NumItems + 1;
    List[NumItems-1] := value;
end;

// Podaj liczb elementw w licie.
function TSimpleList.Count : Longint;
begin
  Count := NumItems;
end;


// podaj warto okrelonego elementu
function TSimpleList.Item(index : Longint) : String10;
begin
    if ((index < 1) or (index > NumItems)) then
        raise ERangeError.CreateFmt(
            'Index %d poza zakresem %d..%d',
            [index, 1, NumItems]);

    Item := List[index-1];
end;




// usu ostatni element z listy
procedure TSimpleList.RemoveLast;
begin
    if (NumItems < 1) then
        raise EInvalidOperation.Create(
            'Lista jest pusta.');

    NumItems := NumItems - 1;
    if (NumItems < ShrinkWhen) then ResizeList;
end;


// Zmie rozmiar tablicy jeeli  udzia wolnych
// elementw przekracza WANT_FREE_PERCENT.

// Normalnie wartoci WANT_FREE_PERCENT i MIN_FREE powinny wynosi
// (odpowiednio)  0.1 i 10. Wartoci 0.5 i 1 pozwol lepiej przeledzi
// prac algorytmu. MIN_FREE musi wynosi co najmniej 1

procedure TSimpleList.ResizeList;
const
    WANT_FREE_PERCENT = 0.5; // udzia wolnych elementw.
    MIN_FREE = 1;            // min. liczba wolnych elementw.
    CR = #13#10;
    CRCR = #13#10#13#10;
var
    want_free, new_size: Longint;

begin
    // Okrelenie nowego rozmiaru tablicy.
    want_free := Round(WANT_FREE_PERCENT * NumItems);
    if (want_free < MIN_FREE)
    then
      want_free := MIN_FREE;
    new_size := NumItems + want_free;

    // zmiana rozmiaru tablicy z zachowaniem zawartoci
    SetLength(List, new_size);

    NumAllocated := new_size;


    // Oblicz graniczn liczb wykorzystanych elementw, poniej ktrej
    // naley wykona kompresj
    ShrinkWhen := NumItems - want_free;


    ShowMessage('Zmiana rozmiaru:' + CRCR +
        'Przydzielonych: ' + IntToStr(NumAllocated) + CR +
        'Uywanych: ' + IntToStr(NumItems) + CR +
        'Wolnych: ' + IntToStr(want_free) + CR +
        'Granica kompresji: ' + IntToStr(ShrinkWhen));
end;







end.
