unit FlowF;
//*******************************************************
// Example program demonstrating a network flow algorithm.
//*******************************************************
// Copyright (C) 1998 John Wiley & Sons, Inc.
// All rights reserved. See additional copyright
// information in Readme.txt.
//*******************************************************

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  Menus, StdCtrls;

const
    crLinkCursor = 10;
    crNodeCursor = 11;
    RADIUS = 10;
    INFINITY = 32767;

type
    TNodeStatus = (nsNotInList, nsWasInList, nsNowInList);
    TEditMode = (emNone, emNewNode, emDeleteNode,
        emNewLink1, emNewLink2, emDeleteLink1,
        emDeleteLink2);

    PLink = ^TLink;
    PNode = ^TNode;
    TLink = record
        Node1    : PNode;
        Node2    : PNode;
        Capacity : Integer;
        Flow     : Integer;
        NextLink : PLink;   // Next link in the node's list of links.
        Residual : Integer;
    end;
    TNode = record
        Id           : Integer;
        X            : Integer;
        Y            : Integer;
        LinkSentinel : TLink;       // Links out of this node.
        NextNode     : PNode;       // Next node in list of all nodes.
        Status       : TNodeStatus; // Has it been in the tree?
        InLink       : PLink;       // The link into this node.
    end;

    // Cell for candidate linked list.
    PCandidate = ^TCandidate;
    TCandidate = record
        Node          : PNode;
        NextCandidate : PCandidate;
    end;

  TFlowForm = class(TForm)
    MainMenu1: TMainMenu;
    mnuNew: TMenuItem;
    mnuOpen: TMenuItem;
    mnuSave: TMenuItem;
    mnuSaveAs: TMenuItem;
    mnuFileSep: TMenuItem;
    mnuExit: TMenuItem;
    mnuNewNode: TMenuItem;
    mnuNewLink: TMenuItem;
    mnuDeleteNode: TMenuItem;
    mnuDeleteLink: TMenuItem;
    File1: TMenuItem;
    Edit1: TMenuItem;
    OpenNetDialog: TOpenDialog;
    SaveNetDialog: TSaveDialog;
    Help1: TMenuItem;
    mnuAbout: TMenuItem;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FreeNetwork;
    procedure FormPaint(Sender: TObject);
    procedure mnuExitClick(Sender: TObject);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure mnuNewClick(Sender: TObject);
    procedure mnuOpenClick(Sender: TObject);
    procedure mnuSaveClick(Sender: TObject);
    procedure mnuSaveAsClick(Sender: TObject);
    procedure mnuNewNodeClick(Sender: TObject);
    procedure mnuNewLinkClick(Sender: TObject);
    procedure mnuDeleteNodeClick(Sender: TObject);
    procedure mnuDeleteLinkClick(Sender: TObject);
    procedure mnuAboutClick(Sender: TObject);
    function DataSafe : Boolean;
    procedure NewNetwork;
    procedure DrawNetwork;
    procedure LoadNetwork(fname : String);
    function FindNodeById(id : Integer) : PNode;
    procedure SaveNetwork(fname : String);
    procedure ResetEditMode;
    procedure FormMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    function SelectNode(X, Y : Integer) : PNode;
    procedure DeleteNode(target : PNode);
    procedure DeleteLink(n1, n2 : PNode);
    procedure RemoveLinkFromNode(n1, n2 : PNode);
    procedure CreateNode(X, Y : Integer);
    procedure CreateLink;
    procedure FindMaxFlows;
    procedure ResetFlows;
  private
    { Private declarations }
    FileName     : String;    // Name of file loaded.
    EditMode     : TEditMode; // Click mode.
    DataModified : Boolean;   // Has the data been modified?
    NodeSentinel : TNode;     // List of all nodes.
    Node_1       : PNode;     // Nodes selected for adding a link, etc.
    Node_2       : PNode;
    SourceNode   : PNode;     // Source and sink nodes for flow calculations.
    SinkNode     : PNode;
    MaxId        : Integer;   // Largest node Id.
    TotalFlow    : Integer;
  public
    { Public declarations }
  end;

var
  FlowForm: TFlowForm;

implementation

{$R *.DFM}

// Load some custom cursors.
procedure TFlowForm.FormCreate(Sender: TObject);
begin
    Screen.Cursors[crNodeCursor] := LoadCursor(HInstance, 'NodeCursor');
    Screen.Cursors[crLinkCursor] := LoadCursor(HInstance, 'LinkCursor');

    // Start with an empty network.
    NodeSentinel.NextNode := nil;
end;

// Free the network's dynamically allocated memory.
procedure TFlowForm.FormDestroy(Sender: TObject);
begin
    FreeNetwork;
end;

// Free the network's dynamically allocated memory.
procedure TFlowForm.FreeNetwork;
var
    node, next_node : PNode;
    link, next_link : PLink;
begin
    // Free all the nodes.
    node := NodeSentinel.NextNode;
    while (node <> nil) do
    begin
        // Free the node's links.
        link := node^.LinkSentinel.NextLink;
        while (link <> nil) do
        begin
            next_link := link^.NextLink;
            FreeMem(link);
            link := next_link;
        end;

        // Free the node itself.
        next_node := node^.NextNode;
        FreeMem(node);
        node := next_node;
    end;
    NodeSentinel.NextNode := nil;
end;

// Redraw the network.
procedure TFlowForm.FormPaint(Sender: TObject);
begin
    DrawNetwork;
end;

procedure TFlowForm.mnuExitClick(Sender: TObject);
begin
    Close;
end;

// Refuse to close if the data is not safe.
procedure TFlowForm.FormCloseQuery(Sender: TObject;
  var CanClose: Boolean);
begin
    CanClose := DataSafe;
end;

// Create a new network.
procedure TFlowForm.mnuNewClick(Sender: TObject);
begin
    // Make sure the current data is safe.
    if (not DataSafe) then exit;

    // Start the new network.
    NewNetwork;

    // Display the empty network.
    DrawNetwork;
end;

// Load a network file.
procedure TFlowForm.mnuOpenClick(Sender: TObject);
begin
    // Make sure the current data is safe.
    if (not DataSafe) then exit;

    // Prepare the file selection dialog
    OpenNetDialog.Options :=
        [ofHideReadOnly, ofFileMustExist];
    OpenNetDialog.FileName := '*.net';
    if (not OpenNetDialog.Execute) then exit;

    // Load the file.
    LoadNetwork(OpenNetDialog.FileName);

    // Display the network.
    DrawNetwork;
end;

// Save the file using the current FileName.
procedure TFlowForm.mnuSaveClick(Sender: TObject);
begin
    // Make sure we have a file name.
    if (FileName = '') then
    begin
        mnuSaveAsClick(Sender);
        exit;
    end;

    // Save the network.
    SaveNetwork(FileName);
end;

// Save the network in a file selected by the user.
procedure TFlowForm.mnuSaveAsClick(Sender: TObject);
begin
    // Prepare the file selection dialog
    SaveNetDialog.Options := [ofOverwritePrompt];
    SaveNetDialog.FileName := '*.net';

    // Get the user's selection.
    if (not SaveNetDialog.Execute) then exit;

    // Save the network.
    SaveNetwork(SaveNetDialog.FileName);
end;

// Prepare to get a new node.
procedure TFlowForm.mnuNewNodeClick(Sender: TObject);
begin
    EditMode := emNewNode;
    Cursor := crNodeCursor;
end;

// Prepare to get a new link.
procedure TFlowForm.mnuNewLinkClick(Sender: TObject);
begin
    EditMode := emNewLink1;
    Cursor := crLinkCursor;
end;

// Prepare to delete a node.
procedure TFlowForm.mnuDeleteNodeClick(Sender: TObject);
begin
    EditMode := emDeleteNode;
    Cursor := crNodeCursor;
end;

// Prepare to delete a link.
procedure TFlowForm.mnuDeleteLinkClick(Sender: TObject);
begin
    EditMode := emDeleteLink1;
    Cursor := crLinkCursor;
end;

procedure TFlowForm.mnuAboutClick(Sender: TObject);
const
    CRCR = #13#10#13#10;
begin


    MessageDlg(
        'Niniejszy program oblicza maksymalny przepyw w sieci.' + CRCR +
        'Kliknij lewym przyciskiem w wze penicy rol rda.' + CRCR +
        'Kliknij prawym przyciskiem w wze penicy rol ujcia.'
        , mtInformation, [mbOK], 0);




end;

// Return true if the data is safely saved.
function TFlowForm.DataSafe : Boolean;
const
    MB_YESNOCANCEL = $3;
    MB_ICONQUESTION = $20;
var
    status : Integer;
begin
    if (not DataModified) then
    begin
        Result := True;
        exit;
    end;

    status := Application.MessageBox(
        'Czy zapisa dokonane zmiany w pliku?',
        'Zapisa zmiany?', MB_YESNOCANCEL + MB_ICONQUESTION);
                                                                                            //Inserted by VB2D for SELECT CASE
    case status of
        IDYES:
            begin
                mnuSaveClick(nil);
                // mnuSaveClick resets DataModified appropriately.
                Result := (not DataModified);
            end;
        IDNO:
            Result := True;
    else
        // IDCANCEL:
            Result := False;
    end;
end;

// Start a new network.
procedure TFlowForm.NewNetwork;
begin
    FreeNetwork;
    MaxId := 0;

    // The new network has not yet been modified.
    ResetEditMode;
    DataModified := False;
    FileName := '';
    Caption := 'Flow []';
end;

// Display the network.
procedure TFlowForm.DrawNetwork;
var
    node                   : PNode;
    link                   : PLink;
    x1, y1, x2, y2, dx, dy : Integer;
    dist                   : Single;
    txt                    : String;
    rect                   : TRect;
    size                   : TSize;
begin
    // Erase the form.
    rect.Left := 0;
    rect.Top := 0;
    rect.Right := ClientWidth;
    rect.Bottom := ClientHeight;
    Canvas.Brush.Color := Color;
    Canvas.FillRect(rect);

    // Draw the links and nodes. Every link has a reverse
    // link so we draw those where Node2 > Node1.
    node := NodeSentinel.NextNode;
    while (node <> nil) do
    begin
        link := node^.LinkSentinel.NextLink;
        while (link <> nil) do
        begin
            if (Longint(link^.Node2) > Longint(node)) then
            begin
                x1 := node^.X;
                y1 := node^.Y;
                x2 := link^.Node2^.X;
                y2 := link^.Node2^.Y;
                dx := x2 - x1;
                dy := y2 - y1;
                dist := Sqrt((dx * dx) + (dy * dy));
                if (dist > 0) then
                begin
                    dx := Round(dx * RADIUS / dist);
                    dy := Round(dy * RADIUS / dist);

                    // Draw the link.
                    x1 := x1 + dx;
                    y1 := y1 + dy;
                    x2 := x2 - dx;
                    y2 := y2 - dy;
                    if (link.Flow <> 0) then
                    begin
                        Canvas.Pen.Color := clBlue;
                        Canvas.Pen.Width := 3;
                    end else begin
                        Canvas.Pen.Color := clBlack;
                        Canvas.Pen.Width := 1;
                    end;
                    Canvas.MoveTo(x1, y1);
                    Canvas.LineTo(x2, y2);
                    Canvas.Pen.Color := clBlack;
                    Canvas.Pen.Width := 1;

                    // Draw the link capacity and flow.
                    x1 := (x1 + x2) div 2;
                    y1 := (y1 + y2) div 2;
                    Canvas.Brush.Color := Color;
                    Canvas.Pen.Color := Color;
                    Canvas.Ellipse(
                        x1 - RADIUS, y1 - RADIUS,
                        x1 + RADIUS, y1 + RADIUS);

                    txt := Format('%d/%d',
                        [link^.Flow, link^.Capacity]);
                    size := Canvas.TextExtent(txt);
                    Canvas.Pen.Color := clBlack;
                    Canvas.TextOut(
                        x1 - (size.cx div 2),
                        y1 - (size.cy div 2),
                        txt);
                end; // End if (dist > 0) then ...
            end; // End if (link^.Node2 > node) then ...

            // Go to the next link for this node.
            link := link^.NextLink;
        end; // End while (link <> nil) do ...

        if ((node = Node_1) or (node = Node_2)) then
        begin
            // Highlight the node.
            Canvas.Brush.Color := clBlack;
            Canvas.Font.Color := clWhite;
        end else begin
            // Do not highlight the node.
            Canvas.Brush.Color := clWhite;
            Canvas.Font.Color := clBlack;
        end;

        // Draw the node.
        x1 := node^.X;
        y1 := node^.Y;
        if ((node = SourceNode) or (node = SinkNode)) then
            Canvas.Pen.Width := 3
        else
            Canvas.Pen.Width := 1;
        Canvas.Ellipse(
            x1 - RADIUS, y1 - RADIUS,
            x1 + RADIUS, y1 + RADIUS);
        Canvas.Pen.Width := 1;

        txt := IntToStr(node^.Id);
        size := Canvas.TextExtent(txt);
        Canvas.TextOut(
            x1 - (size.cx div 2),
            y1 - (size.cy div 2),
            txt);
        Canvas.Font.Color := clBlack;

        // Go to the next node.
        node := node^.NextNode;
    end; // End while (node <> nil) do ...

    Canvas.Brush.Color := Color;
    Canvas.Font.Color := clBlack;
    Canvas.TextOut(0, 0, Format('Cakowity przepyw: %d', [TotalFlow]));
end;

// Load the network data from a file with format:
//   # Nodes
//   For each node: Id, X, Y
// After Id, X, and Y for all nodes:
//   For each node: # Links, for each link:
//       Node2, Capacity
procedure TFlowForm.LoadNetwork(fname : String);
var
    i, j, num_nodes, num_links, to_node : Integer;
    net_file                         : TextFile;
    node                             : PNode;
    link                             : PLink;
begin
    // Start a new network.
    NewNetwork;

    // Open the file.
    AssignFile(net_file, fname);
    Reset(net_file);

    // Read the number of nodes and links.
    Readln(net_file, num_nodes);

    // Read the nodes' Id, X, and Y.
    MaxId := 0;
    node := @NodeSentinel;
    for i := 1 to num_nodes do
    begin
        GetMem(node^.NextNode, SizeOf(TNode));
        node := node^.NextNode;
        with node^ do
        begin
            Readln(net_file, Id, X, Y);
            LinkSentinel.NextLink := nil;
            if (MaxId < Id) then MaxId := Id;
        end;
    end;
    node^.NextNode := nil;

    // Read the link information.
    node := NodeSentinel.NextNode;
    for i := 1 to num_nodes do
    begin
        node^.LinkSentinel.NextLink := nil;
        Readln(net_file, num_links);
        for j := 1 to num_links do
        begin
            GetMem(link, SizeOf(TLink));
            link^.NextLink := node^.LinkSentinel.NextLink;
            node^.LinkSentinel.NextLink := link;
            Readln(net_file, to_node, link^.Capacity);
            link^.Node2 := FindNodeById(to_node);
            link^.Node1 := node;
        end;

        // Get the link info for the next node.
        node := node^.NextNode;
    end;

    // Close the file.
    CloseFile(net_file);
    ResetFlows;

    // The network has not yet been modified.
    FileName := fname;
    Caption := 'Flow [' + fname + ']';
end;

// Find a node given its Id.
function TFlowForm.FindNodeById(id : Integer) : PNode;
var
    node : PNode;
begin
    node := NodeSentinel.NextNode;
    while (node <> nil) do
    begin
        if (node^.Id = id) then break;
        node := node^.NextNode;
    end;
    if (node <> nil) then
        Result := node
    else
        Result := nil;
end;

// Save the network data into a file with format:
//   # Nodes
//   For each node: Id, X, Y
// After Id, X, and Y for all nodes:
//   For each node: # Links, for each link:
//       Node2, Capacity
procedure TFlowForm.SaveNetwork(fname : String);
var
    i        : Integer;
    net_file : TextFile;
    node     : PNode;
    link     : PLink;
begin
    // Open the file.
    AssignFile(net_file, fname);
    Rewrite(net_file);

    // Save the number of nodes.
    i := 0;
    node := NodeSentinel.NextNode;
    while (node <> nil) do
    begin
        i := i + 1;
        node := node^.NextNode;
    end;
    Writeln(net_file, i);

    // Save the node identifiers and coordinates.
    node := NodeSentinel.NextNode;
    while (node <> nil) do
    begin
        Writeln(net_file, node^.Id, ' ', node^.X, ' ', node^.Y);
        node := node^.NextNode;
    end;

    // Save the link information.
    node := NodeSentinel.NextNode;
    while (node <> nil) do
    begin
        // Count the links.
        i := 0;
        link := node^.LinkSentinel.NextLink;
        while (link <> nil) do
        begin
            i := i + 1;
            link := link^.NextLink;
        end;
        Writeln(net_file, i);

        // Save the link information.
        link := node^.LinkSentinel.NextLink;
        while (link <> nil) do
        begin
            Writeln(net_file, link^.Node2^.Id, ' ', link^.Capacity);
            link := link^.NextLink;
        end;

        // Go to the next node.
        node := node^.NextNode;
    end;

    // Close the file.
    CloseFile(net_file);

    DataModified := False;
    FileName := fname;
    Caption := 'Flow [' + fname + ']';
end;

// Reset the edit mode.
procedure TFlowForm.ResetEditMode;
begin
    EditMode := emNone;
    Cursor := crDefault;
    Node_1 := nil;
    Node_2 := nil;
end;

// Perform the task appropriate for this edit mode.
procedure TFlowForm.FormMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
begin
    case EditMode of
        emNone:
            begin
                if (Button = mbLeft) then
                    SourceNode := SelectNode(X, Y)
                else
                    SinkNode := SelectNode(X, Y);
                FindMaxFlows;
            end;
        emNewNode:
            begin
                CreateNode(X, Y);
                EditMode := emNone;
            end;
        emDeleteNode:
            begin
                DeleteNode((SelectNode(X, Y)));
                EditMode := emNone;
            end;
        emNewLink1:
            begin
                Node_1 := SelectNode(X, Y);
                if (Node_1 = nil) then
                    EditMode := emNone
                else
                    EditMode := emNewLink2;
            end;
        emNewLink2:
            begin
                Node_2 := SelectNode(X, Y);
                if (Node_2 <> nil) then CreateLink;
                EditMode := emNone;
            end;
        emDeleteLink1:
            begin
                Node_1 := SelectNode(X, Y);
                if (Node_1 = nil) then
                    EditMode := emNone
                else
                    EditMode := emDeleteLink2;
            end;
        emDeleteLink2:
            begin
                Node_2 := SelectNode(X, Y);
                if (Node_2 <> nil) then DeleteLink(Node_1, Node_2);
                EditMode := emNone;
            end;
    end; // End case EditMode of ...

    if (EditMode = emNone) then
    begin
        Cursor := crDefault;
        Node_1 := nil;
        Node_2 := nil;
    end;

    DrawNetwork;
end;

// Find the node that contains this point.
function TFlowForm.SelectNode(X, Y : Integer) : PNode;
var
    dx, dy, radius2 : Integer;
    node            : PNode;
begin
    radius2 := RADIUS * RADIUS;
    node := NodeSentinel.NextNode;
    while (node <> nil) do
    begin
        dx := node^.X - X;
        dy := node^.Y - Y;
        if ((dx * dx + dy * dy) <= radius2) then
        begin
            Result := node;
            exit;
        end;
        node := node^.NextNode;
    end;
    Result := nil;
end;

// Delete the node from the network.
procedure TFlowForm.DeleteNode(target : PNode);
var
    node, next_node : PNode;
begin
    if (target = nil) then exit;

    // Free the target's links.
    while (target^.LinkSentinel.NextLink <> nil) do
    begin
        DeleteLink(target,
            target^.LinkSentinel.NextLink^.Node2);
    end;

    // Find the target in the node list.
    node := @NodeSentinel;
    next_node := node^.NextNode;
    while (next_node <> nil) do
    begin
        if (next_node = target) then break;
        node := next_node;
        next_node := node^.NextNode;
    end;
    if (next_node = nil) then exit; // Not found.

    // Remove the node.
    node^.NextNode := target^.NextNode;
    FreeMem(target);

    // Redraw the network.
    ResetFlows;
    DrawNetwork;

    // The data has been modified.
    DataModified := True;
    Caption := 'Flow*[' + FileName + ']';
end;

// Delete a link from the network.
procedure TFlowForm.DeleteLink(n1, n2 : PNode);
begin
    if ((n1 = nil) or (n2 = nil)) then exit;

    // Remove the link from the nodes' link lists.
    RemoveLinkFromNode(n1, n2);
    RemoveLinkFromNode(n2, n1);

    // Redraw the network.
    ResetFlows;
    DrawNetwork;

    // The data has been modified.
    DataModified := True;
    Caption := 'Flow*[' + FileName + ']';
end;

// Remove the link to node n2 from node n1's Links array.
procedure TFlowForm.RemoveLinkFromNode(n1, n2 : PNode);
var
    link, next_link : PLink;
begin
    // Find the link.
    link := @n1^.LinkSentinel;
    next_link := link^.NextLink;
    while (next_link <> nil) do
    begin
        if (next_link^.Node2 = n2) then break;
        link := next_link;
        next_link := link^.NextLink;
    end;
    if (next_link = nil) then exit;

    // Remove the link.
    link^.NextLink := next_link^.NextLink;
    FreeMem(next_link);
end;

// Create a new node.
procedure TFlowForm.CreateNode(X, Y : Integer);
var
    node : PNode;
begin
    // Create the new node.
    GetMem(node, SizeOf(TNode));
    MaxId := MaxId + 1;
    node^.Id := MaxId;
    node^.X := X;
    node^.Y := Y;
    node^.LinkSentinel.NextLink := nil;

    // Add the node to the node list.
    node^.NextNode := NodeSentinel.NextNode;
    NodeSentinel.NextNode := node;

    // Redraw the network.
    ResetFlows;
    DrawNetwork;

    // The data has been modified.
    DataModified := True;
    Caption := 'Flow*[' + FileName + ']';
end;

// Create a new link between nodes Node_1 and Node_2.
procedure TFlowForm.CreateLink;
var
    link1, link2 : PLink;
    txt          : String;
begin
    if ((Node_1 = nil) or (Node_2 = nil)) then exit;

    // Do not allow links from a node to itself.
    if (Node_1 = Node_2) then
    begin
        ShowMessage('Krawd musi przebiega midzy dwoma rnymi wzami.');
        exit;
    end;

    // Get the link Capacity from the user.
    txt := InputBox('Koszt krawdzi', 'Koszt krawdzi', '10');

    // Create the links.
    GetMem(link1, SizeOf(TLink));
    GetMem(link2, SizeOf(TLink));
    link1^.Capacity := StrToInt(txt);
    link2^.Capacity := link1^.Capacity;
    link1^.Node1 := Node_1;
    link1^.Node2 := Node_2;
    link2^.Node1 := Node_2;
    link2^.Node2 := Node_1;
    link1^.NextLink := nil;
    link2^.NextLink := nil;

    // Add the links to the nodes' link lists.
    link1^.NextLink := Node_1^.LinkSentinel.NextLink;
    Node_1^.LinkSentinel.NextLink := link1;

    link2^.NextLink := Node_2^.LinkSentinel.NextLink;
    Node_2^.LinkSentinel.NextLink := link2;

    // Redraw the network.
    ResetFlows;
    DrawNetwork;

    // The data has been modified.
    DataModified := True;
    Caption := 'Flow*[' + FileName + ']';
end;

// Find the maximum flows.
procedure TFlowForm.FindMaxFlows;
var
    top_candidate, candidate : PCandidate;
    node, to_node            : PNode;
    link, rev_link           : PLink;
    min_residual             : Integer;
begin
    if ((SourceNode = nil) or (SinkNode = nil)) then exit;

    // Initially residual value equals capacity.
    node := NodeSentinel.NextNode;
    while (node <> nil) do
    begin
        link := node^.LinkSentinel.NextLink;
        while (link <> nil) do
        begin
            link^.Residual := link^.Capacity;
            link := link^.NextLink;
        end;
        node := node^.NextNode;
    end;

    // Repeat until we can find no more augmenting paths.
    repeat
        // Find an augmenting path in the residual network.
        // Reset the nodes' NodeStatus and InLink values.
        node := NodeSentinel.NextNode;
        while (node <> nil) do
        begin
            node^.Status := nsNotInList;
            node^.InLink := nil;
            node := node^.NextNode;
        end;

        // Put the source on the candidate list.
        SourceNode^.Status := nsNowInList;
        GetMem(top_candidate, SizeOf(TCandidate));
        top_candidate^.Node := SourceNode;
        top_candidate^.NextCandidate := nil;

        // Repeat until the candidate list is empty.
        while (top_candidate <> nil) do
        begin
            // Remove the top candidate from the list.
            node := top_candidate^.Node;
            node^.Status := nsWasInList;
            candidate := top_candidate^.NextCandidate;
            FreeMem(top_candidate);
            top_candidate := candidate;

            // Examine the links out of this node.
            link := node^.LinkSentinel.NextLink;
            while (link <> nil) do
            begin
                // See if the residual > 0 and this node
                // has never been on the list.
                to_node := link^.Node2;
                if ((link^.Residual > 0) and
                    (to_node^.Status = nsNotInList)) then
                begin
                    // Add it to the list.
                    to_node^.Status := nsNowInList;
                    to_node^.InLink := link;
                    GetMem(candidate, SizeOf(TCandidate));
                    candidate^.Node := to_node;
                    candidate^.NextCandidate := top_candidate;
                    top_candidate := candidate;
                end;
                link := link^.NextLink;
            end; // End examining links out of the node.

            // Stop if the sink has been labeled.
            if (SinkNode.InLink <> nil) then break;
        end; // End while (top_candidate <> nil) do ...

        // Stop if we found no augmenting path.
        if (SinkNode.InLink = nil) then break;

        // Trace the augmenting path from SinkNode back to
        // SourceNode to find the smallest residual.
        min_residual := INFINITY;
        to_node := SinkNode;
        while (to_node <> SourceNode) do
        begin
            link := to_node^.InLink;
            if (link^.Residual < min_residual) then
                min_residual := link^.Residual;
            to_node := link^.Node1;
        end;

        // Update the residuals using the augmenting path.
        to_node := SinkNode;
        while (to_node <> SourceNode) do
        begin
            link := to_node^.InLink;
            link^.Residual := link^.Residual - min_residual;

            // Find and update the reverse link.
            node := link^.Node1;
            rev_link := to_node^.LinkSentinel.NextLink;
            while (rev_link <> nil) do
            begin
                if (rev_link^.Node2 = node) then break;
                rev_link := rev_link^.NextLink;
            end;
            if (rev_link <> nil) then
                rev_link^.Residual :=
                    rev_link^.Residual + min_residual;

            // Update the next link in the augmenting path.
            to_node := link^.Node1;
        end;

        // Free any items remaining in the candidate list.
        while (top_candidate <> nil) do
        begin
            candidate := top_candidate^.NextCandidate;
            FreeMem(top_candidate);
            top_candidate := candidate;
        end;
    until (False); // End infinite loop looking for augmenting paths
    // The loop ends when there are no more augmenting paths.

    // Calculate the flows from the residuals.
    node := NodeSentinel.NextNode;
    while (node <> nil) do
    begin
        link := node^.LinkSentinel.NextLink;
        while (link <> nil) do
        begin
            if (link^.Capacity > link^.Residual) then
                link^.Flow := link^.Capacity - link^.Residual
            else
                // Negative to indicate backwards flow.
                link^.Flow := link^.Residual - link^.Capacity;
            link := link^.NextLink;
        end;
        node := node^.NextNode;
    end;

    // Calculate the total flow.
    TotalFlow := 0;
    link := SourceNode^.LinkSentinel.NextLink;
    while (link <> nil) do
    begin
        TotalFlow := TotalFlow + link^.Flow;
        link := link^.NextLink;
    end;
end;

// Reset the flows.
procedure TFlowForm.ResetFlows;
var
    node : PNode;
    link : PLink;
begin
    node := NodeSentinel.NextNode;
    while (node <> nil) do
    begin
        link := node^.LinkSentinel.NextLink;
        while (link <> nil) do
        begin
            link^.Flow := 0;
            link := link^.NextLink;
        end;
        node := node^.NextNode;
    end;

    SourceNode := nil;
    SinkNode := nil;
end;

end.
