unit RandF;
//*******************************************************
// Example program demonstrating a hash table with open
// addressing and pseudo-random probing.
//*******************************************************
// Copyright (C) 1998 John Wiley & Sons, Inc.
// All rights reserved. See additional copyright
// information in Readme.txt.
//*******************************************************

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  Menus, StdCtrls,
  RandC;

type
  TRandForm = class(TForm)
    Frame2: TGroupBox;
    Label1_0: TLabel;
    NewItemText: TEdit;
    CmdAdd: TButton;
    CmdFind: TButton;
    Frame3: TGroupBox;
    Label1_6: TLabel;
    Label1_2: TLabel;
    NumItemsText: TEdit;
    MaxValueText: TEdit;
    CmdCreateItems: TButton;
    Frame1: TGroupBox;
    Label1_3: TLabel;
    CmdCreateTable: TButton;
    TableSizeText: TEdit;
    StatusLabel: TLabel;
    MainMenu1: TMainMenu;
    mnuFile: TMenuItem;
    mnuFileExit: TMenuItem;
    mnuHelp: TMenuItem;
    mnuHelpAbout: TMenuItem;
    procedure FormCreate(Sender: TObject);
    procedure mnuFileExitClick(Sender: TObject);
    procedure mnuHelpAboutClick(Sender: TObject);
    procedure CmdCreateTableClick(Sender: TObject);
    procedure CmdCreateItemsClick(Sender: TObject);
    procedure CmdAddClick(Sender: TObject);
    procedure CmdFindClick(Sender: TObject);
    procedure DisplayStatus(msg : String; probes : Integer);
    procedure ComputeAverageProbe(var ave_succ, ave_unsucc : Single);
    procedure TableSizeTextChange(Sender: TObject);
    procedure NewItemTextChange(Sender: TObject);
    procedure FormPaint(Sender: TObject);
  private
    { Private declarations }
    MaxValue     : TTableData;
    HashTable    : TRandomHashTable;
    SelectedItem : TTableData;
  public
    { Public declarations }
    destructor Destroy; override;
  end;

var
  RandForm: TRandForm;

implementation

{$R *.DFM}

// Create the hash table.
procedure TRandForm.FormCreate(Sender: TObject);
begin
    Randomize;

    HashTable := TRandomHashTable.Create;
end;

// Free the hash table.
destructor TRandForm.Destroy;
begin
    HashTable.Free;
    inherited Destroy;
end;

procedure TRandForm.mnuFileExitClick(Sender: TObject);
begin
    Close;
end;

procedure TRandForm.mnuHelpAboutClick(Sender: TObject);
const
    CRCR = #13#10#13#10;
    CR = #13#10;
begin


    MessageDlg(
        'Niniejszy program demonstruje adresowanie otwarte wykorzystujce prbkowanie pseudolosowe.' + CRCR +
        'W sekcji "Tworzenie tablicy" podaj maksymaln liczb elementw w tablicy.' + CRCR +
        'W sekcji "Generowanie elementw" podaj liczb generowanych losowo elementw' +CR+
        'oraz maksymaln warto elementu.' +CRCR+
        'Sekcja "Szukanie" suy do wyszukiwania lub dodawania elementw.'
        , mtInformation, [mbOK], 0);

        
end;

// Create a new hash table.
procedure TRandForm.CmdCreateTableClick(Sender: TObject);
begin
    Screen.Cursor := crHourGlass;

    // This makes the HashTable create the lists.
    HashTable.CreateTable(StrToInt(TableSizeText.Text));

    MaxValue := 1;
    StatusLabel.Caption := 'Utworzono tablic.';
    CmdCreateItems.Enabled := True;
    Screen.Cursor := crDefault;
    SelectedItem := -1;
    Refresh;
end;

// Add random items to the hash table.
procedure TRandForm.CmdCreateItemsClick(Sender: TObject);
var
    num, created, probes : Integer;
    value, max_value     : TTableData;
begin
    max_value := StrToInt(MaxValueText.Text);
    num := StrToInt(NumItemsText.Text);
    if (max_value < num) then
    begin
        ShowMessage('Zbyt maa warto maksymalna.');
        exit;
    end;

    Screen.Cursor := crHourGlass;
    created := 0; // The number we have created.
    while (created < num) do
    begin
        value := Trunc(Random(max_value) + 1);
        if (value > MaxValue) then MaxValue := value;
        case HashTable.InsertItem(value, probes) of
            rndInserted:
                created := created + 1;
            rndTableFull:
                break;
        end;

        // Be sure to randomize after calls to InsertItem.
        Randomize;
    end;
    DisplayStatus('Wykonano.', probes);

    Screen.Cursor := crDefault;
    SelectedItem := -1;
    Refresh;
end;

// Add an item to the hash table.
procedure TRandForm.CmdAddClick(Sender: TObject);
var
    probes : Integer;
    value  : TTableData;
    status : TRandomReturnValue;
begin
    Screen.Cursor := crHourGlass;

    value := StrToInt(NewItemText.Text);
    if (value > MaxValue) then MaxValue := value;

    // Insert the item.
    status := HashTable.InsertItem(value, probes);
    case status of
        rndFound:
            DisplayStatus('Element jest ju obecny w tablicy.', probes);
        rndInserted:
            DisplayStatus('Dodano.', probes);
        rndTableFull:
            DisplayStatus('Tablica przepeniona.', probes);
    end;

    // Be sure to randomize after calls to InsertItem.
    Randomize;

    Screen.Cursor := crDefault;
    SelectedItem := value;
    Refresh;
end;

// Find an item in the hash table.
procedure TRandForm.CmdFindClick(Sender: TObject);
var
    value  : TTableData;
    probes : Integer;
    status : TRandomReturnValue;
begin
    Screen.Cursor := crHourGlass;

    value := StrToInt(NewItemText.Text);
    if (value > MaxValue) then MaxValue := value;

    status := HashTable.LocateItem(value, probes);
    case status of
        rndNotFound:
            DisplayStatus('Nie znaleziono.', probes);
        rndFound:
            DisplayStatus('Znaleziono.', probes);
    end;

    // Be sure to randomize after calls to LocateItem.
    Randomize;

    Screen.Cursor := crDefault;
    SelectedItem := value;
    Refresh;
end;

// Display a status message.
procedure TRandForm.DisplayStatus(msg : String; probes : Integer);
const
    CR = #13#10;
var
    ave_succ, ave_unsucc : single;
begin
    ComputeAverageProbe(ave_succ, ave_unsucc);
    StatusLabel.Caption := msg + CR +
        Format('Ta prba: %d.', [probes]) + CR +
        Format('rednio prb pomylnych: %.2f.', [ave_succ]) + CR +
        Format('rednio prb niepomylnych: %.2f.', [ave_unsucc]);
end;

// Compute the average lengths of probe sequences.
procedure TRandForm.ComputeAverageProbe(var ave_succ, ave_unsucc : Single);
var
    status                         : TRandomReturnValue;
    trial, probes                  : Integer;
    succ_searches, succ_probes     : Integer;
    unsucc_searches, unsucc_probes : Integer;
begin
    succ_searches := 0;
    succ_probes := 0;
    unsucc_searches := 0;
    unsucc_probes := 0;
    for trial := 1 to MaxValue do
    begin
        status := HashTable.LocateItem(trial, probes);
        if (status = rndFound) then
        begin
            succ_searches := succ_searches + 1;
            succ_probes := succ_probes + probes;
        end else begin
            unsucc_searches := unsucc_searches + 1;
            unsucc_probes := unsucc_probes + probes;
        end;
    end;

    // Be sure to randomize after calls to LocateItem.
    Randomize;

    if (succ_searches > 0) then
        ave_succ := succ_probes / succ_searches
    else
        ave_succ := 0;
    if (unsucc_searches > 0) then
        ave_unsucc := unsucc_probes / unsucc_searches
    else
        ave_unsucc := 0;
end;

procedure TRandForm.TableSizeTextChange(Sender: TObject);
begin
    CmdCreateTable.Enabled := (TableSizeText.Text <> '');
end;

procedure TRandForm.NewItemTextChange(Sender: TObject);
begin
    CmdAdd.Enabled := (NewItemText.Text <> '');
    CmdFind.Enabled := CmdAdd.Enabled;
end;

// Display the table highlighting SelectedItem.
procedure TRandForm.FormPaint(Sender: TObject);
const
    WID = 27;
    HGT = 17;
var
    i, x0, x, y, bx, by : Integer;
    value               : TTableData;
    txt                 : String;
    rect                : TRect;
    size                : TSize;
begin
    rect.Left := 0;
    rect.Right := ClientWidth;
    rect.Top := 0;
    rect.Bottom := ClientHeight;
    Canvas.Brush.Color := Brush.Color;
    Canvas.FillRect(rect);

    if (HashTable.NumItems < 1) then exit;

    // Draw headings for the columns.
    x0 := Frame1.Left + Frame1.Width + 20;
    x := x0;
    y := 0;
    for i := 0 to 9 do
    begin
        Canvas.TextOut(x, y, Format('%3d', [i]));
        x := x + WID;
    end;

    // Display the table values.
    x := x0;
    y := y + Round(1.5 * HGT);
    for i := 0 to HashTable.NumItems - 1 do
    begin
        value := HashTable.TableValue(i);
        if (value = UNUSED) then
            txt := '---'
        else
            txt := Format('%3d', [value]);

        // Draw a box around it if it's selected.
        if (value = SelectedItem) then
        begin
            size := Canvas.TextExtent(txt);
            bx := x + (size.cx - WID) div 2;
            by := y + (size.cy - HGT) div 2;
            Canvas.Rectangle(bx, by, bx + WID, by + HGT);
        end;

        Canvas.TextOut(x, y, txt);

        if (i mod 10 = 9) then
        begin
            x := x0;
            y := y + HGT;
        end else
            x := x + WID;
    end;
end;

end.
